// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_OFFLINE_PAGES_CORE_OFFLINE_PAGE_CLIENT_POLICY_H_
#define COMPONENTS_OFFLINE_PAGES_CORE_OFFLINE_PAGE_CLIENT_POLICY_H_

#include <stdint.h>

#include <string>

#include "base/time/time.h"

namespace offline_pages {

static const size_t kUnlimitedPages = 0;

// Enum whose values specify the lifetime characteristics of pages pertaining to
// a given client.
enum class LifetimeType {
  // A temporary offline page, which is automatically downloaded and cleaned
  // up when storage limits are reached or when caches are cleared. They are
  // normally stored in Chrome internal directories, not directly accessible to
  // the user.
  TEMPORARY,
  // A persistent offline page, manually downloaded by the user. It is stored in
  // the public Downloads directory and only the user can delete it.
  PERSISTENT,
};

// The struct describing the lifetime policy of offline pages.
// The following behaviors are controlled by policy:
//    a. Persistency of the offline page.
//    b. Expiration time of an offline page
//    c. Limit of number of pages offline.
struct LifetimePolicy {
  // Type of the page generated by the client.
  LifetimeType lifetime_type;

  // The time after which the page expires.
  // A TimeDelta of 0 means no expiration.
  base::TimeDelta expiration_period;

  // The maximum number of pages allowed to be saved by the namespace.
  // kUnlimitedPages (defined above) means no limit set.
  size_t page_limit;

  LifetimePolicy(LifetimeType init_lifetime_type, size_t init_page_limit)
      : lifetime_type(init_lifetime_type),
        expiration_period(base::TimeDelta::FromDays(0)),
        page_limit(init_page_limit) {}
};

// The struct describing feature set of the offline pages.
struct FeaturePolicy {
  // Whether pages are shown in download ui.
  bool is_supported_by_download = false;
  // Whether pages can only be viewed in a specific tab. Pages controlled by
  // this policy must have their ClientId::id field set to their assigned tab's
  // id.
  bool is_restricted_to_tab_from_client_id = false;
  // Whether the namespace should be disabled if any of these user settings are
  // set to:
  // * 3rd party cookies are blocked (prefs::kBlockThirdPartyCookies).
  // * Network predictions (prefs::kNetworkPredictionOptions) are fully
  //   disabled.
  bool requires_specific_user_settings = false;
  // Whether the pages originated from suggestions by zine or elsewhere.
  bool is_suggested = false;
  // Whether a background page download is allowed to be converted to a regular
  // download if the URL turns out to point to a file (i.e. a PDF).
  bool allows_conversion_to_background_file_download = false;
};

// The struct describing policies for various namespaces (Bookmark, Last-N etc.)
// used by offline page model. The name_space is supposed to be key, so that
// it's sufficient to compare name_space only when doing comparisons.
struct OfflinePageClientPolicy {
  // Namespace to which the policy applied.
  std::string name_space;

  // Policy to control the lifetime of a page generated by this namespace.
  LifetimePolicy lifetime_policy;

  // How many pages for the same online URL can be stored at any time.
  // kUnlimitedPages means there's no limit.
  size_t pages_allowed_per_url;

  FeaturePolicy feature_policy;

  // Whether background fetches are deferred while the active tab matches the
  // SavePageRequestURL.
  bool defer_background_fetch_while_page_is_active = false;

  OfflinePageClientPolicy(std::string namespace_val,
                          LifetimePolicy lifetime_policy_val,
                          size_t pages_allowed_per_url_val,
                          FeaturePolicy feature_policy_val)
      : name_space(namespace_val),
        lifetime_policy(lifetime_policy_val),
        pages_allowed_per_url(pages_allowed_per_url_val),
        feature_policy(feature_policy_val) {}

  OfflinePageClientPolicy(std::string namespace_val,
                          LifetimePolicy lifetime_policy_val,
                          size_t pages_allowed_per_url_val)
      : OfflinePageClientPolicy(namespace_val,
                                lifetime_policy_val,
                                pages_allowed_per_url_val,
                                FeaturePolicy()) {}
};

class OfflinePageClientPolicyBuilder {
 public:
  OfflinePageClientPolicyBuilder(const std::string& name_space,
                                 LifetimeType lifetime_type,
                                 size_t page_limit,
                                 size_t pages_allowed_per_url)
      : policy_(
            OfflinePageClientPolicy(name_space,
                                    LifetimePolicy(lifetime_type, page_limit),
                                    pages_allowed_per_url)) {}

  ~OfflinePageClientPolicyBuilder() {}

  // Calling build does not reset the object inside.
  const OfflinePageClientPolicy Build() const { return policy_; }

  OfflinePageClientPolicyBuilder& SetExpirePeriod(
      const base::TimeDelta& expire_period) {
    policy_.lifetime_policy.expiration_period = expire_period;
    return *this;
  }

  OfflinePageClientPolicyBuilder& SetIsSupportedByDownload(
      const bool is_downloaded) {
    policy_.feature_policy.is_supported_by_download = is_downloaded;
    return *this;
  }

  OfflinePageClientPolicyBuilder& SetIsRestrictedToTabFromClientId(
      const bool is_restricted_to_tab_from_client_id) {
    policy_.feature_policy.is_restricted_to_tab_from_client_id =
        is_restricted_to_tab_from_client_id;
    return *this;
  }

  OfflinePageClientPolicyBuilder& SetRequiresSpecificUserSettings(
      const bool requires_specific_user_settings) {
    policy_.feature_policy.requires_specific_user_settings =
        requires_specific_user_settings;
    return *this;
  }

  OfflinePageClientPolicyBuilder& SetIsSuggested(const bool is_suggested) {
    policy_.feature_policy.is_suggested = is_suggested;
    return *this;
  }

  OfflinePageClientPolicyBuilder& SetAllowConversionToBackgroundFileDownload(
      const bool allows_conversion_to_background_file_download) {
    policy_.feature_policy.allows_conversion_to_background_file_download =
        allows_conversion_to_background_file_download;
    return *this;
  }

  OfflinePageClientPolicyBuilder& SetDeferBackgroundFetchWhilePageIsActive(
      bool defer) {
    policy_.defer_background_fetch_while_page_is_active = defer;
    return *this;
  }

 private:
  OfflinePageClientPolicy policy_;

  DISALLOW_COPY_AND_ASSIGN(OfflinePageClientPolicyBuilder);
};

}  // namespace offline_pages

#endif  // COMPONENTS_OFFLINE_PAGES_CORE_OFFLINE_PAGE_CLIENT_POLICY_H_
