// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/password_manager/core/browser/password_generation_state.h"

#include <utility>

#include "base/time/default_clock.h"
#include "components/password_manager/core/browser/form_saver.h"
#include "components/password_manager/core/browser/password_manager_driver.h"

namespace password_manager {

using autofill::PasswordForm;

PasswordGenerationState::PasswordGenerationState(FormSaver* form_saver,
                                                 PasswordManagerClient* client)
    : form_saver_(form_saver),
      client_(client),
      clock_(new base::DefaultClock) {}

PasswordGenerationState::~PasswordGenerationState() = default;

std::unique_ptr<PasswordGenerationState> PasswordGenerationState::Clone(
    FormSaver* form_saver) const {
  auto clone = std::make_unique<PasswordGenerationState>(form_saver, client_);
  clone->presaved_ = presaved_;
  return clone;
}

void PasswordGenerationState::GeneratedPasswordAccepted(
    const PasswordForm& generated,
    const std::vector<const autofill::PasswordForm*>& matches,
    base::WeakPtr<PasswordManagerDriver> driver) {
  // TODO(936011): more logic is coming.
  driver->GeneratedPasswordAccepted(generated.password_value);
}

void PasswordGenerationState::PresaveGeneratedPassword(
    PasswordForm generated,
    const std::vector<const autofill::PasswordForm*>& matches) {
  DCHECK(!generated.password_value.empty());
  // Clear the username value if there are already saved credentials with
  // the same username in order to prevent overwriting.
  if (std::any_of(matches.begin(), matches.end(),
                  [&generated](const autofill::PasswordForm* form) {
                    return form->username_value == generated.username_value;
                  }))
    generated.username_value.clear();
  generated.date_created = clock_->Now();
  if (presaved_) {
    form_saver_->UpdateReplace(generated, {} /* matches */,
                               base::string16() /* old_password */,
                               presaved_.value() /* old_primary_key */);
  } else {
    form_saver_->Save(generated, {} /* matches */,
                      base::string16() /* old_password */);
  }
  presaved_ = std::move(generated);
}

void PasswordGenerationState::PasswordNoLongerGenerated() {
  DCHECK(presaved_);
  form_saver_->Remove(*presaved_);
  presaved_.reset();
}

void PasswordGenerationState::CommitGeneratedPassword(
    PasswordForm generated,
    const std::vector<const autofill::PasswordForm*>& matches,
    const base::string16& old_password) {
  DCHECK(presaved_);
  generated.preferred = true;
  generated.date_created = clock_->Now();
  form_saver_->UpdateReplace(generated, matches, old_password,
                             presaved_.value() /* old_primary_key */);
}

}  // namespace password_manager
