// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/chromeos/scheduler_configuration_manager.h"

#include <memory>

#include "base/test/scoped_feature_list.h"
#include "base/test/scoped_task_environment.h"
#include "chrome/common/chrome_features.h"
#include "chrome/common/pref_names.h"
#include "chromeos/dbus/fake_debug_daemon_client.h"
#include "components/prefs/testing_pref_service.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "third_party/cros_system_api/dbus/debugd/dbus-constants.h"

namespace chromeos {

class SchedulerConfigurationManagerTest : public testing::Test {
 public:
  SchedulerConfigurationManagerTest() {
    SchedulerConfigurationManager::RegisterLocalStatePrefs(
        local_state_.registry());
  }

  base::test::ScopedTaskEnvironment scoped_task_environment_;

  FakeDebugDaemonClient debug_daemon_client_;
  TestingPrefServiceSimple local_state_;
};

TEST_F(SchedulerConfigurationManagerTest, Startup) {
  local_state_.SetString(prefs::kSchedulerConfiguration, "initial");
  debug_daemon_client_.SetServiceIsAvailable(false);

  // Manager waits on initialization for service to be available.
  SchedulerConfigurationManager manager(&debug_daemon_client_, &local_state_);
  scoped_task_environment_.RunUntilIdle();
  EXPECT_EQ("", debug_daemon_client_.scheduler_configuration_name());

  // Config changes don't lead to updates while debugd isn't ready.
  local_state_.SetString(prefs::kSchedulerConfiguration, "config");
  scoped_task_environment_.RunUntilIdle();
  EXPECT_EQ("", debug_daemon_client_.scheduler_configuration_name());

  // Once the debugd service becomes available, the config gets set.
  debug_daemon_client_.SetServiceIsAvailable(true);
  scoped_task_environment_.RunUntilIdle();
  EXPECT_EQ("config", debug_daemon_client_.scheduler_configuration_name());
}

TEST_F(SchedulerConfigurationManagerTest, ConfigChange) {
  // Correct default is used when there is no configured value.
  SchedulerConfigurationManager manager(&debug_daemon_client_, &local_state_);
  scoped_task_environment_.RunUntilIdle();
  EXPECT_EQ(debugd::scheduler_configuration::kConservativeScheduler,
            debug_daemon_client_.scheduler_configuration_name());

  // Change user pref, which should trigger a config change.
  local_state_.SetUserPref(prefs::kSchedulerConfiguration,
                           std::make_unique<base::Value>("user"));
  scoped_task_environment_.RunUntilIdle();
  EXPECT_EQ("user", debug_daemon_client_.scheduler_configuration_name());

  // Set a policy, which should override the user setting
  local_state_.SetManagedPref(prefs::kSchedulerConfiguration,
                              std::make_unique<base::Value>("policy"));
  scoped_task_environment_.RunUntilIdle();
  EXPECT_EQ("policy", debug_daemon_client_.scheduler_configuration_name());

  // Dropping the user pref doesn't change anything.
  local_state_.RemoveUserPref(prefs::kSchedulerConfiguration);
  scoped_task_environment_.RunUntilIdle();
  EXPECT_EQ("policy", debug_daemon_client_.scheduler_configuration_name());

  // Dropping the policy as well reverts to the default configuration.
  local_state_.RemoveManagedPref(prefs::kSchedulerConfiguration);
  scoped_task_environment_.RunUntilIdle();
  EXPECT_EQ(debugd::scheduler_configuration::kConservativeScheduler,
            debug_daemon_client_.scheduler_configuration_name());
}

TEST_F(SchedulerConfigurationManagerTest, FinchDefault) {
  auto feature_list = std::make_unique<base::test::ScopedFeatureList>();
  feature_list->InitAndEnableFeatureWithParameters(
      features::kSchedulerConfiguration, {{"config", "finch"}});

  // Finch parameter selects the default.
  SchedulerConfigurationManager manager(&debug_daemon_client_, &local_state_);
  scoped_task_environment_.RunUntilIdle();
  EXPECT_EQ("finch", debug_daemon_client_.scheduler_configuration_name());

  // Config values override finch default.
  local_state_.SetString(prefs::kSchedulerConfiguration, "config");
  scoped_task_environment_.RunUntilIdle();
  EXPECT_EQ("config", debug_daemon_client_.scheduler_configuration_name());
}

}  // namespace chromeos
