// Copyright (c) 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_GCM_DRIVER_GCM_PROFILE_SERVICE_H_
#define COMPONENTS_GCM_DRIVER_GCM_PROFILE_SERVICE_H_

#include <memory>
#include <string>

#include "base/callback_forward.h"
#include "base/compiler_specific.h"
#include "base/files/file_path.h"
#include "base/macros.h"
#include "base/memory/ref_counted.h"
#include "base/memory/weak_ptr.h"
#include "build/build_config.h"
#include "components/gcm_driver/gcm_buildflags.h"
#include "components/keyed_service/core/keyed_service.h"
#include "components/version_info/version_info.h"
#include "services/network/public/mojom/proxy_resolving_socket.mojom.h"

class PrefService;

namespace base {
class SequencedTaskRunner;
}

namespace identity {
class IdentityManager;
}

namespace network {
class NetworkConnectionTracker;
class SharedURLLoaderFactory;
}

namespace gcm {

class GCMClientFactory;
class GCMDriver;

// Providing GCM service, via GCMDriver.
class GCMProfileService : public KeyedService {
 public:
  using GetProxyResolvingFactoryCallback = base::RepeatingCallback<void(
      network::mojom::ProxyResolvingSocketFactoryRequest)>;

#if BUILDFLAG(USE_GCM_FROM_PLATFORM)
  GCMProfileService(
      base::FilePath path,
      scoped_refptr<base::SequencedTaskRunner>& blocking_task_runner,
      scoped_refptr<network::SharedURLLoaderFactory> url_loader_factory);
#else
  GCMProfileService(
      PrefService* prefs,
      base::FilePath path,
      base::RepeatingCallback<
          void(base::WeakPtr<GCMProfileService>,
               network::mojom::ProxyResolvingSocketFactoryRequest)>
          get_socket_factory_callback,
      scoped_refptr<network::SharedURLLoaderFactory> url_loader_factory,
      network::NetworkConnectionTracker* network_connection_tracker,
      version_info::Channel channel,
      const std::string& product_category_for_subtypes,
      identity::IdentityManager* identity_manager,
      std::unique_ptr<GCMClientFactory> gcm_client_factory,
      const scoped_refptr<base::SequencedTaskRunner>& ui_task_runner,
      const scoped_refptr<base::SequencedTaskRunner>& io_task_runner,
      scoped_refptr<base::SequencedTaskRunner>& blocking_task_runner);
#endif
  ~GCMProfileService() override;

  // Returns whether GCM is enabled.
  static bool IsGCMEnabled(PrefService* prefs);

  // KeyedService:
  void Shutdown() override;

  // For testing purposes.
  void SetDriverForTesting(std::unique_ptr<GCMDriver> driver);

  GCMDriver* driver() const { return driver_.get(); }

 protected:
  // Used for constructing fake GCMProfileService for testing purpose.
  GCMProfileService();

 private:
  std::unique_ptr<GCMDriver> driver_;

#if !BUILDFLAG(USE_GCM_FROM_PLATFORM)
  identity::IdentityManager* identity_manager_;

  scoped_refptr<network::SharedURLLoaderFactory> url_loader_factory_;

  // Used for both account tracker and GCM.UserSignedIn UMA.
  class IdentityObserver;
  std::unique_ptr<IdentityObserver> identity_observer_;
#endif

  GetProxyResolvingFactoryCallback get_socket_factory_callback_;

  // WeakPtr generated by the factory must be dereferenced on the UI thread.
  base::WeakPtrFactory<GCMProfileService> weak_ptr_factory_{this};

  DISALLOW_COPY_AND_ASSIGN(GCMProfileService);
};

}  // namespace gcm

#endif  // COMPONENTS_GCM_DRIVER_GCM_PROFILE_SERVICE_H_
