// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/password_manager/core/browser/http_auth_manager_impl.h"

#include <utility>

#include "components/autofill/core/common/password_form.h"
#include "components/autofill/core/common/save_password_progress_logger.h"
#include "components/password_manager/core/browser/form_saver_impl.h"
#include "components/password_manager/core/browser/new_password_form_manager.h"
#include "components/password_manager/core/browser/password_form_manager_for_ui.h"
#include "components/password_manager/core/browser/password_manager_client.h"
#include "components/password_manager/core/browser/password_manager_util.h"

using autofill::PasswordForm;

namespace password_manager {

namespace {
using Logger = autofill::SavePasswordProgressLogger;
}  // anonymous namespace

HttpAuthManagerImpl::HttpAuthManagerImpl(PasswordManagerClient* client)
    : client_(client), observer_(nullptr) {
  DCHECK(client_);
}

HttpAuthManagerImpl::HttpAuthManagerImpl(PasswordManagerClient* client,
                                         HttpAuthObserver* observer,
                                         const PasswordForm& observed_form)
    : HttpAuthManagerImpl(client) {
  SetObserverAndDeliverCredentials(observer, observed_form);
}

HttpAuthManagerImpl::~HttpAuthManagerImpl() {
  if (observer_)
    observer_->OnLoginModelDestroying();
}

void HttpAuthManagerImpl::DetachObserver(HttpAuthObserver* observer) {
  // Only detach |observer| if it has the same memory address as the stored
  // |observer_|.
  if (observer == observer_) {
    LogMessage(Logger::STRING_HTTPAUTH_ON_DETACH_OBSERVER);
    observer_ = nullptr;
  }
}

void HttpAuthManagerImpl::SetObserverAndDeliverCredentials(
    HttpAuthObserver* observer,
    const PasswordForm& observed_form) {
  LogMessage(Logger::STRING_HTTPAUTH_ON_SET_OBSERVER);
  // Set the observer and communicate the signon_realm.
  // If a previous view registered itself as an observer, it must be notified
  // about its replacement.
  if (observer_)
    observer_->OnLoginModelDestroying();
  observer_ = observer;
  // Initialize the form manager.
  form_manager_ = std::make_unique<NewPasswordFormManager>(
      client_, PasswordStore::FormDigest(observed_form),
      nullptr, /* form_fetcher */
      std::make_unique<FormSaverImpl>(client_->GetPasswordStore()));
}

void HttpAuthManagerImpl::ProvisionallySaveForm(
    const PasswordForm& password_form) {
  if (form_manager_)
    form_manager_->ProvisionallySaveHttpAuthForm(password_form);
}

void HttpAuthManagerImpl::Autofill(
    const PasswordForm& preferred_match,
    const PasswordFormManagerForUI* form_manager) const {
  DCHECK_NE(PasswordForm::Scheme::kHtml, preferred_match.scheme);
  if (observer_ && (form_manager_.get() == form_manager) &&
      client_->IsFillingEnabled(form_manager_->GetOrigin())) {
    observer_->OnAutofillDataAvailable(preferred_match.username_value,
                                       preferred_match.password_value);
  }
}

void HttpAuthManagerImpl::OnPasswordFormSubmitted(
    const PasswordForm& password_form) {
  if (client_->IsSavingAndFillingEnabled(password_form.origin))
    ProvisionallySaveForm(password_form);
}

void HttpAuthManagerImpl::OnDidFinishMainFrameNavigation() {
  // The login was successful if and only if there were no HTTP errors.
  if (!client_->WasLastNavigationHTTPError()) {
    OnLoginSuccesfull();
  }
  form_manager_.reset();
}

void HttpAuthManagerImpl::OnLoginSuccesfull() {
  LogMessage(Logger::STRING_HTTPAUTH_ON_ASK_USER_OR_SAVE_PASSWORD);
  if (!form_manager_ ||
      !client_->IsSavingAndFillingEnabled(form_manager_->GetOrigin())) {
    return;
  }

  // ProvisionallySaveForm() might not have called, so |form_manager_| might be
  // not in submitted state. Do nothing in that case.
  if (!form_manager_->is_submitted())
    return;

  // TODO(crbug/831123) Move the logic into the PasswordFormManager.
  bool is_update = form_manager_->IsPasswordUpdate();
  bool is_new_login = form_manager_->IsNewLogin();
  if (is_update || is_new_login) {
    client_->PromptUserToSaveOrUpdatePassword(std::move(form_manager_),
                                              is_update);
    LogMessage(Logger::STRING_HTTPAUTH_ON_PROMPT_USER);
  }
}

void HttpAuthManagerImpl::LogMessage(const Logger::StringID msg) const {
  if (password_manager_util::IsLoggingActive(client_)) {
    BrowserSavePasswordProgressLogger logger(client_->GetLogManager());
    logger.LogMessage(msg);
  }
}
}  // namespace password_manager
