#!/usr/bin/env vpython
# Copyright 2018 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Runs WPT as an isolate bundle.

This script maps flags supported by run_isolate_script_test.py to flags that are
understood by WPT.

Here's the mapping [isolate script flag] : [wpt flag]
--isolated-script-test-output : --log-chromium
--total-shards : --total-chunks
--shard-index : -- this-chunk
"""

import json
import os
import sys

import common


class WPTTestAdapter(common.BaseIsolatedScriptArgsAdapter):

    def generate_test_output_args(self, output):
        return ['--log-chromium', output]

    def generate_sharding_args(self, total_shards, shard_index):
        return ['--total-chunks=%d' % total_shards,
                # shard_index is 0-based but WPT's this-chunk to be 1-based
                '--this-chunk=%d' % (shard_index + 1)]

    @property
    def rest_args(self):
        rest_args = super(WPTTestAdapter, self).rest_args
        # Here we add all of the arguments required to run WPT tests on Chrome.
        rest_args.extend([
            "../../third_party/blink/web_tests/external/wpt/wpt",
            "--venv=../../",
            "--skip-venv-setup",
            "run",
            "chrome",
            "--binary=../../out/Release/chrome",
            "--binary-arg=--host-resolver-rules="
                "MAP nonexistent.*.test ~NOTFOUND, MAP *.test 127.0.0.1",
            "--binary-arg=--enable-experimental-web-platform-features",
            "--binary-arg=--enable-blink-features=MojoJS,MojoJSTest",
            "--webdriver-binary=../../out/Release/chromedriver",
            "--headless",
            # TODO(lpz): Consider removing --processes and compute automatically
            # from multiprocessing.cpu_count()
            "--processes=10",
            "--no-capture-stdio",
            "--no-manifest-update",
            "--no-manifest-download",
            "--no-pause-after-test",
            "--no-fail-on-unexpected",
            "--metadata=../../out/Release/wpt_expectations_metadata/",
            # By specifying metadata above, WPT will try to find manifest in the
            # metadata directory. So here we point it back to the correct path
            # for the manifest.
            "--manifest=../../third_party/blink/web_tests/external/"
                "WPT_BASE_MANIFEST_6.json"
        ])
        return rest_args

    def add_extra_arguments(self, parser):
        # These args are used to rewrite the output generated by WPT to include
        # missing features, such as flakineess expectations.
        parser.add_argument("--old-json-output-file-path")
        parser.add_argument("--new-json-output-dir")
        parser.add_argument("--new-json-output-filename")

    def clean_up_after_test_run(self):
        common.run_command([
            sys.executable,
            os.path.join(common.SRC_DIR, 'third_party', 'blink', 'tools',
                         'update_wpt_output.py'),
            '--old-json-output-file-path',
            self.options.old_json_output_file_path,
            '--new-json-output-dir', self.options.new_json_output_dir,
            '--new-json-output-filename', self.options.new_json_output_filename,
        ])


def main():
    adapter = WPTTestAdapter()
    return adapter.run_test()


# This is not really a "script test" so does not need to manually add
# any additional compile targets.
def main_compile_targets(args):
    json.dump([], args.output)


if __name__ == '__main__':
    # Conform minimally to the protocol defined by ScriptTest.
    if 'compile_targets' in sys.argv:
        funcs = {
            'run': None,
            'compile_targets': main_compile_targets,
        }
        sys.exit(common.run_script(sys.argv[1:], funcs))
    sys.exit(main())
