//===----------------------------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//
//
// REQUIRES: long_tests

// <random>

// template<class RealType = double>
// class lognormal_distribution

// template<class _URNG> result_type operator()(_URNG& g, const param_type& parm);

#include <random>
#include <cassert>
#include <vector>
#include <numeric>

template <class T>
inline
T
sqr(T x)
{
    return x * x;
}

void
test1()
{
    typedef std::lognormal_distribution<> D;
    typedef D::param_type P;
    typedef std::mt19937 G;
    G g;
    D d;
    P p(-1./8192, 0.015625);
    const int N = 1000000;
    std::vector<D::result_type> u;
    for (int i = 0; i < N; ++i)
    {
        D::result_type v = d(g, p);
        assert(v > 0);
        u.push_back(v);
    }
    double mean = std::accumulate(u.begin(), u.end(), 0.0) / u.size();
    double var = 0;
    double skew = 0;
    double kurtosis = 0;
    for (unsigned i = 0; i < u.size(); ++i)
    {
        double dbl = (u[i] - mean);
        double d2 = sqr(dbl);
        var += d2;
        skew += dbl * d2;
        kurtosis += d2 * d2;
    }
    var /= u.size();
    double dev = std::sqrt(var);
    skew /= u.size() * dev * var;
    kurtosis /= u.size() * var * var;
    kurtosis -= 3;
    double x_mean = std::exp(p.m() + sqr(p.s())/2);
    double x_var = (std::exp(sqr(p.s())) - 1) * std::exp(2*p.m() + sqr(p.s()));
    double x_skew = (std::exp(sqr(p.s())) + 2) *
          std::sqrt((std::exp(sqr(p.s())) - 1));
    double x_kurtosis = std::exp(4*sqr(p.s())) + 2*std::exp(3*sqr(p.s())) +
                      3*std::exp(2*sqr(p.s())) - 6;
    assert(std::abs((mean - x_mean) / x_mean) < 0.01);
    assert(std::abs((var - x_var) / x_var) < 0.01);
    assert(std::abs((skew - x_skew) / x_skew) < 0.05);
    assert(std::abs((kurtosis - x_kurtosis) / x_kurtosis) < 0.25);
}

void
test2()
{
    typedef std::lognormal_distribution<> D;
    typedef D::param_type P;
    typedef std::mt19937 G;
    G g;
    D d;
    P p(-1./32, 0.25);
    const int N = 1000000;
    std::vector<D::result_type> u;
    for (int i = 0; i < N; ++i)
    {
        D::result_type v = d(g, p);
        assert(v > 0);
        u.push_back(v);
    }
    double mean = std::accumulate(u.begin(), u.end(), 0.0) / u.size();
    double var = 0;
    double skew = 0;
    double kurtosis = 0;
    for (unsigned i = 0; i < u.size(); ++i)
    {
        double dbl = (u[i] - mean);
        double d2 = sqr(dbl);
        var += d2;
        skew += dbl * d2;
        kurtosis += d2 * d2;
    }
    var /= u.size();
    double dev = std::sqrt(var);
    skew /= u.size() * dev * var;
    kurtosis /= u.size() * var * var;
    kurtosis -= 3;
    double x_mean = std::exp(p.m() + sqr(p.s())/2);
    double x_var = (std::exp(sqr(p.s())) - 1) * std::exp(2*p.m() + sqr(p.s()));
    double x_skew = (std::exp(sqr(p.s())) + 2) *
          std::sqrt((std::exp(sqr(p.s())) - 1));
    double x_kurtosis = std::exp(4*sqr(p.s())) + 2*std::exp(3*sqr(p.s())) +
                      3*std::exp(2*sqr(p.s())) - 6;
    assert(std::abs((mean - x_mean) / x_mean) < 0.01);
    assert(std::abs((var - x_var) / x_var) < 0.01);
    assert(std::abs((skew - x_skew) / x_skew) < 0.01);
    assert(std::abs((kurtosis - x_kurtosis) / x_kurtosis) < 0.03);
}

void
test3()
{
    typedef std::lognormal_distribution<> D;
    typedef D::param_type P;
    typedef std::mt19937 G;
    G g;
    D d;
    P p(-1./8, 0.5);
    const int N = 1000000;
    std::vector<D::result_type> u;
    for (int i = 0; i < N; ++i)
    {
        D::result_type v = d(g, p);
        assert(v > 0);
        u.push_back(v);
    }
    double mean = std::accumulate(u.begin(), u.end(), 0.0) / u.size();
    double var = 0;
    double skew = 0;
    double kurtosis = 0;
    for (unsigned i = 0; i < u.size(); ++i)
    {
        double dbl = (u[i] - mean);
        double d2 = sqr(dbl);
        var += d2;
        skew += dbl * d2;
        kurtosis += d2 * d2;
    }
    var /= u.size();
    double dev = std::sqrt(var);
    skew /= u.size() * dev * var;
    kurtosis /= u.size() * var * var;
    kurtosis -= 3;
    double x_mean = std::exp(p.m() + sqr(p.s())/2);
    double x_var = (std::exp(sqr(p.s())) - 1) * std::exp(2*p.m() + sqr(p.s()));
    double x_skew = (std::exp(sqr(p.s())) + 2) *
          std::sqrt((std::exp(sqr(p.s())) - 1));
    double x_kurtosis = std::exp(4*sqr(p.s())) + 2*std::exp(3*sqr(p.s())) +
                      3*std::exp(2*sqr(p.s())) - 6;
    assert(std::abs((mean - x_mean) / x_mean) < 0.01);
    assert(std::abs((var - x_var) / x_var) < 0.01);
    assert(std::abs((skew - x_skew) / x_skew) < 0.02);
    assert(std::abs((kurtosis - x_kurtosis) / x_kurtosis) < 0.05);
}

void
test4()
{
    typedef std::lognormal_distribution<> D;
    typedef D::param_type P;
    typedef std::mt19937 G;
    G g;
    D d(3, 4);
    P p;
    const int N = 1000000;
    std::vector<D::result_type> u;
    for (int i = 0; i < N; ++i)
    {
        D::result_type v = d(g, p);
        assert(v > 0);
        u.push_back(v);
    }
    double mean = std::accumulate(u.begin(), u.end(), 0.0) / u.size();
    double var = 0;
    double skew = 0;
    double kurtosis = 0;
    for (unsigned i = 0; i < u.size(); ++i)
    {
        double dbl = (u[i] - mean);
        double d2 = sqr(dbl);
        var += d2;
        skew += dbl * d2;
        kurtosis += d2 * d2;
    }
    var /= u.size();
    double dev = std::sqrt(var);
    skew /= u.size() * dev * var;
    kurtosis /= u.size() * var * var;
    kurtosis -= 3;
    double x_mean = std::exp(p.m() + sqr(p.s())/2);
    double x_var = (std::exp(sqr(p.s())) - 1) * std::exp(2*p.m() + sqr(p.s()));
    double x_skew = (std::exp(sqr(p.s())) + 2) *
          std::sqrt((std::exp(sqr(p.s())) - 1));
    double x_kurtosis = std::exp(4*sqr(p.s())) + 2*std::exp(3*sqr(p.s())) +
                      3*std::exp(2*sqr(p.s())) - 6;
    assert(std::abs((mean - x_mean) / x_mean) < 0.01);
    assert(std::abs((var - x_var) / x_var) < 0.02);
    assert(std::abs((skew - x_skew) / x_skew) < 0.08);
    assert(std::abs((kurtosis - x_kurtosis) / x_kurtosis) < 0.4);
}

void
test5()
{
    typedef std::lognormal_distribution<> D;
    typedef D::param_type P;
    typedef std::mt19937 G;
    G g;
    D d;
    P p(-0.78125, 1.25);
    const int N = 1000000;
    std::vector<D::result_type> u;
    for (int i = 0; i < N; ++i)
    {
        D::result_type v = d(g, p);
        assert(v > 0);
        u.push_back(v);
    }
    double mean = std::accumulate(u.begin(), u.end(), 0.0) / u.size();
    double var = 0;
    double skew = 0;
    double kurtosis = 0;
    for (unsigned i = 0; i < u.size(); ++i)
    {
        double dbl = (u[i] - mean);
        double d2 = sqr(dbl);
        var += d2;
        skew += dbl * d2;
        kurtosis += d2 * d2;
    }
    var /= u.size();
    double dev = std::sqrt(var);
    skew /= u.size() * dev * var;
    kurtosis /= u.size() * var * var;
    kurtosis -= 3;
    double x_mean = std::exp(p.m() + sqr(p.s())/2);
    double x_var = (std::exp(sqr(p.s())) - 1) * std::exp(2*p.m() + sqr(p.s()));
    double x_skew = (std::exp(sqr(p.s())) + 2) *
          std::sqrt((std::exp(sqr(p.s())) - 1));
    double x_kurtosis = std::exp(4*sqr(p.s())) + 2*std::exp(3*sqr(p.s())) +
                      3*std::exp(2*sqr(p.s())) - 6;
    assert(std::abs((mean - x_mean) / x_mean) < 0.01);
    assert(std::abs((var - x_var) / x_var) < 0.04);
    assert(std::abs((skew - x_skew) / x_skew) < 0.2);
    assert(std::abs((kurtosis - x_kurtosis) / x_kurtosis) < 0.7);
}

int main(int, char**)
{
    test1();
    test2();
    test3();
    test4();
    test5();

  return 0;
}
