# Copyright 2019 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Implements the interface of the results_processor module.

Provides functions to process intermediate results, and the entry point to
the standalone version of Results Processor.
"""

import datetime
import json
import logging
import os
import random
import re

from py_utils import cloud_storage
from core.results_processor import command_line
from core.results_processor import compute_metrics
from core.results_processor import formatters
from core.results_processor import util

from tracing.trace_data import trace_data
from tracing.value.diagnostics import date_range
from tracing.value.diagnostics import generic_set
from tracing.value.diagnostics import reserved_infos
from tracing.value import histogram
from tracing.value import histogram_set
from tracing.value import legacy_unit_info

# Telemetry results file is deprecated.
# TODO(crbug.com/981349): Remove this constant after Telemetry swithes to
# the new file.
TELEMETRY_RESULTS = '_telemetry_results.jsonl'
TEST_RESULTS = '_test_results.jsonl'
DIAGNOSTICS_NAME = 'diagnostics.json'
MEASUREMENTS_NAME = 'measurements.json'

FORMATS_WITH_METRICS = ['csv', 'histograms', 'html']


def ProcessResults(options):
  """Process intermediate results and produce the requested outputs.

  This function takes the intermediate results generated by Telemetry after
  running benchmarks (including artifacts such as traces, etc.), and processes
  them as requested by the result processing options.

  Args:
    options: An options object with values parsed from the command line and
      after any adjustments from ProcessOptions were applied.
  """
  if not getattr(options, 'output_formats', None):
    return 0

  test_results = _LoadTestResults(options.intermediate_dir)
  if not test_results:
    # TODO(crbug.com/981349): Make sure that no one is expecting Results
    # Processor to output results in the case of empty input
    # and make this an error.
    logging.warning('No test results to process.')

  upload_bucket = options.upload_bucket
  results_label = options.results_label
  test_suite_start = (test_results[0]['startTime'] if test_results
                      else datetime.datetime.utcnow().isoformat() + 'Z')
  run_identifier = RunIdentifier(results_label, test_suite_start)
  should_compute_metrics = any(
      fmt in FORMATS_WITH_METRICS for fmt in options.output_formats)

  util.ApplyInParallel(
      lambda result: ProcessTestResult(
          result, upload_bucket, results_label, run_identifier,
          test_suite_start, should_compute_metrics),
      test_results,
      on_failure=lambda result: result.update(status='FAIL'),
  )

  if should_compute_metrics:
    histogram_dicts = ExtractHistograms(test_results)

  for output_format in options.output_formats:
    logging.info('Processing format: %s', output_format)
    formatter = formatters.FORMATTERS[output_format]
    if output_format in FORMATS_WITH_METRICS:
      formatter.ProcessHistogramDicts(histogram_dicts, options)
    else:
      formatter.ProcessIntermediateResults(test_results, options)

  return GenerateExitCode(test_results)


def ProcessTestResult(test_result, upload_bucket, results_label,
                      run_identifier, test_suite_start, should_compute_metrics):
  AggregateTraces(test_result)
  if upload_bucket is not None:
    UploadArtifacts(test_result, upload_bucket, run_identifier)

  if should_compute_metrics:
    test_result['_histograms'] = histogram_set.HistogramSet()
    compute_metrics.ComputeTBMv2Metrics(test_result)
    ExtractMeasurements(test_result)
    AddDiagnosticsToHistograms(test_result, test_suite_start, results_label)


def ExtractHistograms(test_results):
  histograms = histogram_set.HistogramSet()
  for result in test_results:
    histograms.Merge(result['_histograms'])
  histograms.DeduplicateDiagnostics()
  return histograms.AsDicts()


def GenerateExitCode(test_results):
  """Generate an exit code as expected by callers.

  Returns:
    1 if there were failed tests.
    -1 if all tests were skipped.
    0 otherwise.
  """
  if any(r['status'] == 'FAIL' for r in test_results):
    return 1
  if all(r['status'] == 'SKIP' for r in test_results):
    return -1
  return 0


def _LoadTestResults(intermediate_dir):
  """Load intermediate results from a file into a list of test results."""
  # Try to load the results from the new file first, then from the old one.
  # TODO(crbug.com/981349): Remove fallback when Telemetry switches to the
  # new format.
  intermediate_file = os.path.join(intermediate_dir, TEST_RESULTS)
  if not os.path.exists(intermediate_file):
    intermediate_file = os.path.join(intermediate_dir, TELEMETRY_RESULTS)

  benchmark_run = {}
  test_results = []
  with open(intermediate_file) as f:
    for line in f:
      record = json.loads(line)
      # TODO(crbug.com/981349): Stop reading benchmarkRun messages when
      # Telemetry switches to the new format.
      if 'benchmarkRun' in record:
        benchmark_run.update(record['benchmarkRun'])
      if 'testResult' in record:
        test_results.append(record['testResult'])
  for test_result in test_results:
    test_result['_benchmarkRun'] = benchmark_run
  return test_results


def AggregateTraces(test_result):
  """Replace individual traces with an aggregate one for each test result.

  For a test run with traces, generates an aggregate HTML trace. Removes
  all entries for individual traces and adds one entry for aggregate one.
  """
  artifacts = test_result.get('outputArtifacts', {})
  traces = [name for name in artifacts if name.startswith('trace/')]
  # TODO(crbug.com/981349): Stop checking for HTML_TRACE_NAME after
  # Telemetry does not aggregate traces anymore.
  if traces and compute_metrics.HTML_TRACE_NAME not in artifacts:
    trace_files = [artifacts[name]['filePath'] for name in traces]
    html_path = os.path.join(
        os.path.dirname(os.path.commonprefix(trace_files)),
        compute_metrics.HTML_TRACE_NAME)
    trace_data.SerializeAsHtml(trace_files, html_path)
    artifacts[compute_metrics.HTML_TRACE_NAME] = {
      'filePath': html_path,
      'contentType': 'text/html',
    }
  for name in traces:
    del artifacts[name]


def RunIdentifier(results_label, test_suite_start):
  """Construct an identifier for the current script run"""
  if results_label:
    identifier_parts = [re.sub(r'\W+', '_', results_label)]
  else:
    identifier_parts = []
  # Time is rounded to seconds and delimiters are removed.
  # The first 19 chars of the string match 'YYYY-MM-DDTHH:MM:SS'.
  identifier_parts.append(re.sub(r'\W+', '', test_suite_start[:19]))
  identifier_parts.append(str(random.randint(1, 1e5)))
  return '_'.join(identifier_parts)


def UploadArtifacts(test_result, upload_bucket, run_identifier):
  """Upload all artifacts to cloud.

  For a test run, uploads all its artifacts to cloud and sets remoteUrl
  fields in intermediate_results.
  """
  artifacts = test_result.get('outputArtifacts', {})
  for name, artifact in artifacts.iteritems():
    if 'remoteUrl' in artifact:
      continue
    # TODO(crbug.com/981349): Remove check for HISTOGRAM_DICTS_FILE
    # after Telemetry does not save histograms as an artifact anymore.
    # Another TODO(crbug.com/981349): Think of a more general way to
    # specify which artifacts deserve uploading.
    if name in [compute_metrics.HISTOGRAM_DICTS_FILE, MEASUREMENTS_NAME]:
      continue
    remote_name = '/'.join([run_identifier, test_result['testPath'], name])
    artifact['remoteUrl'] = cloud_storage.Insert(
        upload_bucket, remote_name, artifact['filePath'])
    logging.info('Uploaded %s of %s to %s', name, test_result['testPath'],
                 artifact['remoteUrl'])


def AddDiagnosticsToHistograms(test_result, test_suite_start, results_label):
  """Add diagnostics to all histograms of a test run.

  Reads diagnostics from the test artifact and adds them to all histograms.
  This overwrites the corresponding diagnostics previously set by e.g.
  run_metrics.
  """
  artifacts = test_result.get('outputArtifacts', {})
  if DIAGNOSTICS_NAME in artifacts:
    with open(artifacts[DIAGNOSTICS_NAME]['filePath']) as f:
      diagnostics = json.load(f)['diagnostics']
  # TODO(crbug.com/981349): Remove this branch when Telemetry switches to the
  # new format.
  else:
    diagnostics = test_result.get('_benchmarkRun', {}).get('diagnostics', {})

  for name, diag in diagnostics.items():
    # For now, we only support GenericSet diagnostics that are serialized
    # as lists of values.
    assert isinstance(diag, list)
    test_result['_histograms'].AddSharedDiagnosticToAllHistograms(
        name, generic_set.GenericSet(diag))

  timestamp_ms = util.IsoTimestampToEpoch(test_suite_start) * 1e3
  test_result['_histograms'].AddSharedDiagnosticToAllHistograms(
      reserved_infos.BENCHMARK_START.name, date_range.DateRange(timestamp_ms))


  if results_label is not None:
    test_result['_histograms'].AddSharedDiagnosticToAllHistograms(
        reserved_infos.LABELS.name,
        generic_set.GenericSet([results_label]))


def MeasurementToHistogram(name, measurement):
  unit = measurement['unit']
  samples = measurement['samples']
  description = measurement.get('description')
  if unit in legacy_unit_info.LEGACY_UNIT_INFO:
    info = legacy_unit_info.LEGACY_UNIT_INFO[unit]
    unit = info.name
    samples = [s * info.conversion_factor for s in samples]
  if unit not in histogram.UNIT_NAMES:
    raise ValueError('Unknown unit: %s' % unit)
  return histogram.Histogram.Create(name, unit, samples,
                                    description=description)


def _StoryDiagnostics(test_result):
  """Extract diagnostics information about the specific story.

  These diagnostics will be added only to ad-hoc measurements recorded by
  benchmarks.
  """
  benchmark_name, story_name = test_result['testPath'].split('/', 1)
  story_tags = [tag['value'] for tag in test_result.get('tags', [])
                if tag['key'] == 'story_tag']
  return {
      reserved_infos.BENCHMARKS.name: generic_set.GenericSet([benchmark_name]),
      reserved_infos.STORIES.name: generic_set.GenericSet([story_name]),
      reserved_infos.STORY_TAGS.name: generic_set.GenericSet(story_tags),
  }


def ExtractMeasurements(test_result):
  """Add ad-hoc measurements to histogram dicts"""
  artifacts = test_result.get('outputArtifacts', {})
  if MEASUREMENTS_NAME in artifacts:
    with open(artifacts[MEASUREMENTS_NAME]['filePath']) as f:
      measurements = json.load(f)['measurements']
    diagnostics = _StoryDiagnostics(test_result)
    for name, measurement in measurements.iteritems():
      test_result['_histograms'].AddHistogram(
          MeasurementToHistogram(name, measurement), diagnostics=diagnostics)


def main(args=None):
  """Entry point for the standalone version of the results_processor script."""
  parser = command_line.ArgumentParser(standalone=True)
  options = parser.parse_args(args)
  command_line.ProcessOptions(options, standalone=True)
  return ProcessResults(options)
