// Copyright (c) 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "net/third_party/quiche/src/quic/core/qpack/qpack_decoded_headers_accumulator.h"

#include "net/third_party/quiche/src/quic/core/qpack/qpack_decoder.h"

namespace quic {

QpackDecodedHeadersAccumulator::QpackDecodedHeadersAccumulator(
    QuicStreamId id,
    QpackDecoder* qpack_decoder,
    Visitor* visitor,
    size_t max_header_list_size)
    : decoder_(qpack_decoder->CreateProgressiveDecoder(id, this)),
      visitor_(visitor),
      uncompressed_header_bytes_(0),
      compressed_header_bytes_(0),
      headers_decoded_(false),
      error_detected_(false) {
  quic_header_list_.set_max_header_list_size(max_header_list_size);
  quic_header_list_.OnHeaderBlockStart();
}

void QpackDecodedHeadersAccumulator::OnHeaderDecoded(QuicStringPiece name,
                                                     QuicStringPiece value) {
  DCHECK(!error_detected_);

  uncompressed_header_bytes_ += name.size() + value.size();
  quic_header_list_.OnHeader(name, value);
}

void QpackDecodedHeadersAccumulator::OnDecodingCompleted() {
  DCHECK(!headers_decoded_);
  DCHECK(!error_detected_);

  headers_decoded_ = true;
  quic_header_list_.OnHeaderBlockEnd(uncompressed_header_bytes_,
                                     compressed_header_bytes_);

  // Might destroy |this|.
  visitor_->OnHeadersDecoded(std::move(quic_header_list_));
}

void QpackDecodedHeadersAccumulator::OnDecodingErrorDetected(
    QuicStringPiece error_message) {
  DCHECK(!error_detected_);
  DCHECK(!headers_decoded_);

  error_detected_ = true;
  // Might destroy |this|.
  visitor_->OnHeaderDecodingError(error_message);
}

void QpackDecodedHeadersAccumulator::Decode(QuicStringPiece data) {
  DCHECK(!error_detected_);

  compressed_header_bytes_ += data.size();
  // Might destroy |this|.
  decoder_->Decode(data);
}

void QpackDecodedHeadersAccumulator::EndHeaderBlock() {
  DCHECK(!error_detected_);
  DCHECK(!headers_decoded_);

  // Might destroy |this|.
  decoder_->EndHeaderBlock();
}

}  // namespace quic
