// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CONTENT_TEST_STORAGE_PARTITION_TEST_UTILS_H_
#define CONTENT_TEST_STORAGE_PARTITION_TEST_UTILS_H_

#include <memory>

#include "base/macros.h"
#include "base/memory/scoped_refptr.h"
#include "content/public/browser/browser_thread.h"
#include "services/network/public/cpp/shared_url_loader_factory.h"
#include "services/network/public/mojom/url_loader_factory.mojom.h"

namespace content {

class URLLoaderFactoryGetter;

// Class to own the SharedURLLoaderFactory for use on the IO thread.
//
// Created on the UI thread and destroyed on the IO thread.
class IOThreadSharedURLLoaderFactoryOwner {
 public:
  using IOThreadSharedURLLoaderFactoryOwnerPtr =
      std::unique_ptr<IOThreadSharedURLLoaderFactoryOwner,
                      BrowserThread::DeleteOnIOThread>;

  // To be called on the UI thread. Will block and finish initialization on the
  // IO thread.
  static IOThreadSharedURLLoaderFactoryOwnerPtr Create(
      URLLoaderFactoryGetter* url_loader_factory_getter);
  static IOThreadSharedURLLoaderFactoryOwnerPtr Create(
      std::unique_ptr<network::PendingSharedURLLoaderFactory> info);

  // Load the given |url| with the internal |shared_url_loader_factory_| on IO
  // thread and return the |net::Error| code.
  int LoadBasicRequestOnIOThread(const GURL& url);

 private:
  friend struct BrowserThread::DeleteOnThread<BrowserThread::IO>;
  friend class base::DeleteHelper<IOThreadSharedURLLoaderFactoryOwner>;

  explicit IOThreadSharedURLLoaderFactoryOwner(
      URLLoaderFactoryGetter* url_loader_factory_getter);
  explicit IOThreadSharedURLLoaderFactoryOwner(
      std::unique_ptr<network::PendingSharedURLLoaderFactory> info);
  ~IOThreadSharedURLLoaderFactoryOwner();

  // Lives on the IO thread.
  scoped_refptr<network::SharedURLLoaderFactory> shared_url_loader_factory_;

  DISALLOW_COPY_AND_ASSIGN(IOThreadSharedURLLoaderFactoryOwner);
};

}  // namespace content

#endif  // CONTENT_TEST_STORAGE_PARTITION_TEST_UTILS_H_
