# Copyright 2019 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Implements the interface of the results_processor module.

Provides functions to parse command line arguments and process options.
"""

import argparse
import datetime
import logging
import os
import re
import sys

from py_utils import cloud_storage

from core import path_util
from core.results_processor import formatters
from core.results_processor import util
from core.tbmv3 import trace_processor


def ArgumentParser(standalone=False):
  """Create an ArgumentParser defining options required by the processor."""
  all_output_formats = formatters.FORMATTERS.keys()
  if not standalone:
    all_output_formats.append('none')
  parser, group = _CreateTopLevelParser(standalone)
  parser.add_argument(
      '-v', '--verbose', action='count', dest='verbosity',
      help='Increase verbosity level (repeat as needed)')
  group.add_argument(
      '--output-format', action='append', dest='output_formats',
      metavar='FORMAT', choices=all_output_formats, required=standalone,
      help=Sentences(
          'Output format to produce.',
          'May be used multiple times to produce multiple outputs.',
          'Avaliable formats: %(choices)s.',
          '' if standalone else 'Defaults to: html.'))
  group.add_argument(
      '--intermediate-dir', metavar='DIR_PATH', required=standalone,
      help=Sentences(
          'Path to a directory where intermediate results are stored.',
          '' if standalone else 'If not provided, the default is to create a '
          'new directory within "{output_dir}/artifacts/".'))
  group.add_argument(
      '--output-dir', default=_DefaultOutputDir(), metavar='DIR_PATH',
      help=Sentences(
          'Path to a directory where to write final results.',
          'Default: %(default)s.'))
  group.add_argument(
      '--max-values-per-test-case', type=int, metavar='NUM',
      help=Sentences(
          'Fail a test run if it produces more than this number of values.'
          'This includes both ad hoc and metric generated measurements.'))
  group.add_argument(
      '--reset-results', action='store_true',
      help=Sentences(
          'Overwrite any previous output files in the output directory.',
          'The default is to append to existing results.'))
  group.add_argument(
      '--results-label', metavar='LABEL',
      help='Label to identify the results generated by this run.')
  group.add_argument(
      '--test-path-format', metavar='FORMAT',
      choices=[util.TELEMETRY_TEST_PATH_FORMAT, util.GTEST_TEST_PATH_FORMAT],
      default=util.TELEMETRY_TEST_PATH_FORMAT,
      help=Sentences(
          'How to interpret the testPath attribute.',
          'Available options: %(choices)s. Default: %(default)s.'))
  group.add_argument(
      '--trace-processor-path',
      help=Sentences(
          'Path to trace processor shell.',
          'Default: try to guess based on common build directory names.'))
  group.add_argument(
      '--upload-results', action='store_true',
      help='Upload generated artifacts to cloud storage.')
  group.add_argument(
      '--upload-bucket', default='output', metavar='BUCKET',
      help=Sentences(
          'Storage bucket to use for uploading artifacts.',
          'Supported values are: %s; or a valid cloud storage bucket name.'
          % ', '.join(sorted(cloud_storage.BUCKET_ALIASES)),
          'Defaults to: %(default)s.'))
  group.add_argument(
      '--experimental-tbmv3-metrics', action='store_true',
      help='Enable running experimental TBMv3 metrics.')
  return parser


def ProcessOptions(options):
  """Adjust result processing options as needed before running benchmarks.

  Note: The intended scope of this function is limited to only adjust options
  defined by the ArgumentParser above. One should not attempt to read or modify
  any other attributes that the options object may have.

  Currently the main job of this function is to tease out and separate output
  formats to be handled by the results processor, from those that should fall
  back to the legacy output formatters in Telemetry.

  Args:
    options: An options object with values parsed from the command line.
  """
  if options.verbosity >= 2:
    logging.getLogger().setLevel(logging.DEBUG)
  elif options.verbosity == 1:
    logging.getLogger().setLevel(logging.INFO)
  else:
    logging.getLogger().setLevel(logging.WARNING)

  # The output_dir option is None or missing if the selected Telemetry command
  # does not involve output generation, e.g. "run_benchmark list", and the
  # argument parser defined above was not invoked.
  if getattr(options, 'output_dir', None) is None:
    return

  def resolve_dir(path):
    return os.path.realpath(os.path.expanduser(path))

  options.output_dir = resolve_dir(options.output_dir)

  if options.intermediate_dir:
    options.intermediate_dir = resolve_dir(options.intermediate_dir)
  else:
    if options.results_label:
      filesafe_label = re.sub(r'\W+', '_', options.results_label)
    else:
      filesafe_label = 'run'
    start_time = datetime.datetime.utcnow().strftime('%Y%m%dT%H%M%SZ')
    options.intermediate_dir = os.path.join(
        options.output_dir, 'artifacts', '%s_%s' % (filesafe_label, start_time))

  if options.upload_results:
    options.upload_bucket = cloud_storage.BUCKET_ALIASES.get(
        options.upload_bucket, options.upload_bucket)
  else:
    options.upload_bucket = None

  if not options.output_formats:
    options.output_formats = ['html']
  else:
    options.output_formats = sorted(set(options.output_formats))
  if 'none' in options.output_formats:
    options.output_formats.remove('none')

  if not options.trace_processor_path:
    options.trace_processor_path = _GuessTraceProcessorPath()


def _CreateTopLevelParser(standalone):
  """Create top level parser, and group for result options."""
  if standalone:
    parser = argparse.ArgumentParser(
        description='Standalone command line interface to results_processor.')
    # In standalone mode, both the parser and group are the same thing.
    return parser, parser
  else:
    parser = argparse.ArgumentParser(add_help=False)
    group = parser.add_argument_group(title='Result processor options')
    return parser, group


def _DefaultOutputDir():
  """Default output directory.

  Points to the directory of the benchmark runner script, if found, or the
  current working directory otherwise.
  """
  main_module = sys.modules['__main__']
  if hasattr(main_module, '__file__'):
    return os.path.realpath(os.path.dirname(main_module.__file__))
  else:
    return os.getcwd()


def _GuessTraceProcessorPath():
  """Return path to trace processor binary.

  When we run on bots, there's only one build directory, so we just return
  the path to trace processor binary located in that directory. Otherwise
  we don't guess, but leave it to the user to supply a path.
  """
  executable_names = [trace_processor.TP_BINARY_NAME,
                      trace_processor.TP_BINARY_NAME + '.exe']
  chromium_output_dir = os.environ.get('CHROMIUM_OUTPUT_DIR')
  if chromium_output_dir:
    for executable_name in executable_names:
      candidate_path = os.path.join(chromium_output_dir, executable_name)
      if os.path.isfile(candidate_path):
        return candidate_path

  build_dirs = ['build', 'out', 'xcodebuild']
  build_types = ['Debug', 'Debug_x64', 'Release', 'Release_x64', 'Default']
  candidate_paths = []
  for build_dir in build_dirs:
    for build_type in build_types:
      for executable_name in executable_names:
        candidate_path = os.path.join(
            path_util.GetChromiumSrcDir(), build_dir, build_type,
            executable_name)
        if os.path.isfile(candidate_path):
          candidate_paths.append(candidate_path)
  if len(candidate_paths) == 1:
    return candidate_paths[0]
  else:
    return None


def Sentences(*args):
  return ' '.join(s for s in args if s)
