# Copyright 2019 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Implements the interface of the results_processor module.

Provides functions to process intermediate results, and the entry point to
the standalone version of Results Processor.
"""

from __future__ import print_function

import datetime
import gzip
import json
import logging
import os
import posixpath
import random
import re
import shutil
import time

from py_utils import cloud_storage
from core.results_processor import command_line
from core.results_processor import compute_metrics
from core.results_processor import formatters
from core.results_processor import util
from core.tbmv3 import trace_processor

from tracing.trace_data import trace_data
from tracing.value.diagnostics import all_diagnostics
from tracing.value.diagnostics import generic_set
from tracing.value.diagnostics import reserved_infos
from tracing.value import histogram
from tracing.value import histogram_set
from tracing.value import legacy_unit_info

TEST_RESULTS = '_test_results.jsonl'
DIAGNOSTICS_NAME = 'diagnostics.json'
MEASUREMENTS_NAME = 'measurements.json'
CONVERTED_JSON_SUFFIX = '_converted.json'

FORMATS_WITH_METRICS = ['csv', 'histograms', 'html']


def ProcessResults(options):
  """Process intermediate results and produce the requested outputs.

  This function takes the intermediate results generated by Telemetry after
  running benchmarks (including artifacts such as traces, etc.), and processes
  them as requested by the result processing options.

  Args:
    options: An options object with values parsed from the command line and
      after any adjustments from ProcessOptions were applied.
  """
  if not getattr(options, 'output_formats', None):
    return 0

  test_results = _LoadTestResults(options.intermediate_dir)
  if not test_results:
    # TODO(crbug.com/981349): Make sure that no one is expecting Results
    # Processor to output results in the case of empty input
    # and make this an error.
    logging.warning('No test results to process.')

  test_suite_start = (test_results[0]['startTime']
      if test_results and 'startTime' in test_results[0]
      else datetime.datetime.utcnow().isoformat() + 'Z')
  run_identifier = RunIdentifier(options.results_label, test_suite_start)
  should_compute_metrics = any(
      fmt in FORMATS_WITH_METRICS for fmt in options.output_formats)

  begin_time = time.time()
  util.ApplyInParallel(
      lambda result: ProcessTestResult(
          test_result=result,
          upload_bucket=options.upload_bucket,
          results_label=options.results_label,
          run_identifier=run_identifier,
          test_suite_start=test_suite_start,
          should_compute_metrics=should_compute_metrics,
          max_num_values=options.max_values_per_test_case,
          test_path_format=options.test_path_format,
          trace_processor_path=options.trace_processor_path,
          enable_tbmv3=options.experimental_tbmv3_metrics),
      test_results,
      on_failure=util.SetUnexpectedFailure,
  )
  processing_duration = time.time() - begin_time
  _AmortizeProcessingDuration(processing_duration, test_results)

  if should_compute_metrics:
    histogram_dicts = ExtractHistograms(test_results)

  for output_format in options.output_formats:
    logging.info('Processing format: %s', output_format)
    formatter = formatters.FORMATTERS[output_format]
    if output_format in FORMATS_WITH_METRICS:
      output_file = formatter.ProcessHistogramDicts(histogram_dicts, options)
    else:
      output_file = formatter.ProcessIntermediateResults(test_results, options)
    print('View results at file://', output_file, sep='')

  return GenerateExitCode(test_results)


def _AmortizeProcessingDuration(processing_duration, test_results):
  test_results_count = len(test_results)
  if test_results_count:
    per_story_cost = processing_duration / len(test_results)
    logging.info(
        'Amortizing processing cost to story runtimes: %.2fs per story.',
        per_story_cost)
    for result in test_results:
      if 'runDuration' in result and result['runDuration']:
        current_duration = float(result['runDuration'].rstrip('s'))
        new_story_cost = current_duration + per_story_cost
        result['runDuration'] = unicode(str(new_story_cost) + 's', 'utf-8')


def ProcessTestResult(test_result, upload_bucket, results_label,
                      run_identifier, test_suite_start, should_compute_metrics,
                      max_num_values, test_path_format, trace_processor_path,
                      enable_tbmv3):
  ConvertProtoTraces(test_result, trace_processor_path)
  AggregateTBMv2Traces(test_result)
  if enable_tbmv3:
    AggregateTBMv3Traces(test_result)
  if upload_bucket is not None:
    UploadArtifacts(test_result, upload_bucket, run_identifier)

  if should_compute_metrics:
    test_result['_histograms'] = histogram_set.HistogramSet()
    compute_metrics.ComputeTBMv2Metrics(test_result)
    if enable_tbmv3:
      compute_metrics.ComputeTBMv3Metrics(test_result, trace_processor_path)
    ExtractMeasurements(test_result)
    num_values = len(test_result['_histograms'])
    if max_num_values is not None and num_values > max_num_values:
      logging.error('%s produced %d values, but only %d are allowed.',
                    test_result['testPath'], num_values, max_num_values)
      util.SetUnexpectedFailure(test_result)
      del test_result['_histograms']
    else:
      AddDiagnosticsToHistograms(test_result, test_suite_start, results_label,
                                 test_path_format)


def ExtractHistograms(test_results):
  histograms = histogram_set.HistogramSet()
  for result in test_results:
    if '_histograms' in result:
      histograms.Merge(result['_histograms'])
  histograms.DeduplicateDiagnostics()
  return histograms.AsDicts()


def GenerateExitCode(test_results):
  """Generate an exit code as expected by callers.

  Returns:
    1 if there were failed tests.
    111 if all tests were skipped. (See crbug.com/1019139#c8 for details).
    0 otherwise.
  """
  if any(r['status'] == 'FAIL' for r in test_results):
    return 1
  if all(r['status'] == 'SKIP' for r in test_results):
    return 111
  return 0


def _LoadTestResults(intermediate_dir):
  """Load intermediate results from a file into a list of test results."""
  intermediate_file = os.path.join(intermediate_dir, TEST_RESULTS)
  test_results = []
  with open(intermediate_file) as f:
    for line in f:
      record = json.loads(line)
      if 'testResult' in record:
        test_results.append(record['testResult'])
  return test_results


def _IsProtoTrace(trace_name):
  return (trace_name.startswith('trace/') and
          (trace_name.endswith('.pb') or trace_name.endswith('.pb.gz')))


def _IsTBMv2Trace(trace_name):
  return (trace_name.startswith('trace/') and
          (trace_name.endswith('.json') or trace_name.endswith('.json.gz') or
          trace_name.endswith('.txt') or trace_name.endswith('.txt.gz')))


def _BuildOutputPath(input_files, output_name):
  """Build a path to a file in the same folder as input_files."""
  return os.path.join(
      os.path.dirname(os.path.commonprefix(input_files)),
      output_name
  )


def ConvertProtoTraces(test_result, trace_processor_path):
  """Convert proto traces to json.

  For a test result with proto traces, converts them to json using
  trace_processor and stores the json trace as a separate artifact.
  """
  artifacts = test_result.get('outputArtifacts', {})
  proto_traces = [name for name in artifacts if _IsProtoTrace(name)]

  # TODO(crbug.com/990304): After implementation of TBMv3-style clock sync,
  # it will be possible to convert the aggregated proto trace, not
  # individual ones.
  for proto_trace_name in proto_traces:
    proto_file_path = artifacts[proto_trace_name]['filePath']
    json_file_path = (os.path.splitext(proto_file_path)[0] +
                      CONVERTED_JSON_SUFFIX)
    json_trace_name = (posixpath.splitext(proto_trace_name)[0] +
                       CONVERTED_JSON_SUFFIX)
    trace_processor.ConvertProtoTraceToJson(
        trace_processor_path, proto_file_path, json_file_path)
    artifacts[json_trace_name] = {
        'filePath': json_file_path,
        'contentType': 'application/json',
    }
    logging.info('%s: Proto trace converted. Source: %s. Destination: %s.',
                 test_result['testPath'], proto_file_path, json_file_path)


def AggregateTBMv2Traces(test_result):
  """Replace individual non-proto traces with an aggregate HTML trace.

  For a test result with non-proto traces, generates an aggregate HTML trace.
  Removes all entries for individual traces and adds one entry for
  the aggregate one.
  """
  artifacts = test_result.get('outputArtifacts', {})
  traces = [name for name in artifacts if _IsTBMv2Trace(name)]
  if traces:
    trace_files = [artifacts[name]['filePath'] for name in traces]
    html_path = _BuildOutputPath(trace_files, compute_metrics.HTML_TRACE_NAME)
    trace_data.SerializeAsHtml(trace_files, html_path)
    artifacts[compute_metrics.HTML_TRACE_NAME] = {
      'filePath': html_path,
      'contentType': 'text/html',
    }
    logging.info('%s: TBMv2 traces aggregated. Sources: %s. Destination: %s.',
                 test_result['testPath'], trace_files, html_path)
  for name in traces:
    del artifacts[name]


def AggregateTBMv3Traces(test_result):
  """Replace individual proto traces with an aggregate one.

  For a test result with proto traces, concatenates them into one file.
  Removes all entries for individual traces and adds one entry for
  the aggregate one.
  """
  artifacts = test_result.get('outputArtifacts', {})
  traces = [name for name in artifacts if _IsProtoTrace(name)]
  if traces:
    proto_files = [artifacts[name]['filePath'] for name in traces]
    concatenated_path = _BuildOutputPath(
        proto_files, compute_metrics.CONCATENATED_PROTO_NAME)
    with open(concatenated_path, 'w') as concatenated_trace:
      for trace_file in proto_files:
        if trace_file.endswith('.pb.gz'):
          with gzip.open(trace_file, 'rb') as f:
            shutil.copyfileobj(f, concatenated_trace)
        else:
          with open(trace_file, 'rb') as f:
            shutil.copyfileobj(f, concatenated_trace)
    artifacts[compute_metrics.CONCATENATED_PROTO_NAME] = {
      'filePath': concatenated_path,
      'contentType': 'application/x-protobuf',
    }
    logging.info('%s: Proto traces aggregated. Sources: %s. Destination: %s.',
                 test_result['testPath'], proto_files, concatenated_path)
  for name in traces:
    del artifacts[name]


def RunIdentifier(results_label, test_suite_start):
  """Construct an identifier for the current script run"""
  if results_label:
    identifier_parts = [re.sub(r'\W+', '_', results_label)]
  else:
    identifier_parts = []
  # Time is rounded to seconds and delimiters are removed.
  # The first 19 chars of the string match 'YYYY-MM-DDTHH:MM:SS'.
  identifier_parts.append(re.sub(r'\W+', '', test_suite_start[:19]))
  identifier_parts.append(str(random.randint(1, 1e5)))
  return '_'.join(identifier_parts)


def UploadArtifacts(test_result, upload_bucket, run_identifier):
  """Upload all artifacts to cloud.

  For a test run, uploads all its artifacts to cloud and sets fetchUrl and
  viewUrl fields in intermediate_results.
  """
  artifacts = test_result.get('outputArtifacts', {})
  for name, artifact in artifacts.iteritems():
    # TODO(crbug.com/981349): Think of a more general way to
    # specify which artifacts deserve uploading.
    if name in [DIAGNOSTICS_NAME, MEASUREMENTS_NAME]:
      continue
    retry_identifier = 'retry_%s' % test_result.get('resultId', '0')
    remote_name = '/'.join(
        [run_identifier, test_result['testPath'], retry_identifier, name])
    urlsafe_remote_name = re.sub(r'[^A-Za-z0-9/.-]+', '_', remote_name)
    cloud_filepath = cloud_storage.Upload(
        upload_bucket, urlsafe_remote_name, artifact['filePath'])
    # Per crbug.com/1033755 some services require fetchUrl.
    artifact['fetchUrl'] = cloud_filepath.fetch_url
    artifact['viewUrl'] = cloud_filepath.view_url
    logging.info('%s: Uploaded %s to %s', test_result['testPath'], name,
                 artifact['viewUrl'])


def GetTraceUrl(test_result):
  artifacts = test_result.get('outputArtifacts', {})
  trace_artifact = artifacts.get(compute_metrics.HTML_TRACE_NAME, {})
  if 'viewUrl' in trace_artifact:
    return trace_artifact['viewUrl']
  elif 'filePath' in trace_artifact:
    return 'file://' + trace_artifact['filePath']
  else:
    return None


def AddDiagnosticsToHistograms(test_result, test_suite_start, results_label,
                               test_path_format):
  """Add diagnostics to all histograms of a test result.

  Reads diagnostics from the test artifact and adds them to all histograms.
  Also sets additional diagnostics based on test result metadata.
  This overwrites the corresponding diagnostics previously set by e.g.
  run_metrics.
  """
  artifacts = test_result.get('outputArtifacts', {})
  if DIAGNOSTICS_NAME in artifacts:
    with open(artifacts[DIAGNOSTICS_NAME]['filePath']) as f:
      diagnostics = json.load(f)['diagnostics']
    for name, diag in diagnostics.items():
      # For now, we only support GenericSet diagnostics that are serialized
      # as lists of values.
      assert isinstance(diag, list)
      test_result['_histograms'].AddSharedDiagnosticToAllHistograms(
          name, generic_set.GenericSet(diag))
    del artifacts[DIAGNOSTICS_NAME]

  test_suite, test_case = util.SplitTestPath(test_result, test_path_format)
  if 'startTime' in test_result:
    test_start_ms = util.IsoTimestampToEpoch(test_result['startTime']) * 1e3
  else:
    test_start_ms = None
  test_suite_start_ms = util.IsoTimestampToEpoch(test_suite_start) * 1e3
  story_tags = [tag['value'] for tag in test_result.get('tags', [])
                if tag['key'] == 'story_tag']
  result_id = int(test_result.get('resultId', 0))
  trace_url = GetTraceUrl(test_result)

  additional_diagnostics = [
      (reserved_infos.BENCHMARKS, test_suite),
      (reserved_infos.BENCHMARK_START, test_suite_start_ms),
      (reserved_infos.LABELS, results_label),
      (reserved_infos.STORIES, test_case),
      (reserved_infos.STORYSET_REPEATS, result_id),
      (reserved_infos.STORY_TAGS, story_tags),
      (reserved_infos.TRACE_START, test_start_ms),
      (reserved_infos.TRACE_URLS, trace_url),
  ]
  for name, value in _WrapDiagnostics(additional_diagnostics):
    test_result['_histograms'].AddSharedDiagnosticToAllHistograms(name, value)


def MeasurementToHistogram(name, measurement):
  unit = measurement['unit']
  samples = measurement['samples']
  description = measurement.get('description')
  if unit in legacy_unit_info.LEGACY_UNIT_INFO:
    info = legacy_unit_info.LEGACY_UNIT_INFO[unit]
    unit = info.name
    samples = [s * info.conversion_factor for s in samples]
  if unit not in histogram.UNIT_NAMES:
    raise ValueError('Unknown unit: %s' % unit)
  return histogram.Histogram.Create(name, unit, samples,
                                    description=description)


def _WrapDiagnostics(info_value_pairs):
  """Wrap diagnostic values in corresponding Diagnostics classes.

  Args:
    info_value_pairs: any iterable of pairs (info, value), where info is one
        of reserved infos defined in tracing.value.diagnostics.reserved_infos,
        and value can be any json-serializable object.

  Returns:
    An iterator over pairs (diagnostic name, diagnostic value).
  """
  for info, value in info_value_pairs:
    if value is None or value == []:
      continue
    if info.type == 'GenericSet' and not isinstance(value, list):
      value = [value]
    diag_class = all_diagnostics.GetDiagnosticClassForName(info.type)
    yield info.name, diag_class(value)


def ExtractMeasurements(test_result):
  """Add ad-hoc measurements to histogram dicts"""
  artifacts = test_result.get('outputArtifacts', {})
  if MEASUREMENTS_NAME in artifacts:
    with open(artifacts[MEASUREMENTS_NAME]['filePath']) as f:
      measurements = json.load(f)['measurements']
    for name, measurement in measurements.iteritems():
      test_result['_histograms'].AddHistogram(
          MeasurementToHistogram(name, measurement))
    del artifacts[MEASUREMENTS_NAME]


def main(args=None):
  """Entry point for the standalone version of the results_processor script."""
  parser = command_line.ArgumentParser(standalone=True)
  options = parser.parse_args(args)
  command_line.ProcessOptions(options)
  return ProcessResults(options)
