// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "services/tracing/public/cpp/perfetto/perfetto_producer.h"

#include "third_party/perfetto/include/perfetto/ext/tracing/core/shared_memory_arbiter.h"
#include "third_party/perfetto/include/perfetto/ext/tracing/core/startup_trace_writer_registry.h"
#include "third_party/perfetto/include/perfetto/ext/tracing/core/trace_writer.h"

namespace tracing {

// static
constexpr size_t PerfettoProducer::kSMBPageSizeBytes;

// static
constexpr size_t PerfettoProducer::kSMBSizeBytes;

PerfettoProducer::PerfettoProducer(PerfettoTaskRunner* task_runner)
    : task_runner_(task_runner) {
  DCHECK(task_runner_);
}

PerfettoProducer::~PerfettoProducer() = default;

std::unique_ptr<perfetto::TraceWriter>
PerfettoProducer::CreateStartupTraceWriter(uint32_t startup_session_id) {
  DCHECK(MaybeSharedMemoryArbiter());
  return MaybeSharedMemoryArbiter()->CreateStartupTraceWriter(
      startup_session_id);
}

void PerfettoProducer::BindStartupTargetBuffer(
    uint32_t startup_session_id,
    perfetto::BufferID startup_target_buffer) {
  DCHECK(MaybeSharedMemoryArbiter());
  MaybeSharedMemoryArbiter()->BindStartupTargetBuffer(startup_session_id,
                                                      startup_target_buffer);
}

std::unique_ptr<perfetto::TraceWriter> PerfettoProducer::CreateTraceWriter(
    perfetto::BufferID target_buffer,
    perfetto::BufferExhaustedPolicy buffer_exhausted_policy) {
  DCHECK(MaybeSharedMemoryArbiter());
  // Chromium uses BufferExhaustedPolicy::kDrop to avoid stalling trace writers
  // when the chunks in the SMB are exhausted. Stalling could otherwise lead to
  // deadlocks in chromium, because a stalled mojo IPC thread could prevent
  // CommitRequest messages from reaching the perfetto service.
  buffer_exhausted_policy = perfetto::BufferExhaustedPolicy::kDrop;
  return MaybeSharedMemoryArbiter()->CreateTraceWriter(target_buffer,
                                                       buffer_exhausted_policy);
}

void PerfettoProducer::DeleteSoonForTesting(
    std::unique_ptr<PerfettoProducer> perfetto_producer) {
  PerfettoTracedProcess::GetTaskRunner()->GetOrCreateTaskRunner()->DeleteSoon(
      FROM_HERE, std::move(perfetto_producer));
}

PerfettoTaskRunner* PerfettoProducer::task_runner() {
  return task_runner_;
}

}  // namespace tracing
