// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef WEBLAYER_BROWSER_NAVIGATION_IMPL_H_
#define WEBLAYER_BROWSER_NAVIGATION_IMPL_H_

#include "base/macros.h"
#include "build/build_config.h"
#include "weblayer/public/navigation.h"

#if defined(OS_ANDROID)
#include "base/android/scoped_java_ref.h"
#endif

namespace content {
class NavigationHandle;
}

namespace weblayer {

class NavigationImpl : public Navigation {
 public:
  explicit NavigationImpl(content::NavigationHandle* navigation_handle);
  ~NavigationImpl() override;

  void set_should_stop_when_throttle_created() {
    should_stop_when_throttle_created_ = true;
  }
  bool should_stop_when_throttle_created() const {
    return should_stop_when_throttle_created_;
  }

  void set_safe_to_set_request_headers(bool value) {
    safe_to_set_request_headers_ = value;
  }

#if defined(OS_ANDROID)
  void SetJavaNavigation(
      JNIEnv* env,
      const base::android::JavaParamRef<jobject>& java_navigation);
  int GetState(JNIEnv* env, const base::android::JavaParamRef<jobject>& obj) {
    return static_cast<int>(GetState());
  }
  base::android::ScopedJavaLocalRef<jstring> GetUri(
      JNIEnv* env,
      const base::android::JavaParamRef<jobject>& obj);
  base::android::ScopedJavaLocalRef<jobjectArray> GetRedirectChain(
      JNIEnv* env,
      const base::android::JavaParamRef<jobject>& obj);
  int GetHttpStatusCode(JNIEnv* env,
                        const base::android::JavaParamRef<jobject>& obj) {
    return GetHttpStatusCode();
  }
  bool IsSameDocument(JNIEnv* env,
                      const base::android::JavaParamRef<jobject>& obj) {
    return IsSameDocument();
  }
  bool IsErrorPage(JNIEnv* env,
                   const base::android::JavaParamRef<jobject>& obj) {
    return IsErrorPage();
  }
  int GetLoadError(JNIEnv* env,
                   const base::android::JavaParamRef<jobject>& obj) {
    return static_cast<int>(GetLoadError());
  }
  jboolean SetRequestHeader(JNIEnv* env,
                            const base::android::JavaParamRef<jobject>& obj,
                            const base::android::JavaParamRef<jstring>& name,
                            const base::android::JavaParamRef<jstring>& value);

  base::android::ScopedJavaGlobalRef<jobject> java_navigation() {
    return java_navigation_;
  }
#endif

 private:
  // Navigation implementation:
  GURL GetURL() override;
  const std::vector<GURL>& GetRedirectChain() override;
  NavigationState GetState() override;
  int GetHttpStatusCode() override;
  bool IsSameDocument() override;
  bool IsErrorPage() override;
  LoadError GetLoadError() override;
  void SetRequestHeader(const std::string& name,
                        const std::string& value) override;

  content::NavigationHandle* navigation_handle_;

  // Used to delay calling Stop() until safe. See
  // NavigationControllerImpl::NavigationThrottleImpl for details.
  bool should_stop_when_throttle_created_ = false;

  // Whether SetRequestHeader() is allowed at this time.
  bool safe_to_set_request_headers_ = false;

#if defined(OS_ANDROID)
  base::android::ScopedJavaGlobalRef<jobject> java_navigation_;
#endif

  DISALLOW_COPY_AND_ASSIGN(NavigationImpl);
};

}  // namespace weblayer

#endif  // WEBLAYER_BROWSER_NAVIGATION_IMPL_H_
