// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROMEOS_PRINTING_CUPS_PRINTER_STATUS_H_
#define CHROMEOS_PRINTING_CUPS_PRINTER_STATUS_H_

#include <string>

#include "base/containers/flat_set.h"
#include "base/time/time.h"
#include "chromeos/chromeos_export.h"

namespace chromeos {

// A container for the results of a printer status query. A printer status query
// can return multiple error reasons so CupsPrinterStatus contains multiple
// CupsPrinterStatusReasons. |timestamp| is set at the time of CupsPrinterStatus
// creation.
class CHROMEOS_EXPORT CupsPrinterStatus {
 public:
  // A combination of a reason, which describes the state of a printer, and a
  // severity, which is the level of seriousness of that state.
  class CHROMEOS_EXPORT CupsPrinterStatusReason {
   public:
    enum class Reason {
      kConnectingToDevice = 0,
      kDeviceError,
      kDoorOpen,
      kLowOnInk,
      kLowOnPaper,
      kNoError,
      kOutOfInk,
      kOutOfPaper,
      kOutputAreaAlmostFull,
      kOutputFull,
      kPaperJam,
      kPaused,
      kPrinterQueueFull,
      kPrinterUnreachable,
      kStopped,
      kTrayMissing,
      kUnknownReason,
    };

    enum class Severity {
      kUnknownSeverity = 0,
      kReport,
      kWarning,
      kError,
    };

    CupsPrinterStatusReason(const Reason& reason, const Severity& severity);
    ~CupsPrinterStatusReason();

    const Reason& GetReason() const;
    const Severity& GetSeverity() const;

    bool operator==(const CupsPrinterStatusReason& other) const {
      return reason_ == other.reason_ && severity_ == other.severity_;
    }

    // Comparison operator to support storing CupsPrinterStatusReason in a
    // flat_set. Two CupsPrinterStatusReasons are considered matching iff
    // |reason| and |severity| are the same.
    bool operator<(const CupsPrinterStatusReason& other) const {
      return reason_ < other.reason_ ||
             (reason_ == other.reason_ && severity_ < other.severity_);
    }

   private:
    Reason reason_;
    Severity severity_;
  };

  // Creates a CupsPrinterStatus from an existing printing::PrinterStatus.
  explicit CupsPrinterStatus(const std::string& printer_id);

  CupsPrinterStatus();
  CupsPrinterStatus(const CupsPrinterStatus& other);
  CupsPrinterStatus& operator=(const CupsPrinterStatus& other);

  ~CupsPrinterStatus();

  const std::string& GetPrinterId() const;

  // Returns set of status reasons. Each reason describing status of the
  // printer.
  const base::flat_set<CupsPrinterStatusReason>& GetStatusReasons() const;

  const base::Time& GetTimestamp() const;

  // Adds a new CupsPrinterStatusReason to an existing CupsPrinterStatus.
  void AddStatusReason(const CupsPrinterStatusReason::Reason& reason,
                       const CupsPrinterStatusReason::Severity& severity);

 private:
  std::string printer_id_;
  base::flat_set<CupsPrinterStatusReason> status_reasons_;
  base::Time timestamp_;
};

}  // namespace chromeos

#endif  // CHROMEOS_PRINTING_CUPS_PRINTER_STATUS_H_
