// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROMEOS_SERVICES_ASSISTANT_PLATFORM_AUDIO_DEVICE_OWNER_H_
#define CHROMEOS_SERVICES_ASSISTANT_PLATFORM_AUDIO_DEVICE_OWNER_H_

#include <memory>
#include <string>
#include <vector>

#include "base/component_export.h"
#include "base/macros.h"
#include "chromeos/services/assistant/media_session/assistant_media_session.h"
#include "libassistant/shared/public/platform_audio_output.h"
#include "media/base/audio_block_fifo.h"
#include "media/base/audio_parameters.h"
#include "media/base/audio_renderer_sink.h"
#include "mojo/public/cpp/bindings/pending_remote.h"
#include "services/audio/public/cpp/output_device.h"
#include "services/audio/public/mojom/stream_factory.mojom.h"

namespace chromeos {
namespace assistant {

class COMPONENT_EXPORT(ASSISTANT_SERVICE) AudioDeviceOwner
    : public media::AudioRendererSink::RenderCallback,
      media_session::mojom::MediaSessionObserver {
 public:
  AudioDeviceOwner(
      scoped_refptr<base::SequencedTaskRunner> task_runner,
      scoped_refptr<base::SequencedTaskRunner> background_task_runner,
      const std::string& device_id);
  ~AudioDeviceOwner() override;

  void StartOnMainThread(
      AssistantMediaSession* media_session,
      assistant_client::AudioOutput::Delegate* delegate,
      mojo::PendingRemote<audio::mojom::StreamFactory> stream_factory,
      const assistant_client::OutputStreamFormat& format);

  void StopOnBackgroundThread();

  // media_session::mojom::MediaSessionObserver overrides:
  void MediaSessionInfoChanged(
      media_session::mojom::MediaSessionInfoPtr info) override;
  void MediaSessionMetadataChanged(
      const base::Optional<::media_session::MediaMetadata>& metadata) override {
  }
  void MediaSessionActionsChanged(
      const std::vector<media_session::mojom::MediaSessionAction>& action)
      override {}
  void MediaSessionImagesChanged(
      const base::flat_map<media_session::mojom::MediaSessionImageType,
                           std::vector<::media_session::MediaImage>>& images)
      override {}
  void MediaSessionPositionChanged(
      const base::Optional<::media_session::MediaPosition>& position) override {
  }

  // media::AudioRenderSink::RenderCallback overrides:
  int Render(base::TimeDelta delay,
             base::TimeTicks delay_timestamp,
             int prior_frames_skipped,
             media::AudioBus* dest) override;

  void OnRenderError() override;

  void SetDelegate(assistant_client::AudioOutput::Delegate* delegate);

 private:
  void StartDeviceOnBackgroundThread(
      mojo::PendingRemote<audio::mojom::StreamFactory> stream_factory,
      AssistantMediaSession* media_session);

  // Requests assistant to fill buffer with more data.
  void ScheduleFillLocked(const base::TimeTicks& time);

  // Callback for assistant to notify that it completes the filling.
  void BufferFillDone(int num_bytes);

  scoped_refptr<base::SequencedTaskRunner> main_task_runner_;
  scoped_refptr<base::SequencedTaskRunner> background_task_runner_;

  base::Lock lock_;
  std::unique_ptr<media::AudioBlockFifo> audio_fifo_;  // guarded by lock_.
  // Whether assistant is filling the buffer -- delegate_->FillBuffer is called
  // and BufferFillDone() is not called yet.
  // guarded by lock_.
  bool is_filling_ = false;

  // Guarded by |lock_|.
  assistant_client::AudioOutput::Delegate* delegate_;

  // Audio output device id used for output.
  std::string device_id_;
  std::unique_ptr<audio::OutputDevice> output_device_;
  // Stores audio frames generated by assistant.
  std::vector<uint8_t> audio_data_;
  assistant_client::OutputStreamFormat format_;
  media::AudioParameters audio_param_;

  mojo::Receiver<media_session::mojom::MediaSessionObserver> session_receiver_{
      this};

  DISALLOW_COPY_AND_ASSIGN(AudioDeviceOwner);
};

}  // namespace assistant
}  // namespace chromeos

#endif  // CHROMEOS_SERVICES_ASSISTANT_PLATFORM_AUDIO_DEVICE_OWNER_H_
