// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "services/device/serial/serial_device_enumerator_linux.h"

#include <stdint.h>

#include <memory>
#include <utility>
#include <vector>

#include "base/check_op.h"
#include "base/files/file_util.h"
#include "base/strings/string_number_conversions.h"
#include "base/strings/string_split.h"
#include "base/strings/string_util.h"
#include "base/strings/stringprintf.h"
#include "base/threading/scoped_blocking_call.h"
#include "device/udev_linux/udev.h"

namespace device {

namespace {

constexpr char kSubsystemTty[] = "tty";

// Holds information about a TTY driver for serial devices. Each driver creates
// device nodes with a given major number and in a range of minor numbers.
struct SerialDriverInfo {
  int major;
  int minor_start;
  int minor_end;  // Inclusive.
};

std::vector<SerialDriverInfo> ReadSerialDriverInfo() {
  std::string tty_drivers;
  if (!base::ReadFileToString(base::FilePath("/proc/tty/drivers"),
                              &tty_drivers)) {
    return {};
  }

  // Each line has information on a single TTY driver.
  std::vector<SerialDriverInfo> serial_drivers;
  for (const auto& line :
       base::SplitStringPiece(tty_drivers, "\n", base::KEEP_WHITESPACE,
                              base::SPLIT_WANT_NONEMPTY)) {
    std::vector<base::StringPiece> fields = base::SplitStringPiece(
        line, " ", base::TRIM_WHITESPACE, base::SPLIT_WANT_NONEMPTY);

    // The format of each line is:
    //
    //   driver name<SPACE>name<SPACE>major<SPACE>minor range<SPACE>type
    //
    // We only care about drivers that provide the "serial" type. The rest are
    // things like pseudoterminals.
    if (fields.size() < 5 || fields[4] != "serial")
      continue;

    SerialDriverInfo info;
    if (!base::StringToInt(fields[2], &info.major))
      continue;

    std::vector<base::StringPiece> minor_range = base::SplitStringPiece(
        fields[3], "-", base::KEEP_WHITESPACE, base::SPLIT_WANT_NONEMPTY);
    if (minor_range.size() == 1) {
      if (!base::StringToInt(minor_range[0], &info.minor_start))
        continue;
      info.minor_end = info.minor_start;
    } else if (minor_range.size() == 2) {
      if (!base::StringToInt(minor_range[0], &info.minor_start) ||
          !base::StringToInt(minor_range[1], &info.minor_end)) {
        continue;
      }
    } else {
      continue;
    }

    serial_drivers.push_back(info);
  }

  return serial_drivers;
}

}  // namespace

SerialDeviceEnumeratorLinux::SerialDeviceEnumeratorLinux() {
  DETACH_FROM_SEQUENCE(sequence_checker_);

  watcher_ = UdevWatcher::StartWatching(
      this, {UdevWatcher::Filter(kSubsystemTty, "")});
  watcher_->EnumerateExistingDevices();
}

SerialDeviceEnumeratorLinux::~SerialDeviceEnumeratorLinux() {
  DCHECK_CALLED_ON_VALID_SEQUENCE(sequence_checker_);
}

void SerialDeviceEnumeratorLinux::OnDeviceAdded(ScopedUdevDevicePtr device) {
  DCHECK_CALLED_ON_VALID_SEQUENCE(sequence_checker_);
  base::ScopedBlockingCall scoped_blocking_call(FROM_HERE,
                                                base::BlockingType::MAY_BLOCK);

#if DCHECK_IS_ON()
  const char* subsystem = udev_device_get_subsystem(device.get());
  DCHECK(subsystem);
  DCHECK_EQ(base::StringPiece(subsystem), kSubsystemTty);
#endif

  const char* syspath_str = udev_device_get_syspath(device.get());
  if (!syspath_str)
    return;
  std::string syspath(syspath_str);

  const char* major_str = udev_device_get_property_value(device.get(), "MAJOR");
  const char* minor_str = udev_device_get_property_value(device.get(), "MINOR");

  int major, minor;
  if (!major_str || !minor_str || !base::StringToInt(major_str, &major) ||
      !base::StringToInt(minor_str, &minor)) {
    return;
  }

  for (const auto& driver : ReadSerialDriverInfo()) {
    if (major == driver.major && minor >= driver.minor_start &&
        minor <= driver.minor_end) {
      CreatePort(std::move(device), syspath);
      return;
    }
  }
}

void SerialDeviceEnumeratorLinux::OnDeviceChanged(ScopedUdevDevicePtr device) {}

void SerialDeviceEnumeratorLinux::OnDeviceRemoved(ScopedUdevDevicePtr device) {
  DCHECK_CALLED_ON_VALID_SEQUENCE(sequence_checker_);
  base::ScopedBlockingCall scoped_blocking_call(FROM_HERE,
                                                base::BlockingType::MAY_BLOCK);

  const char* syspath = udev_device_get_syspath(device.get());
  if (!syspath)
    return;

  auto it = paths_.find(syspath);
  if (it == paths_.end())
    return;
  base::UnguessableToken token = it->second;

  paths_.erase(it);
  RemovePort(token);
}

void SerialDeviceEnumeratorLinux::CreatePort(ScopedUdevDevicePtr device,
                                             const std::string& syspath) {
  const char* path = udev_device_get_property_value(device.get(), "DEVNAME");
  if (!path)
    return;

  auto token = base::UnguessableToken::Create();
  auto info = mojom::SerialPortInfo::New();
  info->path = base::FilePath(path);
  info->token = token;

  const char* vendor_id =
      udev_device_get_property_value(device.get(), "ID_VENDOR_ID");
  const char* product_id =
      udev_device_get_property_value(device.get(), "ID_PRODUCT_ID");
  const char* product_name_enc =
      udev_device_get_property_value(device.get(), "ID_MODEL_ENC");
  const char* serial_number =
      udev_device_get_property_value(device.get(), "ID_SERIAL_SHORT");

  uint32_t int_value;
  if (vendor_id && base::HexStringToUInt(vendor_id, &int_value)) {
    info->vendor_id = int_value;
    info->has_vendor_id = true;
  }
  if (product_id && base::HexStringToUInt(product_id, &int_value)) {
    info->product_id = int_value;
    info->has_product_id = true;
  }
  if (product_name_enc)
    info->display_name = device::UdevDecodeString(product_name_enc);

  if (info->has_vendor_id && info->has_product_id && serial_number) {
    info->persistent_id = base::StringPrintf("%04X-%04X-%s", info->vendor_id,
                                             info->product_id, serial_number);
  }

  paths_.insert(std::make_pair(syspath, token));
  AddPort(std::move(info));
}

}  // namespace device
