# Copyright 2020 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

load("//lib/try.star", "INFRA_CONFIG_LOCATION_REGEXP")

_MD_HEADER = """\
<!-- Auto-generated by lucicfg (via cq-builders-md.star). -->
<!-- Do not modify manually. -->

# List of CQ builders

[TOC]

Changes that only modify files in //infra/config, which do not change any build
outputs, only require the chromium_presubmit builder to pass before landing.

Each builder name links to that builder on Milo. The "matching builders" links
point to the file used to determine which configurations a builder should copy
when running. These links might 404 or error; they are hard-coded right now,
using common assumptions about how builders are configured.
"""

_REQUIRED_HEADER = """\
These builders must pass before a CL may land that affects files outside of
//infra/config.
"""

_OPTIONAL_HEADER = """\
These builders optionally run, depending on the files in a CL. For example, a CL
which touches `//gpu/BUILD.gn` would trigger the builder
`android_optional_gpu_tests_rel`, due to the `location_regexp` values for that
builder.
"""

_EXPERIMENTAL_HEADER = """\
These builders are run on some percentage of builds. Their results are ignored
by CQ. These are often used to test new configurations before they are added
as required builders.
"""

_TRY_BUILDER_VIEW_URL = "https://ci.chromium.org/p/chromium/builders/try"

_REGEX_PREFIX = ".+/[+]/"

def _get_main_config_group_builders(ctx):
    cq_cfg = ctx.output["commit-queue.cfg"]

    for c in cq_cfg.config_groups:
        if len(c.gerrit) != 1:
            continue

        gerrit = c.gerrit[0]
        if len(gerrit.projects) != 1:
            continue

        project = gerrit.projects[0]
        if len(project.ref_regexp) != 1:
            continue

        if (project.name == "chromium/src" and
            # Repeated proto fields have an internal type that won't compare equal
            # to a list, so convert it
            list(project.ref_regexp) == ["refs/heads/.+"]):
            return c.verifiers.tryjob.builders

    fail("Could not find the main CQ group")

def _normalize_builder(builder):
    location_regexp_exclude = [
        r
        for r in builder.location_regexp_exclude
        if r != INFRA_CONFIG_LOCATION_REGEXP
    ]
    location_regexp = builder.location_regexp
    if list(location_regexp) == [".*"] and not location_regexp_exclude:
        location_regexp = None
    return struct(
        name = builder.name,
        experiment_percentage = builder.experiment_percentage,
        includable_only = builder.includable_only,
        location_regexp = location_regexp,
        location_regexp_exclude = location_regexp_exclude,
    )

def _group_builders_by_section(builders):
    required = []
    experimental = []
    optional = []

    for builder in builders:
        builder = _normalize_builder(builder)
        if builder.experiment_percentage:
            experimental.append(builder)
        elif builder.location_regexp or builder.location_regexp_exclude:
            optional.append(builder)
        elif builder.includable_only:
            continue
        else:
            required.append(builder)

    return struct(
        required = required,
        experimental = experimental,
        optional = optional,
    )

def _codesearch_query(*atoms, package = None):
    query = ["https://cs.chromium.org/search?q="]
    if package != None:
        query.append("package:%5E")  # %5E -> encoded ^
        query.append(package)
        query.append("$")
    for atom in atoms:
        query.append("+")
        query.append(atom)
    return "".join(query)

def _get_regex_line_details(regex):
    if regex.startswith(_REGEX_PREFIX):
        regex = regex[len(_REGEX_PREFIX):]
    title = "//" + regex.lstrip("/")
    if regex.endswith(".+"):
        regex = regex[:-len(".+")]

    url = _codesearch_query("file:" + regex, package = "chromium")

    # If the regex doesn't have any interesting characters that might be part of a
    # regex, assume the regex is targeting a single path and direct link to it
    # Equals sign and dashes used by layout tests
    if all([c.isalnum() or c in "/-_=" for c in regex.codepoints()]):
        url = "https://cs.chromium.org/chromium/src/" + regex

    return struct(
        title = title,
        url = url,
    )

def _generate_cq_builders_md(ctx):
    builders = _get_main_config_group_builders(ctx)

    builders_by_section = _group_builders_by_section(builders)

    lines = [_MD_HEADER]

    for title, header, section in (
        ("Required builders", _REQUIRED_HEADER, "required"),
        ("Optional builders", _OPTIONAL_HEADER, "optional"),
        ("Experimental builders", _EXPERIMENTAL_HEADER, "experimental"),
    ):
        builders = getattr(builders_by_section, section)
        if not builders:
            continue

        lines.append("## %s" % title)
        lines.append(header)

        for b in builders:
            name = b.name.rsplit("/", 1)[-1]
            lines.append((
                "* [{name}]({try_builder_view}/{name}) " +
                "([definition]({definition_query}+{name})) " +
                "([matching builders]({trybot_query}+{name}))"
            ).format(
                name = name,
                try_builder_view = _TRY_BUILDER_VIEW_URL,
                definition_query = _codesearch_query(
                    "file:/cq.star$",
                    "-file:/beta/",
                    "-file:/stable/",
                    package = "chromium",
                ),
                trybot_query = _codesearch_query("file:trybots.py"),
            ))

            if b.experiment_percentage:
                lines.append("  * Experiment percentage: {percentage}".format(
                    percentage = b.experiment_percentage,
                ))

            for attr, regexp_header in (
                ("location_regexp", "Path regular expressions:"),
                ("location_regexp_exclude", "Path exclude regular expressions:"),
            ):
                regexes = getattr(b, attr)
                if not regexes:
                    continue
                lines.append("")
                lines.append("  " + regexp_header)
                for regex in regexes:
                    regex_line_details = _get_regex_line_details(regex)
                    lines.append("  * [`{title}`]({url})".format(
                        title = regex_line_details.title,
                        url = regex_line_details.url,
                    ))

            lines.append("")

        lines.append("")

    ctx.output["cq-builders.md"] = "\n".join(lines)

lucicfg.generator(_generate_cq_builders_md)
