// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.weblayer_private.media;

import android.app.Service;
import android.content.Intent;
import android.support.v4.media.session.MediaSessionCompat;

import org.chromium.components.browser_ui.media.MediaNotificationController;
import org.chromium.components.browser_ui.media.MediaNotificationInfo;
import org.chromium.components.browser_ui.media.MediaNotificationManager;
import org.chromium.components.browser_ui.media.MediaSessionHelper;
import org.chromium.components.browser_ui.notifications.ForegroundServiceUtils;
import org.chromium.components.browser_ui.notifications.NotificationMetadata;
import org.chromium.components.browser_ui.notifications.NotificationWrapper;
import org.chromium.components.browser_ui.notifications.NotificationWrapperBuilder;
import org.chromium.weblayer_private.IntentUtils;
import org.chromium.weblayer_private.WebLayerImpl;
import org.chromium.weblayer_private.WebLayerNotificationChannels;
import org.chromium.weblayer_private.WebLayerNotificationWrapperBuilder;

/**
 * A glue class for MediaSession.
 * This class defines delegates that provide WebLayer-specific behavior to shared MediaSession code.
 * It also manages the lifetime of {@link MediaNotificationController} and the {@link Service}
 * associated with the notification.
 */
public class MediaSessionManager {
    private static int sNotificationId;

    public static void serviceStarted(Service service, Intent intent) {
        MediaNotificationController controller = getController();
        if (controller != null && controller.processIntent(service, intent)) return;

        // The service has been started with startForegroundService() but the
        // notification hasn't been shown. See similar logic in {@link
        // ChromeMediaNotificationControllerDelegate}.
        MediaNotificationController.finishStartingForegroundServiceOnO(
                service, createNotificationWrapperBuilder().buildNotificationWrapper());
        // Call stopForeground to guarantee Android unset the foreground bit.
        ForegroundServiceUtils.getInstance().stopForeground(
                service, Service.STOP_FOREGROUND_REMOVE);
        service.stopSelf();
    }

    public static void serviceDestroyed() {
        MediaNotificationController controller = getController();
        if (controller != null) controller.onServiceDestroyed();
        MediaNotificationManager.clear(getNotificationId());
    }

    public static MediaSessionHelper.Delegate createMediaSessionHelperDelegate(int tabId) {
        return new MediaSessionHelper.Delegate() {
            @Override
            public Intent createBringTabToFrontIntent() {
                return IntentUtils.createBringTabToFrontIntent(tabId);
            }

            @Override
            public boolean fetchLargeFaviconImage() {
                // TODO(crbug.com/1076463): WebLayer doesn't support favicons.
                return false;
            }

            @Override
            public MediaNotificationInfo.Builder createMediaNotificationInfoBuilder() {
                return new MediaNotificationInfo.Builder().setInstanceId(tabId).setId(
                        getNotificationId());
            }

            @Override
            public void showMediaNotification(MediaNotificationInfo notificationInfo) {
                assert notificationInfo.id == getNotificationId();
                MediaNotificationManager.show(notificationInfo,
                        () -> { return new WebLayerMediaNotificationControllerDelegate(); });
            }

            @Override
            public void hideMediaNotification() {
                MediaNotificationManager.hide(tabId, getNotificationId());
            }

            @Override
            public void activateAndroidMediaSession() {
                MediaNotificationManager.activateAndroidMediaSession(tabId, getNotificationId());
            }
        };
    }

    private static class WebLayerMediaNotificationControllerDelegate
            implements MediaNotificationController.Delegate {
        @Override
        public Intent createServiceIntent() {
            return WebLayerImpl.createMediaSessionServiceIntent();
        }

        @Override
        public String getAppName() {
            return WebLayerImpl.getClientApplicationName();
        }

        @Override
        public String getNotificationGroupName() {
            return "org.chromium.weblayer.MediaSession";
        }

        @Override
        public NotificationWrapperBuilder createNotificationWrapperBuilder() {
            return MediaSessionManager.createNotificationWrapperBuilder();
        }

        @Override
        public void onMediaSessionUpdated(MediaSessionCompat session) {
            // This is only relevant when casting.
        }

        @Override
        public void logNotificationShown(NotificationWrapper notification) {}
    }

    private static NotificationWrapperBuilder createNotificationWrapperBuilder() {
        // Only the null tag will work as expected, because {@link Service#startForeground()} only
        // takes an ID and no tag. If we pass a tag here, then the notification that's used to
        // display a paused state (no foreground service) will not be identified as the same one
        // that's used with the foreground service.
        return WebLayerNotificationWrapperBuilder.create(
                WebLayerNotificationChannels.ChannelId.MEDIA_PLAYBACK,
                new NotificationMetadata(0, null /*notificationTag*/, getNotificationId()));
    }

    private static int getNotificationId() {
        if (sNotificationId == 0) sNotificationId = WebLayerImpl.getMediaSessionNotificationId();
        return sNotificationId;
    }

    private static MediaNotificationController getController() {
        return MediaNotificationManager.getController(getNotificationId());
    }
}
