#!/usr/bin/env python
# Copyright (c) 2012 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Generate a C++ header from input_methods.txt.

This program generates a C++ header file containing the information on
available input methods.  It parses input_methods.txt, and then generates a
static array definition from the information extracted. The input and output
file names are specified on the command line.
The header input_methods.h is used in input_method_allowlist.cc which is for
testing purpose.

Run it like:
  gen_input_methods.py input_methods.txt input_methods.h

It will produce output that looks like:

// This file is automatically generated by gen_input_methods.py
#ifndef CHROMEOS_IME_INPUT_METHODS_H_
#define CHROMEOS_IME_INPUT_METHODS_H_

namespace chromeos {
namespace input_method {

struct InputMethodsInfo {
  const char* input_method_id;
  const char* language_code;
  const char* xkb_keyboard_id;
  const char* indicator;
  bool is_login_keyboard;
};
const InputMethodsInfo kInputMethods[] = {
  {"xkb:us::eng", "en-US", "us", "US", true},
  {"xkb:us:dvorak:eng", "en-US", "us(dvorak)", "DV", true},
  {"xkb:be::fra", "fr", "be", "BE", true},
  {"xkb:br::por", "pt-BR", "br", "BR", true},
  {"xkb:ru::rus", "ru", "ru", "RU", false},
};

}  // namespace input_method
}  // namespace chromeos

#endif  // CHROMEOS_IME_INPUT_METHODS_H_

"""

import fileinput
import re
import sys

OUTPUT_HEADER = """// Automatically generated by gen_input_methods.py
#ifndef CHROMEOS_IME_INPUT_METHODS_H_
#define CHROMEOS_IME_INPUT_METHODS_H_

namespace chromeos {
namespace input_method {

struct InputMethodsInfo {
  const char* input_method_id;
  const char* language_code;
  const char* xkb_layout_id;
  const char* indicator;
  bool is_login_keyboard;
};
const InputMethodsInfo kInputMethods[] = {
"""

CPP_FORMAT = '#if %s\n'
ENGINE_FORMAT = ('  {"%(input_method_id)s", "%(language_code)s", ' +
                 '"%(xkb_layout_id)s", "%(indicator)s", ' +
                 '%(is_login_keyboard)s},\n')

OUTPUT_FOOTER = """
};

}  // namespace input_method
}  // namespace chromeos

#endif  // CHROMEOS_IME_INPUT_METHODS_H_
"""

def CreateEngineHeader(engines):
  """Create the header file from a list of engines.

  Arguments:
    engines: list of engine objects
  Returns:
    The text of a C++ header file containing the engine data.
  """
  output = []
  output.append(OUTPUT_HEADER)
  for engine in engines:
    if 'if' in engine:
      output.append(CPP_FORMAT % engine['if'])
    output.append(ENGINE_FORMAT % engine)
    if 'if' in engine:
      output.append('#endif\n')
  output.append(OUTPUT_FOOTER)

  return "".join(output)


def main(argv):
  if len(argv) != 3:
    print 'Usage: gen_input_methods.py [allowlist] [output]'
    sys.exit(1)
  engines = []
  for line in fileinput.input(sys.argv[1]):
    line = line.strip()
    if not line or re.match(r'#', line):
      continue
    columns = line.split()
    assert len(columns) == 4 or len(columns) == 5, "Invalid format: " + line
    engine = {}
    engine['input_method_id'] = columns[0]
    engine['xkb_layout_id'] = columns[1]
    engine['language_code'] = columns[2]
    engine['indicator'] = columns[3]
    is_login_keyboard = "false"
    if len(columns) == 5:
      assert columns[4] == "login", "Invalid attribute: " + columns[4]
      is_login_keyboard = "true"
    engine['is_login_keyboard'] = is_login_keyboard
    engines.append(engine)

  output = CreateEngineHeader(engines)
  output_file = open(sys.argv[2], 'w')
  output_file.write(output)


if __name__ == '__main__':
  main(sys.argv)
