// Copyright (c) 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROMEOS_DBUS_HERMES_HERMES_EUICC_CLIENT_H_
#define CHROMEOS_DBUS_HERMES_HERMES_EUICC_CLIENT_H_

#include <string>
#include <vector>

#include "base/callback_forward.h"
#include "base/macros.h"
#include "base/observer_list.h"
#include "chromeos/dbus/dbus_method_call_status.h"
#include "chromeos/dbus/hermes/hermes_response_status.h"
#include "dbus/property.h"
#include "third_party/cros_system_api/dbus/hermes/dbus-constants.h"

namespace chromeos {

// HermesEuiccClient is used to talk to the Hermes Euicc objects.
class COMPONENT_EXPORT(HERMES_CLIENT) HermesEuiccClient {
 public:
  // Callback for profile installation methods. Callback returns status code
  // and the object path for the profile that was just successfully installed.
  using InstallCarrierProfileCallback =
      base::OnceCallback<void(HermesResponseStatus status,
                              const dbus::ObjectPath* carrier_profile_path)>;

  class TestInterface {
   public:
    // Clears a given Euicc and associated profiles.
    virtual void ClearEuicc(const dbus::ObjectPath& euicc_path) = 0;

    // Resets the test interface for a new fake pending profile
    // will be added on subsequent  call to RequestPendingEvents.
    virtual void ResetPendingEventsRequested() = 0;

    // Adds a new carrier profile under given euicc object using fake default
    // values for properties. If |state| is not pending then a corresponding
    // fake cellular service is also created in shill. If |service_only| is true
    // then the service will be created but the profile will not be listed in
    // the euicc until a subsequent call to RequestInstalledProfiles. Returns
    // the path to the newly added profile.
    virtual dbus::ObjectPath AddFakeCarrierProfile(
        const dbus::ObjectPath& euicc_path,
        hermes::profile::State state,
        const std::string& activation_code,
        bool service_only) = 0;

    // Adds a new carrier profile with given path and properties.
    virtual void AddCarrierProfile(const dbus::ObjectPath& path,
                                   const dbus::ObjectPath& euicc_path,
                                   const std::string& iccid,
                                   const std::string& name,
                                   const std::string& service_provider,
                                   const std::string& activation_code,
                                   const std::string& network_service_path,
                                   hermes::profile::State state,
                                   bool service_only) = 0;

    // Queues an error code that will be returned from a subsequent
    // method call.
    virtual void QueueHermesErrorStatus(HermesResponseStatus status) = 0;

    // Set delay for interactive methods.
    virtual void SetInteractiveDelay(base::TimeDelta delay) = 0;

    // Returns a valid fake activation code that can be used to install
    // a new fake carrier profile.
    virtual std::string GenerateFakeActivationCode() = 0;
  };

  // Hermes Euicc properties.
  class Properties : public dbus::PropertySet {
   public:
    Properties(dbus::ObjectProxy* object_proxy,
               const PropertyChangedCallback& callback);
    ~Properties() override;

    dbus::Property<std::string>& eid() { return eid_; }
    dbus::Property<bool>& is_active() { return is_active_; }
    dbus::Property<std::vector<dbus::ObjectPath>>&
    installed_carrier_profiles() {
      return installed_carrier_profiles_;
    }
    dbus::Property<std::vector<dbus::ObjectPath>>& pending_carrier_profiles() {
      return pending_carrier_profiles_;
    }
    dbus::Property<int32_t>& physical_slot() { return physical_slot_; }

   private:
    // EID of the Euicc.
    dbus::Property<std::string> eid_;

    // Boolean that indicates whether this euicc is currently active.
    dbus::Property<bool> is_active_;

    // List of paths to carrier profiles currently installed on the device.
    dbus::Property<std::vector<dbus::ObjectPath>> installed_carrier_profiles_;

    // List of pending carrier profiles from SMDS available for
    // installation on this device.
    dbus::Property<std::vector<dbus::ObjectPath>> pending_carrier_profiles_;

    // Physical slot number of the Euicc.
    dbus::Property<int32_t> physical_slot_;
  };

  // Interface for observing Hermes Euicc changes.
  class Observer {
   public:
    virtual ~Observer() = default;

    // Called when an euicc property changes.
    virtual void OnEuiccPropertyChanged(const dbus::ObjectPath& euicc_path,
                                        const std::string& property_name) = 0;
  };

  // Adds an observer for carrier profile lists changes on Hermes manager.
  virtual void AddObserver(Observer* observer);

  // Removes an observer for Hermes manager.
  virtual void RemoveObserver(Observer* observer);

  // Install a carrier profile in the Euicc at |euicc_path| with given
  // |activation_code| and |conirmation_code|. |confirmation_code| can be empty
  // if no confirmation is required by carrier. Returns the object path to the
  // carrier profile that was just installed.
  virtual void InstallProfileFromActivationCode(
      const dbus::ObjectPath& euicc_path,
      const std::string& activation_code,
      const std::string& confirmation_code,
      InstallCarrierProfileCallback callback) = 0;

  // Installs a pending profile with given |carrier_profile_path| in the Euicc
  // at |euicc_path|. |confirmation_code| can be empty if no confirmation is
  // required by carrier. Returns a response status indicating the install
  // result.
  virtual void InstallPendingProfile(
      const dbus::ObjectPath& euicc_path,
      const dbus::ObjectPath& carrier_profile_path,
      const std::string& confirmation_code,
      HermesResponseCallback callback) = 0;

  // Updates installed profiles for Euicc at |euicc_path|.
  // This updates installed profiles list prior to returning.
  virtual void RequestInstalledProfiles(const dbus::ObjectPath& euicc_path,
                                        HermesResponseCallback callback) = 0;

  // Updates pending profiles for Euicc at |euicc_path| from the SMDS server
  // using the given |root_smds| server address. Passing an empty |root_smds|
  // will use default lpa.ds.gsma.com. This updates pending profiles list prior
  // to returning.
  virtual void RequestPendingProfiles(const dbus::ObjectPath& euicc_path,
                                      const std::string& root_smds,
                                      HermesResponseCallback callback) = 0;

  // Removes the carrier profile with the given |carrier_profile_path| from
  // the Euicc at |euicc_path|. Returns a response status indicating the result
  // of the operation.
  virtual void UninstallProfile(const dbus::ObjectPath& euicc_path,
                                const dbus::ObjectPath& carrier_profile_path,
                                HermesResponseCallback callback) = 0;

  // Returns properties for the Euicc with given |euicc_path|.
  virtual Properties* GetProperties(const dbus::ObjectPath& euicc_path) = 0;

  // Returns an instance of Hermes Euicc Test interface.
  virtual TestInterface* GetTestInterface() = 0;

  // Creates and initializes the global instance.
  static void Initialize(dbus::Bus* bus);

  // Creates and initializes a global fake instance.
  static void InitializeFake();

  // Destroys the global instance.
  static void Shutdown();

  // Returns the global instance.
  static HermesEuiccClient* Get();

 protected:
  HermesEuiccClient();
  virtual ~HermesEuiccClient();

  const base::ObserverList<Observer>::Unchecked& observers() {
    return observers_;
  }

 private:
  base::ObserverList<Observer>::Unchecked observers_;
};

}  // namespace chromeos

#endif  // CHROMEOS_DBUS_HERMES_HERMES_EUICC_CLIENT_H_
