// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import {assert} from '../chrome_util.js';
import * as loadTimeData from '../models/load_time_data.js';
import {DeviceOperator} from '../mojo/device_operator.js';
import * as toast from '../toast.js';
// eslint-disable-next-line no-unused-vars
import {ResolutionList, VideoConfig} from '../type.js';

import {Camera3DeviceInfo} from './camera3_device_info.js';
import {
  PhotoConstraintsPreferrer,  // eslint-disable-line no-unused-vars
  VideoConstraintsPreferrer,  // eslint-disable-line no-unused-vars
} from './constraints_preferrer.js';

/**
 * Thrown for no camera available on the device.
 */
export class NoCameraError extends Error {
  /**
   * @param {string=} message
   * @public
   */
  constructor(message = 'No camera available on the device') {
    super(message);
    this.name = this.constructor.name;
  }
}

/**
 * Contains information of all cameras on the device and will updates its value
 * when any plugin/unplug external camera changes.
 */
export class DeviceInfoUpdater {
  /**
   * @param {!PhotoConstraintsPreferrer} photoPreferrer
   * @param {!VideoConstraintsPreferrer} videoPreferrer
   * @public
   * */
  constructor(photoPreferrer, videoPreferrer) {
    /**
     * @type {!PhotoConstraintsPreferrer}
     * @private
     */
    this.photoPreferrer_ = photoPreferrer;

    /**
     * @type {!VideoConstraintsPreferrer}
     * @private
     */
    this.videoPreferrer_ = videoPreferrer;

    /**
     * Listeners to be called after new camera information is available.
     * @type {!Array<function(!DeviceInfoUpdater): !Promise>}
     * @private
     */
    this.deviceChangeListeners_ = [];

    /**
     * Action locking update of camera information.
     * @type {?Promise}
     * @private
     */
    this.lockingUpdate_ = null;

    /**
     * Pending camera information update while update capability is locked.
     * @type {?Promise}
     * @private
     */
    this.pendingUpdate_ = null;

    /**
     * MediaDeviceInfo of all available video devices.
     * @type {!Promise<!Array<!MediaDeviceInfo>>}
     * @private
     */
    this.devicesInfo_ = this.enumerateDevices_();

    /**
     * Camera3DeviceInfo of all available video devices. Is null on fake cameras
     * which do not have private mojo API support.
     * @type {!Promise<?Array<!Camera3DeviceInfo>>}
     * @private
     */
    this.camera3DevicesInfo_ = this.queryMojoDevicesInfo_();

    /**
     * Filter out lagging 720p on grunt. See https://crbug.com/1122852.
     * @const {!Promise<function(!VideoConfig): boolean>}
     * @private
     */
    this.videoConfigFilter_ = (async () => {
      const board = loadTimeData.getBoard();
      return board === 'grunt' ? ({height}) => height < 720 : () => true;
    })();

    /**
     * Promise of first update.
     * @type {!Promise}
     */
    this.firstUpdate_ = this.update_();

    navigator.mediaDevices.addEventListener(
        'devicechange', this.update_.bind(this));

    this.initCameraChangeAnnouncer_();
  }

  /**
   * Initializes announcer announcing all camera change events.
   * TODO(b/151047537): Moves it into StreamManager whose update won't be
   * blocked by update lock.
   */
  async initCameraChangeAnnouncer_() {
    const queryCameras = async () => {
      try {
        return await this.enumerateDevices_();
      } catch (e) {
        assert(e instanceof NoCameraError);
        return [];
      }
    };

    /**
     * Computes |devices| - |devices2|.
     * @param {!Array<!MediaDeviceInfo>} devices
     * @param {!Array<!MediaDeviceInfo>} devices2
     * @return {!Array<!MediaDeviceInfo>}
     */
    const getDifference = (devices, devices2) => {
      const ids = new Set(devices2.map(({deviceId}) => deviceId));
      return devices.filter(({deviceId}) => !ids.has(deviceId));
    };

    let announcedDevices = await queryCameras();
    navigator.mediaDevices.addEventListener('devicechange', async () => {
      const devices = await queryCameras();
      for (const added of getDifference(devices, announcedDevices)) {
        toast.speak('status_msg_camera_plugged', added.label);
      }
      for (const removed of getDifference(announcedDevices, devices)) {
        toast.speak('status_msg_camera_unplugged', removed.label);
      }
      announcedDevices = devices;
    });
  }

  /**
   * Tries to gain lock and initiates update process.
   * @private
   */
  async update_() {
    if (this.lockingUpdate_) {
      if (this.pendingUpdate_) {
        return;
      }
      this.pendingUpdate_ = (async () => {
        while (this.lockingUpdate_) {
          try {
            await this.lockingUpdate_;
          } catch (e) {
            // Ignore exception from waiting for existing update.
          }
        }
        this.lockingUpdate_ = this.pendingUpdate_;
        this.pendingUpdate_ = null;
        await this.doUpdate_();
        this.lockingUpdate_ = null;
      })();
    } else {
      this.lockingUpdate_ = (async () => {
        await this.doUpdate_();
        this.lockingUpdate_ = null;
      })();
    }
  }

  /**
   * Updates devices information.
   * @private
   */
  async doUpdate_() {
    this.devicesInfo_ = this.enumerateDevices_();
    this.camera3DevicesInfo_ = this.queryMojoDevicesInfo_();
    try {
      await this.devicesInfo_;
      const devices = await this.camera3DevicesInfo_;
      if (devices) {
        this.photoPreferrer_.updateDevicesInfo(devices);
        this.videoPreferrer_.updateDevicesInfo(devices);
      }
      await Promise.all(this.deviceChangeListeners_.map((l) => l(this)));
    } catch (e) {
      console.error(e);
    }
  }

  /**
   * Enumerates all available devices and gets their MediaDeviceInfo.
   * @return {!Promise<!Array<!MediaDeviceInfo>>}
   * @throws {!Error}
   * @private
   */
  async enumerateDevices_() {
    const devices = (await navigator.mediaDevices.enumerateDevices())
                        .filter((device) => device.kind === 'videoinput');
    if (devices.length === 0) {
      throw new NoCameraError();
    }
    return devices;
  }

  /**
   * Queries Camera3DeviceInfo of available devices through private mojo API.
   * @return {!Promise<?Array<!Camera3DeviceInfo>>} Camera3DeviceInfo
   *     of available devices. Maybe null on fake cameras.
   * @throws {!Error} Thrown when camera unplugging happens between enumerating
   *     devices and querying mojo APIs with current device info results.
   * @private
   */
  async queryMojoDevicesInfo_() {
    if (!await DeviceOperator.isSupported()) {
      return null;
    }
    const deviceInfos = await this.devicesInfo_;
    const videoConfigFilter = await this.videoConfigFilter_;
    return Promise.all(
        deviceInfos.map((d) => Camera3DeviceInfo.create(d, videoConfigFilter)));
  }

  /**
   * Registers listener to be called when state of available devices changes.
   * @param {function(!DeviceInfoUpdater): !Promise} listener
   */
  addDeviceChangeListener(listener) {
    this.deviceChangeListeners_.push(listener);
  }

  /**
   * Requests to lock update of device information. This function is preserved
   * for device information reader to lock the update capability so as to ensure
   * getting consistent data between all information providers.
   * @param {function(): !Promise} callback Called after
   *     update capability is locked. Getting information from all providers in
   *     callback are guaranteed to be consistent.
   */
  async lockDeviceInfo(callback) {
    await this.firstUpdate_;
    while (this.lockingUpdate_ || this.pendingUpdate_) {
      try {
        await this.lockingUpdate_;
        await this.pendingUpdate_;
      } catch (e) {
        // Ignore exception from waiting for existing update.
      }
    }
    this.lockingUpdate_ = (async () => {
      try {
        await callback();
      } finally {
        this.lockingUpdate_ = null;
      }
    })();
    await this.lockingUpdate_;
  }

  /**
   * Gets MediaDeviceInfo for all available video devices.
   * @return {!Promise<!Array<!MediaDeviceInfo>>}
   */
  async getDevicesInfo() {
    return this.devicesInfo_;
  }

  /**
   * Gets MediaDeviceInfo of specific video device.
   * @param {string} deviceId Device id of video device to get information from.
   * @return {!Promise<?MediaDeviceInfo>}
   */
  async getDeviceInfo(deviceId) {
    const /** !Array<!MediaDeviceInfo> */ infos = await this.getDevicesInfo();
    return infos.find((d) => d.deviceId === deviceId) || null;
  }

  /**
   * Gets Camera3DeviceInfo for all available video devices.
   * @return {!Promise<?Array<!Camera3DeviceInfo>>}
   */
  async getCamera3DevicesInfo() {
    return this.camera3DevicesInfo_;
  }
}
