// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chromeos/cryptohome/userdataauth_util.h"

#include "testing/gtest/include/gtest/gtest.h"

namespace user_data_auth {

namespace {

constexpr char kKeyLabel[] = "key_label";

constexpr int64_t kKeyRevision = 123;
constexpr char kProviderData1Name[] = "data_1";
constexpr int64_t kProviderData1Number = 12345;
constexpr char kProviderData2Name[] = "data_2";
constexpr char kProviderData2Bytes[] = "data_2 bytes";

}  // namespace

TEST(UserDataAuthUtilTest, ReplyToMountErrorNullOptional) {
  const base::Optional<RemoveReply> reply = base::nullopt;
  EXPECT_EQ(ReplyToMountError(reply), cryptohome::MOUNT_ERROR_FATAL);
}

TEST(UserDataAuthUtilTest, ReplyToMountErrorNoError) {
  RemoveReply result;
  result.set_error(CRYPTOHOME_ERROR_NOT_SET);
  const base::Optional<RemoveReply> reply = std::move(result);
  EXPECT_EQ(ReplyToMountError(reply), cryptohome::MOUNT_ERROR_NONE);
}

TEST(UserDataAuthUtilTest, BaseReplyToMountErrorAuthFailure) {
  RemoveReply result;
  result.set_error(CRYPTOHOME_ERROR_AUTHORIZATION_KEY_NOT_FOUND);
  const base::Optional<RemoveReply> reply = std::move(result);
  EXPECT_EQ(ReplyToMountError(reply), cryptohome::MOUNT_ERROR_KEY_FAILURE);
}

TEST(UserDataAuthUtilTest, CryptohomeErrorToMountError) {
  // Test a few commonly seen input output pair.
  EXPECT_EQ(CryptohomeErrorToMountError(CRYPTOHOME_ERROR_NOT_SET),
            cryptohome::MOUNT_ERROR_NONE);
  EXPECT_EQ(CryptohomeErrorToMountError(CRYPTOHOME_ERROR_MOUNT_FATAL),
            cryptohome::MOUNT_ERROR_FATAL);
  EXPECT_EQ(CryptohomeErrorToMountError(CRYPTOHOME_ERROR_TPM_DEFEND_LOCK),
            cryptohome::MOUNT_ERROR_TPM_DEFEND_LOCK);
  EXPECT_EQ(CryptohomeErrorToMountError(CRYPTOHOME_ERROR_TPM_NEEDS_REBOOT),
            cryptohome::MOUNT_ERROR_TPM_NEEDS_REBOOT);
}

TEST(UserDataAuthUtilTest, GetKeyDataReplyToKeyDefinitionsTwoEntries) {
  GetKeyDataReply result;
  result.set_error(CryptohomeErrorCode::CRYPTOHOME_ERROR_NOT_SET);
  cryptohome::KeyData* key_data = result.add_key_data();
  key_data->set_type(cryptohome::KeyData::KEY_TYPE_PASSWORD);
  key_data->set_label(kKeyLabel);
  key_data->mutable_privileges()->set_update(false);
  key_data->set_revision(kKeyRevision);
  cryptohome::KeyProviderData* data = key_data->mutable_provider_data();
  cryptohome::KeyProviderData::Entry* entry1 = data->add_entry();
  entry1->set_name(kProviderData1Name);
  entry1->set_number(kProviderData1Number);
  cryptohome::KeyProviderData::Entry* entry2 = data->add_entry();
  entry2->set_name(kProviderData2Name);
  entry2->set_bytes(kProviderData2Bytes);
  const base::Optional<GetKeyDataReply> reply = std::move(result);

  std::vector<cryptohome::KeyDefinition> key_definitions =
      GetKeyDataReplyToKeyDefinitions(reply);

  // Verify that the call was successful and the result was correctly parsed.
  ASSERT_EQ(1u, key_definitions.size());
  const cryptohome::KeyDefinition& key_definition = key_definitions.front();
  EXPECT_EQ(cryptohome::KeyDefinition::TYPE_PASSWORD, key_definition.type);
  EXPECT_EQ(kKeyLabel, key_definition.label);
  EXPECT_EQ(cryptohome::PRIV_ADD | cryptohome::PRIV_REMOVE,
            key_definition.privileges);
  EXPECT_EQ(kKeyRevision, key_definition.revision);
  ASSERT_EQ(2u, key_definition.provider_data.size());
  const cryptohome::KeyDefinition::ProviderData* provider_data =
      &key_definition.provider_data[0];
  EXPECT_EQ(kProviderData1Name, provider_data->name);
  ASSERT_TRUE(provider_data->number);
  EXPECT_EQ(kProviderData1Number, *provider_data->number.get());
  EXPECT_FALSE(provider_data->bytes);
  provider_data = &key_definition.provider_data[1];
  EXPECT_EQ(kProviderData2Name, provider_data->name);
  EXPECT_FALSE(provider_data->number);
  ASSERT_TRUE(provider_data->bytes);
  EXPECT_EQ(kProviderData2Bytes, *provider_data->bytes.get());
}

// Test the GetKeyDataReplyToKeyDefinitions() function against the
// GetKeyDataReply proto containing the KeyData proto of the
// |KEY_TYPE_CHALLENGE_RESPONSE| type.
TEST(UserDataAuthUtilTest, GetKeyDataReplyToKeyDefinitions_ChallengeResponse) {
  GetKeyDataReply result;
  result.set_error(CryptohomeErrorCode::CRYPTOHOME_ERROR_NOT_SET);
  cryptohome::KeyData* key_data = result.add_key_data();
  key_data->set_type(cryptohome::KeyData::KEY_TYPE_CHALLENGE_RESPONSE);
  key_data->set_label(kKeyLabel);
  const base::Optional<GetKeyDataReply> reply = std::move(result);

  const std::vector<cryptohome::KeyDefinition> key_definitions =
      GetKeyDataReplyToKeyDefinitions(reply);

  ASSERT_EQ(1u, key_definitions.size());
  const cryptohome::KeyDefinition& key_definition = key_definitions.front();
  EXPECT_EQ(cryptohome::KeyDefinition::TYPE_CHALLENGE_RESPONSE,
            key_definition.type);
  EXPECT_EQ(kKeyLabel, key_definition.label);
}

TEST(UserDataAuthUtilTest, AccountDiskUsageReplyToUsageSizeNullOptional) {
  const base::Optional<GetAccountDiskUsageReply> reply = base::nullopt;

  ASSERT_EQ(AccountDiskUsageReplyToUsageSize(reply), -1);
}

TEST(UserDataAuthUtilTest, AccountDiskUsageReplyToUsageSizeErrorInReply) {
  GetAccountDiskUsageReply result;
  result.set_error(CryptohomeErrorCode::CRYPTOHOME_ERROR_NOT_IMPLEMENTED);
  const base::Optional<GetAccountDiskUsageReply> reply = std::move(result);

  ASSERT_EQ(AccountDiskUsageReplyToUsageSize(reply), -1);
}

TEST(UserDataAuthUtilTest, AccountDiskUsageReplyToUsageSizeValidity) {
  constexpr int64_t kSize = 0x123456789ABCLL;
  GetAccountDiskUsageReply result;
  result.set_size(kSize);
  const base::Optional<GetAccountDiskUsageReply> reply = std::move(result);

  ASSERT_EQ(AccountDiskUsageReplyToUsageSize(reply), kSize);
}

}  // namespace user_data_auth
