// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROMEOS_NETWORK_CELLULAR_ESIM_UNINSTALL_HANDLER_H_
#define CHROMEOS_NETWORK_CELLULAR_ESIM_UNINSTALL_HANDLER_H_

#include <memory>

#include "base/component_export.h"
#include "base/containers/circular_deque.h"
#include "base/containers/queue.h"
#include "base/values.h"
#include "chromeos/dbus/hermes/hermes_response_status.h"
#include "chromeos/network/cellular_esim_profile_handler.h"
#include "chromeos/network/cellular_inhibitor.h"
#include "chromeos/network/network_state_handler.h"
#include "dbus/object_path.h"

namespace chromeos {

class CellularESimProfileHandler;
class CellularInhibitor;
class NetworkState;
class NetworkConfigurationHandler;
class NetworkConnectionHandler;

// Handles Uninstallation of an eSIM profile and it's corresponding network.
//
// Uninstalling and eSIM network involves interacting with both Shill and Hermes
// in the following sequence:
// 1. Disconnect Network with Shill
// 2. Inhibit Cellular Scans
// 3. Request Installed eSIM profiles from Hermes
// 4. Disable eSIM profile through Hermes
// 5. Uninstall eSIM profile in Hermes
// 6. Remove Shill configuration for Network
// 7. Uninhibit Cellular Scans
//
// Uninstallation requests are queued and run in order.
//
// This class also checks for stale Shill eSIM services (Shill services that no
// longer have corresponding eSIM profile in Hermes), and removes their
// configurations from Shill. This allows handling cases where the configuration
// was not removed properly during uninstallation or the eSIM profile was
// removed externally (e.g. Removable EUICC card).
class COMPONENT_EXPORT(CHROMEOS_NETWORK) CellularESimUninstallHandler
    : public CellularESimProfileHandler::Observer,
      public NetworkStateHandlerObserver {
 public:
  CellularESimUninstallHandler();
  CellularESimUninstallHandler(const CellularESimUninstallHandler&) = delete;
  CellularESimUninstallHandler& operator=(const CellularESimUninstallHandler&) =
      delete;
  ~CellularESimUninstallHandler() override;

  void Init(CellularInhibitor* cellular_inhibitor,
            CellularESimProfileHandler* cellular_esim_profile_handler,
            NetworkConfigurationHandler* network_configuration_handler,
            NetworkConnectionHandler* network_connection_handler,
            NetworkStateHandler* network_state_handler);

  // Callback that returns true or false to indicate the success or failure of
  // an uninstallation request.
  using UninstallRequestCallback = base::OnceCallback<void(bool success)>;

  // Uninstalls an ESim profile and network with given |iccid| and
  // |esim_profile_path| that is installed in Euicc with the given
  // |euicc_path|.
  void UninstallESim(const std::string& iccid,
                     const dbus::ObjectPath& esim_profile_path,
                     const dbus::ObjectPath& euicc_path,
                     UninstallRequestCallback callback);

  // CellularESimProfileHandler::Observer:
  void OnESimProfileListUpdated() override;

  // NetworkStateHandlerObserver:
  void NetworkListChanged() override;
  void DevicePropertiesUpdated(const DeviceState* device_state) override;

 private:
  enum class UninstallState {
    kIdle,
    kDisconnectingNetwork,
    kInhibitingShill,
    kRequestingInstalledProfiles,
    kDisablingProfile,
    kUninstallingProfile,
    kRemovingShillService,
    kSuccess,
    kFailure,
  };
  friend std::ostream& operator<<(std::ostream& stream,
                                  const UninstallState& step);

  // Represents ESim uninstallation request parameters. Requests are queued and
  // processed one at a time. |esim_profile_path| and |euicc_path| are nullopt
  // for stale eSIM service removal requests. These requests skip directly to
  // Shill configuration removal.
  struct UninstallRequest {
    UninstallRequest(const std::string& iccid,
                     const base::Optional<dbus::ObjectPath>& esim_profile_path,
                     const base::Optional<dbus::ObjectPath>& euicc_path,
                     UninstallRequestCallback callback);
    ~UninstallRequest();
    std::string iccid;
    base::Optional<dbus::ObjectPath> esim_profile_path;
    base::Optional<dbus::ObjectPath> euicc_path;
    UninstallRequestCallback callback;
    std::unique_ptr<CellularInhibitor::InhibitLock> inhibit_lock;
  };

  void ProcessUninstallRequest();
  void TransitionToUninstallState(UninstallState next_state);
  void AttemptNetworkDisconnectIfRequired();
  void AttemptShillInhibit();
  void OnShillInhibit(
      std::unique_ptr<CellularInhibitor::InhibitLock> inhibit_lock);
  void AttemptRequestInstalledProfiles();
  void OnRefreshProfileListResult(
      std::unique_ptr<CellularInhibitor::InhibitLock> inhibit_lock);
  void AttemptDisableProfileIfRequired();
  void AttemptUninstallProfile();
  void AttemptRemoveShillService();
  void TransitionUninstallStateOnHermesSuccess(UninstallState next_state,
                                               HermesResponseStatus status);
  void TransitionUninstallStateOnSuccessBoolean(UninstallState next_state,
                                                bool success);
  void OnNetworkHandlerError(const std::string& error_name,
                             std::unique_ptr<base::DictionaryValue> error_data);
  const NetworkState* GetNetworkStateForIccid(const std::string& iccid);
  void CheckStaleESimServices();
  NetworkStateHandler::NetworkStateList GetESimCellularNetworks();
  bool HasQueuedRequest(const std::string& iccid) const;

  UninstallState state_ = UninstallState::kIdle;
  const NetworkState* curr_request_network_state_ = nullptr;
  base::circular_deque<std::unique_ptr<UninstallRequest>> uninstall_requests_;

  CellularInhibitor* cellular_inhibitor_ = nullptr;
  CellularESimProfileHandler* cellular_esim_profile_handler_ = nullptr;
  NetworkConfigurationHandler* network_configuration_handler_ = nullptr;
  NetworkConnectionHandler* network_connection_handler_ = nullptr;
  NetworkStateHandler* network_state_handler_ = nullptr;

  base::WeakPtrFactory<CellularESimUninstallHandler> weak_ptr_factory_{this};
};

}  // namespace chromeos

#endif  // CHROMEOS_SERVICES_CELLULAR_SETUP_ESIM_PROFILE_UNINSTALLER_H_