// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/chromeos/fileapi/external_file_url_util.h"

#include <string>
#include <vector>

#include "base/strings/string_util.h"
#include "base/strings/stringprintf.h"
#include "chrome/browser/chromeos/file_manager/app_id.h"
#include "chrome/browser/chromeos/file_manager/fileapi_util.h"
#include "chrome/browser/extensions/extension_util.h"
#include "chrome/browser/profiles/profile.h"
#include "content/public/browser/browser_context.h"
#include "content/public/browser/browser_thread.h"
#include "content/public/browser/storage_partition.h"
#include "content/public/common/url_constants.h"
#include "net/base/escape.h"
#include "storage/browser/file_system/file_system_context.h"
#include "storage/browser/file_system/file_system_url.h"

using content::BrowserThread;

namespace chromeos {

bool IsExternalFileURLType(storage::FileSystemType type, bool force) {
  return type == storage::kFileSystemTypeDeviceMediaAsFileStorage ||
         type == storage::kFileSystemTypeProvided ||
         type == storage::kFileSystemTypeArcContent ||
         type == storage::kFileSystemTypeArcDocumentsProvider || force;
}

GURL FileSystemURLToExternalFileURL(
    const storage::FileSystemURL& file_system_url,
    bool force) {
  if (file_system_url.mount_type() != storage::kFileSystemTypeExternal ||
      !IsExternalFileURLType(file_system_url.type(), force)) {
    return GURL();
  }

  return VirtualPathToExternalFileURL(file_system_url.virtual_path());
}

base::FilePath ExternalFileURLToVirtualPath(const GURL& url) {
  if (!url.is_valid() || url.scheme() != content::kExternalFileScheme)
    return base::FilePath();
  return base::FilePath::FromUTF8Unsafe(
      net::UnescapeBinaryURLComponent(url.path_piece()));
}

GURL VirtualPathToExternalFileURL(const base::FilePath& virtual_path) {
  return GURL(
      base::StringPrintf("%s:%s", content::kExternalFileScheme,
                         net::EscapePath(virtual_path.AsUTF8Unsafe()).c_str()));
}

GURL CreateExternalFileURLFromPath(Profile* profile,
                                   const base::FilePath& path,
                                   bool force) {
  DCHECK_CURRENTLY_ON(BrowserThread::UI);

  GURL raw_file_system_url;
  if (!file_manager::util::ConvertAbsoluteFilePathToFileSystemUrl(
          profile, path, file_manager::util::GetFileManagerURL(),
          &raw_file_system_url)) {
    return GURL();
  }

  const storage::FileSystemURL file_system_url =
      file_manager::util::GetFileManagerFileSystemContext(profile)->CrackURL(
          raw_file_system_url);
  if (!file_system_url.is_valid())
    return GURL();

  return FileSystemURLToExternalFileURL(file_system_url, force);
}

}  // namespace chromeos
