// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import {AvatarIcon} from 'chrome://resources/cr_elements/cr_profile_avatar_selector/cr_profile_avatar_selector.js';
import {addSingletonGetter, sendWithPromise} from 'chrome://resources/js/cr.m.js';

/**
 * This is the data structure sent back and forth between C++ and JS.
 * @typedef {{
 *   profilePath: string,
 *   localProfileName: string,
 *   isSyncing: boolean,
 *   needsSignin: boolean,
 *   gaiaName: string,
 *   userName: string,
 *   isManaged: boolean,
 *   avatarIcon: string,
 * }}
 */
export let ProfileState;

/**
 * This is the data structure sent back and forth between C++ and JS.
 * `colorId` has the following special values:
 *   - `-1` for the default theme.
 *   - `0` for a manually picked color theme.
 * @typedef {{
 *   colorId: number,
 *   color: number,
 *   themeFrameColor: string,
 *   themeShapeColor: string,
 *   themeFrameTextColor: string,
 *   themeGenericAvatar: string,
 * }}
 */
export let AutogeneratedThemeColorInfo;

/**
 * This is the data structure sent back and forth between C++ and JS.
 * `colorId` has the following special values:
 *   - `-1` for the default theme..
 *   - `0` for a manually picked color theme
 * `color` is defined only for manually picked themes.
 * @typedef {{
 *   colorId: number,
 *   color: (number|undefined),
 * }}
 */
export let UserThemeChoice;

/** @interface */
export class ManageProfilesBrowserProxy {
  /**
   * Initializes profile picker main view.
   */
  initializeMainView() {}

  /**
   * Launches picked profile and closes the profile picker.
   * @param {string} profilePath
   */
  launchSelectedProfile(profilePath) {}

  /**
   * Opens profile on manage profile settings sub page and closes the
   * profile picker.
   * @param {string} profilePath
   */
  openManageProfileSettingsSubPage(profilePath) {}

  /** Launches Guest profile. */
  launchGuestProfile() {}

  /**
   * Inform native the user's choice on whether to show the profile picker
   * on startup or not.
   * @param {boolean} shouldShow
   */
  askOnStartupChanged(shouldShow) {}

  /**
   * Retrieves suggested theme for the new profile.
   * @return {!Promise<!AutogeneratedThemeColorInfo>} A promise firing with the
   * suggested theme info, once it has been retrieved.
   */
  getNewProfileSuggestedThemeInfo() {}

  /**
   * Retrieves all relevant theme information for the particular theme.
   * @param {!UserThemeChoice} theme A theme which info needs to be retrieved.
   * @return {!Promise<!AutogeneratedThemeColorInfo>} A promise firing with the
   * theme info, once it has been retrieved.
   */
  getProfileThemeInfo(theme) {}

  /**
   * Retrieves profile statistics to be shown in the remove profile warning.
   * @param {string} profilePath
   */
  getProfileStatistics(profilePath) {}

  /**
   * Removes profile.
   * @param {string} profilePath
   */
  removeProfile(profilePath) {}

  /**
   * Loads Google sign in page (and silently creates a profile with the
   * specified color, if specified).
   * @param {?number} profileColor
   */
  loadSignInProfileCreationFlow(profileColor) {}

  /**
   * Retrieves custom avatar list for the select avatar dialog.
   * * @return {!Promise<!Array<!AvatarIcon>>}
   */
  getAvailableIcons() {}

  /**
   * Creates local profile
   * @param {string} profileName
   * @param {number} profileColor
   * @param {number} avatarIndex
   * @param {boolean} createShortcut
   */
  createProfile(profileName, profileColor, avatarIndex, createShortcut) {}

  /**
   * Sets the local profile name.
   * @param {string} profilePath
   * @param {string} profileName
   */
  setProfileName(profilePath, profileName) {}

  /** Records impression of a sign-in promo to metrics. */
  recordSignInPromoImpression() {}

  /**
   * Gets a profile for which the profile switch screen is shown.
   * @return {!Promise<!ProfileState>}
   */
  getSwitchProfile() {}

  /**
   * Switches to an already existing profile at `profile_path`.
   * @param {string} profilePath
   */
  confirmProfileSwitch(profilePath) {}

  /**
   * Cancels the profile switch which aborts the sign-in profile creation
   * flow.
   */
  cancelProfileSwitch() {}
}

/** @implements {ManageProfilesBrowserProxy} */
export class ManageProfilesBrowserProxyImpl {
  /** @override */
  initializeMainView() {
    chrome.send('mainViewInitialize');
  }

  /** @override */
  launchSelectedProfile(profilePath) {
    chrome.send('launchSelectedProfile', [profilePath]);
  }

  /** @override */
  openManageProfileSettingsSubPage(profilePath) {
    chrome.send('openManageProfileSettingsSubPage', [profilePath]);
  }

  /** @override */
  launchGuestProfile() {
    chrome.send('launchGuestProfile');
  }

  /** @override */
  askOnStartupChanged(shouldShow) {
    chrome.send('askOnStartupChanged', [shouldShow]);
  }

  /** @override */
  getNewProfileSuggestedThemeInfo() {
    return sendWithPromise('getNewProfileSuggestedThemeInfo');
  }

  /** @override */
  getProfileThemeInfo(theme) {
    return sendWithPromise('getProfileThemeInfo', theme);
  }

  /** @override */
  removeProfile(profilePath) {
    chrome.send('removeProfile', [profilePath]);
  }

  /** @override */
  getProfileStatistics(profilePath) {
    chrome.send('getProfileStatistics', [profilePath]);
  }

  /** @override */
  loadSignInProfileCreationFlow(profileColor) {
    chrome.send('loadSignInProfileCreationFlow', [profileColor]);
  }

  /** @override */
  getAvailableIcons() {
    return sendWithPromise('getAvailableIcons');
  }

  /** @override */
  createProfile(profileName, profileColor, avatarIndex, createShortcut) {
    chrome.send(
        'createProfile',
        [profileName, profileColor, avatarIndex, createShortcut]);
  }

  /** @override */
  setProfileName(profilePath, profileName) {
    chrome.send('setProfileName', [profilePath, profileName]);
  }

  /** @override */
  recordSignInPromoImpression() {
    chrome.send('recordSignInPromoImpression');
  }

  /** @override */
  getSwitchProfile() {
    return sendWithPromise('getSwitchProfile');
  }

  /** @override */
  confirmProfileSwitch(profilePath) {
    chrome.send('confirmProfileSwitch', [profilePath]);
  }

  /** @override */
  cancelProfileSwitch() {
    chrome.send('cancelProfileSwitch');
  }
}

addSingletonGetter(ManageProfilesBrowserProxyImpl);
