// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_UI_VIEWS_PROFILES_AVATAR_TOOLBAR_BUTTON_DELEGATE_H_
#define CHROME_BROWSER_UI_VIEWS_PROFILES_AVATAR_TOOLBAR_BUTTON_DELEGATE_H_

#include <string>

#include "base/macros.h"
#include "base/memory/weak_ptr.h"
#include "base/scoped_observation.h"
#include "build/build_config.h"
#include "chrome/browser/profiles/profile_attributes_storage.h"
#include "chrome/browser/sync/sync_ui_util.h"
#include "chrome/browser/ui/browser_list_observer.h"
#include "chrome/browser/ui/views/profiles/avatar_toolbar_button.h"
#include "components/signin/public/identity_manager/identity_manager.h"
#include "components/sync/driver/sync_service.h"
#include "components/sync/driver/sync_service_observer.h"
#include "third_party/abseil-cpp/absl/types/optional.h"
#include "ui/gfx/image/image.h"

class Profile;

// Handles the business logic for AvatarToolbarButton. This includes
// managing the highlight animation and the identity animation.
class AvatarToolbarButtonDelegate : public BrowserListObserver,
                                    public ProfileAttributesStorage::Observer,
                                    public signin::IdentityManager::Observer,
                                    public syncer::SyncServiceObserver {
 public:
  AvatarToolbarButtonDelegate(AvatarToolbarButton* button, Profile* profile);
  ~AvatarToolbarButtonDelegate() override;

  // Called by the AvatarToolbarButton to get information about the profile.
  std::u16string GetProfileName() const;
  std::u16string GetShortProfileName() const;
  gfx::Image GetGaiaAccountImage() const;
  gfx::Image GetProfileAvatarImage(gfx::Image gaia_account_image,
                                   int preferred_size) const;

  // Returns the count of incognito or guest windows attached to the profile.
  int GetWindowCount() const;

  AvatarToolbarButton::State GetState() const;

  void ShowHighlightAnimation();
  bool IsHighlightAnimationVisible() const;

  // Should be called when the icon is updated. This may trigger the identity
  // pill animation if the delegate is waiting for the image.
  void MaybeShowIdentityAnimation(const gfx::Image& gaia_account_image);

  // Enables or disables the IPH highlight.
  void SetHasInProductHelpPromo(bool has_promo);

  // Called by the AvatarToolbarButton to notify the delegate about events.
  void NotifyClick();
  void OnMouseExited();
  void OnBlur();
  void OnHighlightChanged();

 private:
  enum class IdentityAnimationState { kNotShowing, kWaitingForImage, kShowing };

  // BrowserListObserver:
  void OnBrowserAdded(Browser* browser) override;
  void OnBrowserRemoved(Browser* browser) override;

  // ProfileAttributesStorage::Observer:
  void OnProfileAdded(const base::FilePath& profile_path) override;
  void OnProfileWasRemoved(const base::FilePath& profile_path,
                           const std::u16string& profile_name) override;
  void OnProfileAvatarChanged(const base::FilePath& profile_path) override;
  void OnProfileHighResAvatarLoaded(
      const base::FilePath& profile_path) override;
  void OnProfileNameChanged(const base::FilePath& profile_path,
                            const std::u16string& old_profile_name) override;

  // IdentityManager::Observer:
  // Needed if the first sync promo account should be displayed.
  void OnPrimaryAccountChanged(
      const signin::PrimaryAccountChangeEvent& event) override;
  void OnRefreshTokensLoaded() override;
  void OnAccountsInCookieUpdated(
      const signin::AccountsInCookieJarInfo& accounts_in_cookie_jar_info,
      const GoogleServiceAuthError& error) override;
  void OnExtendedAccountInfoUpdated(const AccountInfo& info) override;
  void OnExtendedAccountInfoRemoved(const AccountInfo& info) override;

  // SyncServiceObserver:
  void OnStateChanged(syncer::SyncService*) override;

  // Initiates showing the identity.
  void OnUserIdentityChanged();

  void OnIdentityAnimationTimeout();
  // Called after the user interacted with the button or after some timeout.
  void MaybeHideIdentityAnimation();
  void HideHighlightAnimation();

  // Shows the identity pill animation. If the animation is already showing,
  // this extends the duration of the current animation.
  void ShowIdentityAnimation();

  base::ScopedObservation<ProfileAttributesStorage,
                          ProfileAttributesStorage::Observer>
      profile_observation_{this};
  base::ScopedObservation<syncer::SyncService, syncer::SyncServiceObserver>
      sync_service_observation_{this};
  base::ScopedObservation<signin::IdentityManager,
                          signin::IdentityManager::Observer>
      identity_manager_observation_{this};

  AvatarToolbarButton* const avatar_toolbar_button_;
  Profile* const profile_;
  IdentityAnimationState identity_animation_state_ =
      IdentityAnimationState::kNotShowing;

  // Count of identity pill animation timeouts that are currently scheduled.
  // Multiple timeouts are scheduled when multiple animation triggers happen in
  // a quick sequence (before the first timeout passes). The identity pill tries
  // to close when this reaches 0.
  int identity_animation_timeout_count_ = 0;

  bool refresh_tokens_loaded_ = false;
  bool has_in_product_help_promo_ = false;

  // Whether the avatar highlight animation is visible. The animation is shown
  // when an Autofill datatype is saved. When this is true the avatar button
  // sync paused/error state will be disabled.
  bool highlight_animation_visible_ = false;

  // Caches the value of the last error so the class can detect when it changes
  // and notify |avatar_toolbar_button_|.
  absl::optional<sync_ui_util::AvatarSyncErrorType> last_avatar_error_;

  base::WeakPtrFactory<AvatarToolbarButtonDelegate> weak_ptr_factory_{this};

  DISALLOW_COPY_AND_ASSIGN(AvatarToolbarButtonDelegate);
};

#endif  // CHROME_BROWSER_UI_VIEWS_PROFILES_AVATAR_TOOLBAR_BUTTON_DELEGATE_H_
