// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/webui/signin/enterprise_profile_welcome_handler.h"

#include "base/bind.h"
#include "base/files/file_path.h"
#include "base/strings/utf_string_conversions.h"
#include "chrome/browser/browser_process.h"
#include "chrome/browser/policy/chrome_browser_policy_connector.h"
#include "chrome/browser/policy/profile_policy_connector.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/profiles/profile_attributes_entry.h"
#include "chrome/browser/profiles/profile_avatar_icon_util.h"
#include "chrome/browser/profiles/profile_manager.h"
#include "chrome/browser/ui/managed_ui.h"
#include "chrome/browser/ui/webui/management/management_ui_handler.h"
#include "chrome/common/themes/autogenerated_theme_util.h"
#include "chrome/grit/generated_resources.h"
#include "components/signin/public/identity_manager/account_info.h"
#include "components/strings/grit/components_strings.h"
#include "third_party/skia/include/core/SkBitmap.h"
#include "ui/base/l10n/l10n_util.h"
#include "ui/gfx/color_utils.h"
#include "ui/gfx/image/image.h"

namespace {
const int kAvatarSize = 100;

std::string GetManagedAccountTitle(ProfileAttributesEntry* entry,
                                   const std::string& fallback_domain_name) {
  DCHECK(entry);
  if (entry->GetHostedDomain() == kNoHostedDomainFound)
    return std::string();
  const std::string domain_name = entry->GetHostedDomain().empty()
                                      ? fallback_domain_name
                                      : entry->GetHostedDomain();
  return l10n_util::GetStringFUTF8(
      IDS_ENTERPRISE_PROFILE_WELCOME_ACCOUNT_MANAGED_BY,
      base::UTF8ToUTF16(domain_name));
}

std::string GetManagedDeviceTitle() {
  absl::optional<std::string> device_manager =
      chrome::GetDeviceManagerIdentity();
  if (!device_manager)
    return std::string();
  if (device_manager->empty()) {
    return l10n_util::GetStringUTF8(
        IDS_ENTERPRISE_PROFILE_WELCOME_DEVICE_MANAGED);
  }
  return l10n_util::GetStringFUTF8(
      IDS_ENTERPRISE_PROFILE_WELCOME_DEVICE_MANAGED_BY,
      base::UTF8ToUTF16(*device_manager));
}

SkColor GetFrameColor(SkColor theme_color) {
  auto theme_colors = GetAutogeneratedThemeColors(theme_color);
  return theme_colors.frame_color;
}

}  // namespace

EnterpriseProfileWelcomeHandler::EnterpriseProfileWelcomeHandler(
    EnterpriseProfileWelcomeUI::ScreenType type,
    const std::string& domain_name,
    SkColor profile_color,
    base::OnceCallback<void(bool)> proceed_callback)
    : type_(type),
      domain_name_(domain_name),
      profile_color_(profile_color),
      proceed_callback_(std::move(proceed_callback)) {
  DCHECK(proceed_callback_);
}

EnterpriseProfileWelcomeHandler::~EnterpriseProfileWelcomeHandler() = default;

void EnterpriseProfileWelcomeHandler::RegisterMessages() {
  profile_path_ = Profile::FromWebUI(web_ui())->GetPath();
  web_ui()->RegisterMessageCallback(
      "initialized",
      base::BindRepeating(&EnterpriseProfileWelcomeHandler::HandleInitialized,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "proceed",
      base::BindRepeating(&EnterpriseProfileWelcomeHandler::HandleProceed,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "cancel",
      base::BindRepeating(&EnterpriseProfileWelcomeHandler::HandleCancel,
                          base::Unretained(this)));
}

void EnterpriseProfileWelcomeHandler::OnProfileAvatarChanged(
    const base::FilePath& profile_path) {
  UpdateProfileInfo(profile_path);
}

void EnterpriseProfileWelcomeHandler::OnProfileHighResAvatarLoaded(
    const base::FilePath& profile_path) {
  UpdateProfileInfo(profile_path);
}

void EnterpriseProfileWelcomeHandler::OnProfileHostedDomainChanged(
    const base::FilePath& profile_path) {
  UpdateProfileInfo(profile_path);
}

void EnterpriseProfileWelcomeHandler::OnJavascriptAllowed() {
  observed_profile_.Observe(
      &g_browser_process->profile_manager()->GetProfileAttributesStorage());
}

void EnterpriseProfileWelcomeHandler::OnJavascriptDisallowed() {
  observed_profile_.Reset();
}

void EnterpriseProfileWelcomeHandler::HandleInitialized(
    const base::ListValue* args) {
  CHECK_EQ(1u, args->GetSize());
  AllowJavascript();
  const base::Value& callback_id = args->GetList()[0];
  ResolveJavascriptCallback(callback_id, GetProfileInfoValue());
}

void EnterpriseProfileWelcomeHandler::HandleProceed(
    const base::ListValue* args) {
  if (proceed_callback_)
    std::move(proceed_callback_).Run(true);
}

void EnterpriseProfileWelcomeHandler::HandleCancel(
    const base::ListValue* args) {
  if (proceed_callback_)
    std::move(proceed_callback_).Run(false);
}

void EnterpriseProfileWelcomeHandler::UpdateProfileInfo(
    const base::FilePath& profile_path) {
  DCHECK(IsJavascriptAllowed());
  if (profile_path != profile_path_)
    return;
  FireWebUIListener("on-profile-info-changed", GetProfileInfoValue());
}

base::Value EnterpriseProfileWelcomeHandler::GetProfileInfoValue() {
  base::Value dict(base::Value::Type::DICTIONARY);
  dict.SetStringKey("backgroundColor", color_utils::SkColorToRgbaString(
                                           GetFrameColor(profile_color_)));

  ProfileAttributesEntry* entry = GetProfileEntry();
  const int avatar_icon_size = kAvatarSize * web_ui()->GetDeviceScaleFactor();
  gfx::Image icon =
      profiles::GetSizedAvatarIcon(entry->GetAvatarIcon(avatar_icon_size), true,
                                   avatar_icon_size, avatar_icon_size);
  dict.SetStringKey("pictureUrl", webui::GetBitmapDataUrl(icon.AsBitmap()));

  std::string enterprise_title;
  std::string enterprise_info;

  switch (type_) {
    case EnterpriseProfileWelcomeUI::ScreenType::kEntepriseAccountSyncEnabled:
      dict.SetBoolKey("showEnterpriseBadge", true);
      enterprise_title = GetManagedAccountTitle(entry, domain_name_);
      enterprise_info = l10n_util::GetStringUTF8(
          IDS_ENTERPRISE_PROFILE_WELCOME_MANAGED_DESCRIPTION_WITH_SYNC);
      dict.SetStringKey("proceedLabel", "Next");
      break;
    case EnterpriseProfileWelcomeUI::ScreenType::kEntepriseAccountSyncDisabled:
      dict.SetBoolKey("showEnterpriseBadge", true);
      enterprise_title = GetManagedAccountTitle(entry, domain_name_);
      enterprise_info = l10n_util ::GetStringUTF8(
          IDS_ENTERPRISE_PROFILE_WELCOME_MANAGED_DESCRIPTION_WITHOUT_SYNC);
      dict.SetStringKey("proceedLabel", l10n_util::GetStringUTF8(IDS_DONE));
      break;
    case EnterpriseProfileWelcomeUI::ScreenType::kConsumerAccountSyncDisabled:
      dict.SetBoolKey("showEnterpriseBadge", false);
      enterprise_title = GetManagedDeviceTitle();
      enterprise_info =
          l10n_util::GetStringUTF8(IDS_SYNC_DISABLED_CONFIRMATION_DETAILS);
      dict.SetStringKey("proceedLabel", l10n_util::GetStringUTF8(IDS_DONE));
      break;
  }

  dict.SetStringKey("enterpriseTitle", enterprise_title);
  dict.SetStringKey("enterpriseInfo", enterprise_info);

  return dict;
}

ProfileAttributesEntry* EnterpriseProfileWelcomeHandler::GetProfileEntry()
    const {
  ProfileAttributesEntry* entry =
      g_browser_process->profile_manager()
          ->GetProfileAttributesStorage()
          .GetProfileAttributesWithPath(profile_path_);
  DCHECK(entry);
  return entry;
}

EnterpriseProfileWelcomeUI::ScreenType
EnterpriseProfileWelcomeHandler::GetTypeForTesting() {
  return type_;
}

void EnterpriseProfileWelcomeHandler::CallProceedCallbackForTesting(
    bool proceed) {
  if (proceed_callback_)
    std::move(proceed_callback_).Run(proceed);
}
