// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/app_list/search/files/file_result.h"

#include <string>
#include <utility>
#include <vector>

#include "ash/public/cpp/app_list/app_list_types.h"
#include "ash/public/cpp/ash_features.h"
#include "ash/public/cpp/file_icon_util.h"
#include "ash/public/cpp/style/color_provider.h"
#include "base/bind.h"
#include "base/files/file_path.h"
#include "base/i18n/rtl.h"
#include "base/macros.h"
#include "base/metrics/histogram_macros.h"
#include "base/no_destructor.h"
#include "base/strings/string_util.h"
#include "base/strings/utf_string_conversions.h"
#include "chrome/browser/platform_util.h"
#include "chrome/browser/ui/app_list/search/search_tags_util.h"
#include "chrome/grit/generated_resources.h"
#include "chromeos/components/string_matching/tokenized_string_match.h"
#include "ui/base/l10n/l10n_util.h"

namespace app_list {

namespace {

using chromeos::string_matching::TokenizedString;
using chromeos::string_matching::TokenizedStringMatch;

std::string StripHostedFileExtensions(const std::string& filename) {
  static const base::NoDestructor<std::vector<std::string>> hosted_extensions(
      {".GDOC", ".GSHEET", ".GSLIDES", ".GDRAW", ".GTABLE", ".GLINK", ".GFORM",
       ".GMAPS", ".GSITE"});

  for (const auto& extension : *hosted_extensions) {
    if (EndsWith(filename, extension, base::CompareCase::INSENSITIVE_ASCII)) {
      return filename.substr(0, filename.size() - extension.size());
    }
  }

  return filename;
}

// Helper function for calculating a file's relevance score. Will return a
// default relevance score if the query is missing or the filename is empty.
double CalculateRelevance(const absl::optional<TokenizedString>& query,
                          const std::u16string& raw_title) {
  const TokenizedString title(raw_title, TokenizedString::Mode::kWords);

  const bool use_default_relevance =
      !query || query.value().text().empty() || title.text().empty();
  UMA_HISTOGRAM_BOOLEAN("Apps.AppList.FileResult.DefaultRelevanceUsed",
                        use_default_relevance);
  if (use_default_relevance) {
    static constexpr double kDefaultRelevance = 0.5;
    return kDefaultRelevance;
  }

  TokenizedStringMatch match;
  match.Calculate(query.value(), title);
  return match.relevance();
}

void LogRelevance(ChromeSearchResult::ResultType result_type,
                  const double relevance) {
  // Relevance scores are between 0 and 1, so we scale to 0 to 100 for logging.
  DCHECK((relevance >= 0) && (relevance <= 1));
  const int scaled_relevance = floor(100 * relevance);
  switch (result_type) {
    case FileResult::ResultType::kFileSearch:
      UMA_HISTOGRAM_EXACT_LINEAR("Apps.AppList.FileSearchProvider.Relevance",
                                 scaled_relevance, /*exclusive_max=*/101);
      break;
    case FileResult::ResultType::kDriveSearch:
      UMA_HISTOGRAM_EXACT_LINEAR("Apps.AppList.DriveSearchProvider.Relevance",
                                 scaled_relevance, /*exclusive_max=*/101);
      break;
    case FileResult::ResultType::kZeroStateFile:
      UMA_HISTOGRAM_EXACT_LINEAR("Apps.AppList.ZeroStateFileProvider.Relevance",
                                 scaled_relevance, /*exclusive_max=*/101);
      break;
    case FileResult::ResultType::kZeroStateDrive:
      UMA_HISTOGRAM_EXACT_LINEAR(
          "Apps.AppList.ZeroStateDriveProvider.Relevance", scaled_relevance,
          /*exclusive_max=*/101);
      break;
    default:
      NOTREACHED();
  }
}

}  // namespace

FileResult::FileResult(const std::string& schema,
                       const base::FilePath& filepath,
                       ResultType result_type,
                       DisplayType display_type,
                       Type type,
                       Profile* profile)
    : filepath_(filepath), type_(type), profile_(profile) {
  DCHECK(profile);
  set_id(schema + filepath.value());

  SetResultType(result_type);
  switch (result_type) {
    case ResultType::kDriveChip:
    case ResultType::kZeroStateDrive:
      SetMetricsType(ash::ZERO_STATE_DRIVE);
      break;
    case ResultType::kFileChip:
    case ResultType::kZeroStateFile:
      SetMetricsType(ash::ZERO_STATE_FILE);
      break;
    case ResultType::kFileSearch:
      SetMetricsType(ash::FILE_SEARCH);
      break;
    case ResultType::kDriveSearch:
      SetMetricsType(ash::DRIVE_SEARCH);
      break;
    default:
      NOTREACHED();
  }

  SetDisplayType(display_type);

  // Set the details to the display name of the Files app.
  std::u16string sanitized_name = base::CollapseWhitespace(
      l10n_util::GetStringUTF16(IDS_FILEMANAGER_APP_NAME), true);
  base::i18n::SanitizeUserSuppliedString(&sanitized_name);
  SetDetails(sanitized_name);
  SetTitle(base::UTF8ToUTF16(
      StripHostedFileExtensions(filepath.BaseName().value())));
}

FileResult::FileResult(const std::string& schema,
                       const base::FilePath& filepath,
                       ResultType result_type,
                       DisplayType display_type,
                       float relevance,
                       Profile* profile)
    : FileResult(schema,
                 filepath,
                 result_type,
                 display_type,
                 Type::kFile,
                 profile) {
  set_relevance(relevance);
  if (display_type == DisplayType::kList) {
    // Chip and list results overlap, and only list results are fully launched.
    // So we only log metrics for list results.
    LogRelevance(result_type, relevance);
  }

  // Launcher search results UI is light by default, so use icons for light
  // background if dark/light mode feature is not enabled.
  const bool dark_background = ash::features::IsDarkLightModeEnabled() &&
                               ash::ColorProvider::Get()->IsDarkModeEnabled();
  switch (display_type) {
    case DisplayType::kChip:
      SetChipIcon(ash::GetChipIconForPath(filepath, dark_background));
      break;
    case DisplayType::kList:
      SetIcon(ash::GetIconForPath(filepath, dark_background));
      break;
    default:
      NOTREACHED();
  }
}

FileResult::FileResult(
    const std::string& schema,
    const base::FilePath& filepath,
    ResultType result_type,
    const std::u16string& query,
    const absl::optional<chromeos::string_matching::TokenizedString>&
        tokenized_query,
    Type type,
    Profile* profile)
    : FileResult(schema,
                 filepath,
                 result_type,
                 DisplayType::kList,
                 type,
                 profile) {
  const double relevance = CalculateRelevance(tokenized_query, title());
  set_relevance(relevance);
  LogRelevance(result_type, relevance);

  SetTitleTags(CalculateTags(query, title()));

  // Launcher search results UI is light by default, so use icons for light
  // background if dark/light mode feature is not enabled.
  const bool dark_background = ash::features::IsDarkLightModeEnabled() &&
                               ash::ColorProvider::Get()->IsDarkModeEnabled();
  switch (type) {
    case Type::kFile:
      SetIcon(ash::GetIconForPath(filepath, dark_background));
      break;
    case Type::kDirectory:
      SetIcon(ash::GetIconFromType("folder", dark_background));
      break;
    case Type::kSharedDirectory:
      SetIcon(ash::GetIconFromType("shared", dark_background));
      break;
  }
}

FileResult::~FileResult() = default;

void FileResult::Open(int event_flags) {
  switch (type_) {
    case Type::kFile:
      platform_util::OpenItem(profile_, filepath_,
                              platform_util::OpenItemType::OPEN_FILE,
                              platform_util::OpenOperationCallback());
      break;
    case Type::kDirectory:
    case Type::kSharedDirectory:
      platform_util::OpenItem(profile_, filepath_,
                              platform_util::OpenItemType::OPEN_FOLDER,
                              platform_util::OpenOperationCallback());
      break;
  }
}

::std::ostream& operator<<(::std::ostream& os, const FileResult& result) {
  return os << "{" << result.title() << ", " << result.relevance() << "}";
}

}  // namespace app_list
