// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/ash/sharesheet/sharesheet_header_view.h"

#include <algorithm>
#include <string>
#include <utility>

#include "ash/public/cpp/ash_typography.h"
#include "ash/public/cpp/file_icon_util.h"
#include "ash/public/cpp/style/color_provider.h"
#include "base/bind.h"
#include "base/files/file_util.h"
#include "base/strings/strcat.h"
#include "base/strings/string_util.h"
#include "chrome/app/vector_icons/vector_icons.h"
#include "chrome/browser/chromeos/file_manager/app_id.h"
#include "chrome/browser/chromeos/file_manager/fileapi_util.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/sharesheet/sharesheet_metrics.h"
#include "chrome/browser/sharesheet/sharesheet_types.h"
#include "chrome/browser/ui/ash/sharesheet/sharesheet_bubble_view.h"
#include "chrome/browser/ui/ash/sharesheet/sharesheet_constants.h"
#include "chrome/browser/ui/ash/sharesheet/sharesheet_util.h"
#include "chrome/browser/ui/views/chrome_typography.h"
#include "chrome/common/chrome_features.h"
#include "chrome/grit/generated_resources.h"
#include "chromeos/ui/vector_icons/vector_icons.h"
#include "storage/browser/file_system/file_system_context.h"
#include "storage/browser/file_system/file_system_url.h"
#include "third_party/abseil-cpp/absl/types/optional.h"
#include "ui/accessibility/ax_enums.mojom.h"
#include "ui/accessibility/ax_node_data.h"
#include "ui/base/l10n/l10n_util.h"
#include "ui/base/metadata/metadata_impl_macros.h"
#include "ui/gfx/color_palette.h"
#include "ui/gfx/geometry/size.h"
#include "ui/gfx/image/image.h"
#include "ui/gfx/image/image_skia.h"
#include "ui/gfx/paint_vector_icon.h"
#include "ui/views/controls/image_view.h"
#include "ui/views/controls/label.h"
#include "ui/views/layout/box_layout.h"
#include "ui/views/layout/layout_provider.h"
#include "ui/views/view.h"
#include "ui/views/view_class_properties.h"

namespace {

// Concatenates all the strings in |file_names| with a comma delineator.
const std::u16string ConcatenateFileNames(
    const std::vector<std::u16string>& file_names) {
  return base::JoinString(file_names, u", ");
}

gfx::ImageSkia CreateMimeTypeIcon(const gfx::ImageSkia& file_type_icon,
                                  const gfx::Size& image_size) {
  return ash::HoldingSpaceImage::SuperimposeOverEmptyImage(file_type_icon,
                                                           image_size);
}

gfx::Size GetImagePreviewSize(size_t index, int grid_icon_count) {
  switch (grid_icon_count) {
    case 1:
      return ash::sharesheet::kImagePreviewFullSize;
    case 2:
      return ash::sharesheet::kImagePreviewHalfSize;
    case 3:
      if (index == 0) {
        return ash::sharesheet::kImagePreviewHalfSize;
      } else {
        return ash::sharesheet::kImagePreviewQuarterSize;
      }
    default:
      return ash::sharesheet::kImagePreviewQuarterSize;
  }
}

}  // namespace

namespace ash {
namespace sharesheet {

// SharesheetHeaderView::SharesheetImagePreview
// ------------------------------------------------------

class SharesheetHeaderView::SharesheetImagePreview : public views::View {
 public:
  METADATA_HEADER(SharesheetImagePreview);
  explicit SharesheetImagePreview(size_t file_count) {
    const int border_radius =
        views::LayoutProvider::Get()->GetCornerRadiusMetric(
            views::Emphasis::kMedium);
    SetBackground(views::CreateRoundedRectBackground(
        kImagePreviewPlaceholderBackgroundColor, border_radius));
    SetBorder(views::CreateRoundedRectBorder(
        /* thickness */ 1, border_radius,
        GetNativeTheme()->GetSystemColor(
            ui::NativeTheme::kColorId_UnfocusedBorderColor)));
    SetLayoutManager(std::make_unique<views::BoxLayout>(
        views::BoxLayout::Orientation::kVertical,
        /* inside_border_insets */ gfx::Insets(),
        /* between_child_spacing */ kImagePreviewBetweenChildSpacing,
        /* collapse_margins_spacing */ false));
    SetPreferredSize(kImagePreviewFullSize);
    SetFocusBehavior(View::FocusBehavior::NEVER);

    size_t grid_icon_count =
        (file_count > 0) ? std::min(file_count, kImagePreviewMaxIcons) : 1;
    size_t enumeration = (file_count > kImagePreviewMaxIcons)
                             ? file_count - kImagePreviewMaxIcons + 1
                             : 0;

    if (grid_icon_count == 1) {
      AddImageViewTo(this, kImagePreviewFullSize);
      return;
    }

    // If we need to have more than 1 icon, add two rows so that we can
    // layout the icons in a grid.
    DCHECK_GT(grid_icon_count, 1);
    AddRowToImageContainerView();
    AddRowToImageContainerView();

    for (size_t index = 0; index < grid_icon_count; ++index) {
      // If we have |enumeration|, add it as a label at the bottom right of
      // SharesheetImagePreview.
      if (enumeration != 0 && index == kImagePreviewMaxIcons - 1) {
        // TODO(crbug.com/1189945) : Add a sharesheet context to replace
        // |CONTEXT_DOWNLOAD_SHELF_STATUS|.
        auto* label =
            children()[1]->AddChildView(std::make_unique<views::Label>(
                base::StrCat({u"+", base::NumberToString16(enumeration)}),
                CONTEXT_DOWNLOAD_SHELF_STATUS, STYLE_SHARESHEET));
        label->SetLineHeight(kImagePreviewFileEnumerationLineHeight);
        label->SetEnabledColor(kButtonTextColor);
        label->SetHorizontalAlignment(gfx::ALIGN_CENTER);
        label->SetBackground(views::CreateRoundedRectBackground(
            kImagePreviewPlaceholderBackgroundColor,
            kImagePreviewIconCornerRadius));
        label->SetPreferredSize(kImagePreviewQuarterSize);
        return;
      }
      AddImageViewAt(index, grid_icon_count,
                     GetImagePreviewSize(index, grid_icon_count));
    }
  }

  SharesheetImagePreview(const SharesheetImagePreview&) = delete;
  SharesheetImagePreview& operator=(const SharesheetImagePreview&) = delete;

  ~SharesheetImagePreview() override {
    ::sharesheet::SharesheetMetrics::RecordSharesheetImagePreviewPressed(
        was_pressed_);
  }

  views::ImageView* GetImageViewAt(size_t index) {
    if (index >= image_views_.size()) {
      return nullptr;
    }
    return image_views_[index];
  }

  const size_t GetImageViewCount() { return image_views_.size(); }

  void SetBackgroundColorForIndex(const int index, const SkColor& color) {
    auto alpha_color =
        SkColorSetA(color, kImagePreviewBackgroundAlphaComponent);
    image_views_[index]->SetBackground(views::CreateRoundedRectBackground(
        alpha_color, kImagePreviewIconCornerRadius));
  }

 private:
  // views::View:
  bool OnMousePressed(const ui::MouseEvent& event) override {
    was_pressed_ = true;
    return false;
  }

  void OnGestureEvent(ui::GestureEvent* event) override {
    if (event->type() == ui::ET_GESTURE_TAP)
      was_pressed_ = true;
  }

  void AddRowToImageContainerView() {
    auto* row = AddChildView(std::make_unique<views::View>());
    row->SetLayoutManager(std::make_unique<views::BoxLayout>(
        views::BoxLayout::Orientation::kHorizontal,
        /* inside_border_insets */ gfx::Insets(),
        /* between_child_spacing */ kImagePreviewBetweenChildSpacing,
        /* collapse_margins_spacing */ false));
  }

  void AddImageViewTo(views::View* parent_view, const gfx::Size& size) {
    auto* image_view =
        parent_view->AddChildView(std::make_unique<views::ImageView>());
    image_view->SetImageSize(size);
    image_views_.push_back(image_view);
  }

  void AddImageViewAt(size_t index,
                      size_t grid_icon_count,
                      const gfx::Size& size) {
    views::View* parent_view = this;
    if (grid_icon_count > 1) {
      int row_num = 0;
      // For 2 icons, add to the second row for the second icons.
      // For 3 icons, add to the second row for the second and third icons.
      // For 4+ icons, add to the second row for the third and fourth icons.
      if ((grid_icon_count == 2 && index == 1) ||
          (grid_icon_count == 3 && index != 0) ||
          (grid_icon_count >= 4 && index > 1)) {
        row_num = 1;
      }
      parent_view = children()[row_num];
    }
    AddImageViewTo(parent_view, size);
  }

  std::vector<views::ImageView*> image_views_;

  // Used for recording UMA to indicate whether or not a user tried to interact
  // with the image preview.
  bool was_pressed_ = false;
};

BEGIN_METADATA(SharesheetHeaderView, SharesheetImagePreview, views::View)
END_METADATA

// SharesheetHeaderView --------------------------------------------------------

SharesheetHeaderView::SharesheetHeaderView(apps::mojom::IntentPtr intent,
                                           Profile* profile,
                                           bool show_content_previews)
    : profile_(profile),
      intent_(std::move(intent)),
      thumbnail_loader_(profile) {
  auto* layout = SetLayoutManager(std::make_unique<views::BoxLayout>(
      views::BoxLayout::Orientation::kHorizontal,
      /* inside_border_insets */ gfx::Insets(kSpacing),
      /* between_child_spacing */ kHeaderViewBetweenChildSpacing,
      /* collapse_margins_spacing */ false));
  // Sets all views to be left-aligned.
  layout->set_main_axis_alignment(views::BoxLayout::MainAxisAlignment::kStart);
  // Sets all views to be vertically centre-aligned.
  layout->set_cross_axis_alignment(
      views::BoxLayout::CrossAxisAlignment::kCenter);
  SetFocusBehavior(View::FocusBehavior::ALWAYS);

  const bool has_files =
      (intent_->file_urls.has_value() && !intent_->file_urls.value().empty());
  // The image view is initialised first to ensure its left most placement.
  if (show_content_previews) {
    auto file_count = (has_files) ? intent_->file_urls.value().size() : 0;
    image_preview_ =
        AddChildView(std::make_unique<SharesheetImagePreview>(file_count));
  }
  // A separate view is created for the share title and preview string views.
  text_view_ = AddChildView(std::make_unique<views::View>());
  text_view_->SetLayoutManager(std::make_unique<views::BoxLayout>(
      views::BoxLayout::Orientation::kVertical,
      /* inside_border_insets */ gfx::Insets(),
      /* between_child_spacing */ 0, /* collapse_margins_spacing */ true));
  text_view_->AddChildView(
      CreateShareLabel(l10n_util::GetStringUTF16(IDS_SHARESHEET_TITLE_LABEL),
                       CONTEXT_SHARESHEET_BUBBLE_TITLE, kTitleTextLineHeight,
                       kTitleTextColor, gfx::ALIGN_LEFT));
  if (show_content_previews) {
    ShowTextPreview();
    if (has_files) {
      ResolveImages();
    } else {
      DCHECK_GT(image_preview_->GetImageViewCount(), 0);
      const auto icon_color = ColorProvider::Get()->GetContentLayerColor(
          ColorProvider::ContentLayerType::kIconColorProminent);
      gfx::ImageSkia file_type_icon = gfx::CreateVectorIcon(
          GetTextVectorIcon(),
          sharesheet::kImagePreviewPlaceholderIconContentSize, icon_color);
      image_preview_->GetImageViewAt(0)->SetImage(
          CreateMimeTypeIcon(file_type_icon, kImagePreviewFullSize));
      image_preview_->SetBackgroundColorForIndex(0, icon_color);
    }
  }
}

SharesheetHeaderView::~SharesheetHeaderView() = default;

void SharesheetHeaderView::GetAccessibleNodeData(ui::AXNodeData* node_data) {
  node_data->role = ax::mojom::Role::kGenericContainer;
  node_data->SetNameExplicitlyEmpty();
}

void SharesheetHeaderView::ShowTextPreview() {
  std::vector<std::u16string> text_fields = ExtractShareText();

  std::u16string filenames_tooltip_text = u"";
  if (intent_->file_urls.has_value() && !intent_->file_urls.value().empty()) {
    std::vector<std::u16string> file_names;
    for (const auto& file_url : intent_->file_urls.value()) {
      const auto& file_path = GetFilePathFromFileSystemUrl(file_url);
      file_names.push_back(file_path.BaseName().LossyDisplayName());
    }
    std::u16string file_text;
    if (file_names.size() == 1) {
      file_text = file_names[0];
    } else {
      // If there is more than 1 file, show an enumeration of the number of
      // files.
      auto size = intent_->file_urls.value().size();
      DCHECK_NE(size, 0);
      file_text =
          l10n_util::GetPluralStringFUTF16(IDS_SHARESHEET_FILES_LABEL, size);
      filenames_tooltip_text = ConcatenateFileNames(file_names);
    }
    text_fields.push_back(file_text);
  }

  if (text_fields.size() == 0)
    return;

  int index = 0;
  int max_lines = std::min(text_fields.size(), kTextPreviewMaximumLines);
  for (; index < max_lines - 1; ++index) {
    AddTextLine(text_fields[index]);
  }
  // File names must always be on the last line, so |filenames_tooltip_text| is
  // only passed in on the last line of text. If there are no files, it will be
  // empty and the tooltip will instead be set to what the text says.
  DCHECK_LT(index, text_fields.size());
  AddTextLine(text_fields[index], filenames_tooltip_text);

  // If we have 2 or more lines of text, shorten the vertical insets.
  if (index >= 1) {
    static_cast<views::BoxLayout*>(GetLayoutManager())
        ->set_inside_border_insets(
            gfx::Insets(/* vertical */ kHeaderViewNarrowInsideBorderInsets,
                        /* horizontal */ kSpacing));
  }
}

void SharesheetHeaderView::AddTextLine(const std::u16string& text,
                                       const std::u16string& tooltip_text) {
  auto* new_line = text_view_->AddChildView(CreateShareLabel(
      text, CONTEXT_SHARESHEET_BUBBLE_BODY, kPrimaryTextLineHeight,
      kPrimaryTextColor, gfx::ALIGN_LEFT, views::style::STYLE_PRIMARY));
  new_line->SetHandlesTooltips(true);
  if (tooltip_text.empty()) {
    return;
  }
  new_line->SetTooltipText(tooltip_text);
  // We only get to here if this line is showing the number of files.
  // By default the accessible name is set to the label text. We set it here
  // so that it is also gives the list of file names.
  new_line->SetAccessibleName(
      base::StrCat({new_line->GetText(), u" ", tooltip_text}));
}

std::vector<std::u16string> SharesheetHeaderView::ExtractShareText() {
  std::vector<std::u16string> text_fields;

  if (intent_->share_title.has_value() &&
      !(intent_->share_title.value().empty())) {
    std::string title_text = intent_->share_title.value();
    text_fields.push_back(base::ASCIIToUTF16(title_text));
  }

  if (intent_->share_text.has_value() &&
      !(intent_->share_text.value().empty())) {
    std::string extracted_text = intent_->share_text.value();
    size_t last_space = extracted_text.find_last_of(' ');
    GURL extracted_url;

    if (last_space == std::string::npos) {
      extracted_url = GURL(extracted_text);
      if (extracted_url.is_valid())
        extracted_text.clear();
    } else {
      extracted_url = GURL(extracted_text.substr(last_space + 1));
      if (extracted_url.is_valid())
        extracted_text.erase(last_space);
    }

    if (!extracted_text.empty())
      text_fields.push_back(base::ASCIIToUTF16(extracted_text));

    if (extracted_url.is_valid()) {
      text_fields.push_back(base::ASCIIToUTF16(extracted_url.spec()));
      text_icon_ = TextPlaceholderIcon::kLink;
    }
  }

  return text_fields;
}

const gfx::VectorIcon& SharesheetHeaderView::GetTextVectorIcon() {
  switch (text_icon_) {
    case (TextPlaceholderIcon::kGenericText):
      return kSharesheetTextIcon;
    case (TextPlaceholderIcon::kLink):
      return kSharesheetLinkIcon;
  }
}

void SharesheetHeaderView::ResolveImages() {
  for (int i = 0; i < image_preview_->GetImageViewCount(); ++i) {
    ResolveImage(i);
  }
}

void SharesheetHeaderView::ResolveImage(size_t index) {
  const auto& file_path =
      GetFilePathFromFileSystemUrl(intent_->file_urls.value()[index]);

  const auto& size =
      GetImagePreviewSize(index, intent_->file_urls.value().size());
  auto image = std::make_unique<HoldingSpaceImage>(
      size, file_path,
      base::BindRepeating(&SharesheetHeaderView::LoadImage,
                          weak_ptr_factory_.GetWeakPtr()),
      // We pass our own icon in here because we want the icon to appear
      // while an image has not been loaded. If we didn't pass our own icon in,
      // the container is left blank while we wait for an image to load.
      absl::optional<gfx::ImageSkia>(CreateMimeTypeIcon(
          GetIconForPath(file_path, /* dark_background= */ false), size)));
  DCHECK_GT(image_preview_->GetImageViewCount(), index);
  image_preview_->GetImageViewAt(index)->SetImage(image->GetImageSkia(size));
  // TODO(crbug.com/2896003) Here and above, update this to check whether we're
  // in dark mode or not.
  const auto icon_color =
      GetIconColorForPath(file_path, /* dark_background= */ false);
  image_preview_->SetBackgroundColorForIndex(index, icon_color);
  image_subscription_.push_back(image->AddImageSkiaChangedCallback(
      base::BindRepeating(&SharesheetHeaderView::OnImageLoaded,
                          weak_ptr_factory_.GetWeakPtr(), size, index)));
  images_.push_back(std::move(image));
}

void SharesheetHeaderView::LoadImage(
    const base::FilePath& file_path,
    const gfx::Size& size,
    HoldingSpaceImage::BitmapCallback callback) {
  // This works for all shares right now because currently when we share data
  // that is not from the Files app (web share and ARC),
  // those files are being temporarily saved to disk before being shared.
  // If those implementations change, this will need to be updated.
  thumbnail_loader_.Load({file_path, size}, std::move(callback));
}

void SharesheetHeaderView::OnImageLoaded(const gfx::Size& size, size_t index) {
  DCHECK_GT(image_preview_->GetImageViewCount(), index);
  image_preview_->GetImageViewAt(index)->SetImage(
      images_[index]->GetImageSkia(size));
}

const base::FilePath SharesheetHeaderView::GetFilePathFromFileSystemUrl(
    const GURL& file_system_url) {
  storage::FileSystemContext* fs_context =
      file_manager::util::GetFileManagerFileSystemContext(profile_);
  storage::FileSystemURL fs_url = fs_context->CrackURL(file_system_url);
  return fs_url.path();
}

BEGIN_METADATA(SharesheetHeaderView, views::View)
END_METADATA

}  // namespace sharesheet
}  // namespace ash
