// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/global_media_controls/presentation_request_notification_producer.h"

#include "base/test/mock_callback.h"
#include "base/test/scoped_feature_list.h"
#include "chrome/browser/media/router/chrome_media_router_factory.h"
#include "chrome/browser/media/router/media_router_feature.h"
#include "chrome/browser/ui/global_media_controls/media_notification_service.h"
#include "chrome/browser/ui/global_media_controls/test_helper.h"
#include "chrome/test/base/chrome_render_view_host_test_harness.h"
#include "chrome/test/base/testing_profile.h"
#include "components/media_router/browser/presentation/start_presentation_context.h"
#include "components/media_router/browser/test/mock_media_router.h"
#include "content/public/test/browser_task_environment.h"
#include "content/public/test/mock_navigation_handle.h"
#include "content/public/test/navigation_simulator.h"
#include "testing/gmock/include/gmock/gmock.h"
#include "testing/gtest/include/gtest/gtest.h"

using testing::_;

namespace {
class MockWebContentsPresentationManager
    : public media_router::WebContentsPresentationManager {
 public:
  bool HasDefaultPresentationRequest() const override {
    return default_presentation_request_.has_value();
  }

  const content::PresentationRequest& GetDefaultPresentationRequest()
      const override {
    return *default_presentation_request_;
  }

  void SetDefaultPresentationRequest(
      const content::PresentationRequest& request) {
    default_presentation_request_ = request;
  }

  MOCK_METHOD1(
      AddObserver,
      void(media_router::WebContentsPresentationManager::Observer* observer));
  MOCK_METHOD1(
      RemoveObserver,
      void(media_router::WebContentsPresentationManager::Observer* observer));
  MOCK_METHOD3(OnPresentationResponse,
               void(const content::PresentationRequest&,
                    media_router::mojom::RoutePresentationConnectionPtr,
                    const media_router::RouteRequestResult&));
  MOCK_METHOD0(GetMediaRoutes, std::vector<media_router::MediaRoute>());

  base::WeakPtr<WebContentsPresentationManager> GetWeakPtr() override {
    return weak_factory_.GetWeakPtr();
  }

 private:
  absl::optional<content::PresentationRequest> default_presentation_request_;
  base::WeakPtrFactory<MockWebContentsPresentationManager> weak_factory_{this};
};

media_router::MediaRoute CreateMediaRoute(
    media_router::MediaRoute::Id route_id) {
  media_router::MediaRoute media_route(route_id,
                                       media_router::MediaSource("source_id"),
                                       "sink_id", "description", true, true);
  media_route.set_controller_type(media_router::RouteControllerType::kGeneric);
  return media_route;
}
}  // namespace

class PresentationRequestNotificationProducerTest
    : public ChromeRenderViewHostTestHarness {
 public:
  PresentationRequestNotificationProducerTest()
      : ChromeRenderViewHostTestHarness(
            base::test::TaskEnvironment::TimeSource::MOCK_TIME,
            base::test::TaskEnvironment::MainThreadType::UI) {}
  ~PresentationRequestNotificationProducerTest() override = default;

  void SetUp() override {
    feature_list_.InitAndEnableFeature(
        media_router::kGlobalMediaControlsCastStartStop);
    ChromeRenderViewHostTestHarness::SetUp();

    media_router::ChromeMediaRouterFactory::GetInstance()->SetTestingFactory(
        profile(), base::BindRepeating(&media_router::MockMediaRouter::Create));
    notification_service_ =
        std::make_unique<MediaNotificationService>(profile(), false);
    notification_producer_ =
        notification_service_->presentation_request_notification_producer_
            .get();

    presentation_manager_ =
        std::make_unique<MockWebContentsPresentationManager>();
    notification_producer_->SetTestPresentationManager(
        presentation_manager_->GetWeakPtr());
  }

  void TearDown() override {
    notification_service_.reset();
    media_router::WebContentsPresentationManager::SetTestInstance(nullptr);
    ChromeRenderViewHostTestHarness::TearDown();
  }

  void SimulateDialogOpenedAndWait(MockMediaDialogDelegate* delegate) {
    delegate->Open(notification_service_.get());
    task_environment()->RunUntilIdle();
  }

  void SimulateDialogClosedAndWait(MockMediaDialogDelegate* delegate) {
    delegate->Close();
    task_environment()->RunUntilIdle();
  }

  content::PresentationRequest CreatePresentationRequest() {
    return content::PresentationRequest(
        main_rfh()->GetGlobalFrameRoutingId(),
        {GURL("http://example.com"), GURL("http://example2.com")},
        url::Origin::Create(GURL("http://google.com")));
  }

  void SimulateStartPresentationContextCreated() {
    auto context = std::make_unique<media_router::StartPresentationContext>(
        CreatePresentationRequest(), base::DoNothing(), base::DoNothing());
    notification_producer_->OnStartPresentationContextCreated(
        std::move(context));
  }

  void SimulateMediaRouteChanged(
      const std::vector<media_router::MediaRoute>& routes) {
    notification_producer_->OnMediaRoutesChanged(routes);
  }

  content::RenderFrameHost* CreateChildFrame() {
    NavigateAndCommit(GURL("about:blank"));

    content::RenderFrameHost* child_frame = main_rfh();
    content::RenderFrameHostTester* rfh_tester =
        content::RenderFrameHostTester::For(child_frame);
    child_frame = rfh_tester->AppendChild("childframe");
    content::MockNavigationHandle handle(GURL(), child_frame);
    handle.set_has_committed(true);
    return child_frame;
  }

 protected:
  std::unique_ptr<MediaNotificationService> notification_service_;
  PresentationRequestNotificationProducer* notification_producer_ = nullptr;
  std::unique_ptr<MockWebContentsPresentationManager> presentation_manager_;
  base::test::ScopedFeatureList feature_list_;
};

TEST_F(PresentationRequestNotificationProducerTest,
       HideItemOnMediaRoutesChanged) {
  SimulateStartPresentationContextCreated();
  SimulateMediaRouteChanged({CreateMediaRoute("id")});
  EXPECT_FALSE(notification_service_->HasOpenDialog());
  task_environment()->RunUntilIdle();
}

TEST_F(PresentationRequestNotificationProducerTest, DismissNotification) {
  SimulateStartPresentationContextCreated();
  auto item = notification_producer_->GetNotificationItem();
  ASSERT_TRUE(item);

  notification_producer_->OnContainerDismissed(item->id());
  EXPECT_FALSE(notification_producer_->GetNotificationItem());
}

TEST_F(PresentationRequestNotificationProducerTest, OnMediaDialogOpened) {
  MockMediaDialogDelegate delegate;
  // Open the dialog on a page without a default presentation request.
  SimulateDialogOpenedAndWait(&delegate);
  EXPECT_FALSE(notification_producer_->GetNotificationItem());
  SimulateDialogClosedAndWait(&delegate);

  // Open the dialog on a page with default presentation request and there does
  // not exist a notification for non-default presentation request. A dummy
  // notification should be created.
  presentation_manager_->SetDefaultPresentationRequest(
      CreatePresentationRequest());
  SimulateDialogOpenedAndWait(&delegate);
  EXPECT_TRUE(notification_producer_->GetNotificationItem());
  SimulateDialogClosedAndWait(&delegate);
}

TEST_F(PresentationRequestNotificationProducerTest,
       OnMediaDialogOpenedWithExistingItem) {
  MockMediaDialogDelegate delegate;

  // Open the dialog on a page with default presentation request and there
  // exists a notification for non-default presentation request. The existing
  // notification should not be replaced.
  SimulateStartPresentationContextCreated();
  auto id = notification_producer_->GetNotificationItem()->id();
  presentation_manager_->SetDefaultPresentationRequest(
      CreatePresentationRequest());
  SimulateDialogOpenedAndWait(&delegate);
  EXPECT_TRUE(notification_producer_->GetNotificationItem());
  EXPECT_EQ(id, notification_producer_->GetNotificationItem()->id());
  SimulateDialogClosedAndWait(&delegate);
}

TEST_F(PresentationRequestNotificationProducerTest, DeleteItem) {
  content::RenderFrameHost* child_frame = CreateChildFrame();
  MockMediaDialogDelegate delegate;
  SimulateDialogOpenedAndWait(&delegate);
  // Simulate a PresentationRequest from |child_frame|.
  notification_producer_->OnStartPresentationContextCreated(
      std::make_unique<media_router::StartPresentationContext>(
          content::PresentationRequest(child_frame->GetGlobalFrameRoutingId(),
                                       {GURL(), GURL()},
                                       url::Origin::Create(GURL())),
          base::DoNothing(), base::DoNothing()));

  // Detach |child_frame|.
  content::RenderFrameHostTester::For(child_frame)->Detach();

  SimulateDialogClosedAndWait(&delegate);
}

TEST_F(PresentationRequestNotificationProducerTest,
       InvokeCallbackOnDialogClosed) {
  MockMediaDialogDelegate delegate;

  // PRNP should invoke |mock_error_cb| after the media dialog is closed.
  base::MockCallback<content::PresentationConnectionErrorCallback>
      mock_error_cb;
  EXPECT_CALL(mock_error_cb, Run);
  auto context = std::make_unique<media_router::StartPresentationContext>(
      CreatePresentationRequest(), base::DoNothing(), mock_error_cb.Get());
  notification_service_->OnStartPresentationContextCreated(std::move(context));
  SimulateDialogOpenedAndWait(&delegate);
  SimulateDialogClosedAndWait(&delegate);
}
