// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_CHROMEOS_INPUT_METHOD_MULTI_WORD_SUGGESTER_H_
#define CHROME_BROWSER_CHROMEOS_INPUT_METHOD_MULTI_WORD_SUGGESTER_H_

#include "chrome/browser/chromeos/input_method/suggester.h"
#include "chrome/browser/chromeos/input_method/suggestion_enums.h"
#include "chrome/browser/chromeos/input_method/suggestion_handler_interface.h"
#include "chromeos/services/ime/public/cpp/suggestions.h"

namespace chromeos {

// Integrates multi word suggestions produced by the system with the assistive
// framework. Handles showing / accepting / dismissing any multi word
// suggestions generated by the system.
class MultiWordSuggester : public Suggester {
 public:
  // `suggestion_handler` needs to exist longer than the lifetime of this
  // object.
  explicit MultiWordSuggester(SuggestionHandlerInterface* suggestion_handler);
  ~MultiWordSuggester() override;

  // Suggester overrides:
  void OnFocus(int context_id) override;
  void OnBlur() override;
  void OnExternalSuggestionsUpdated(
      const std::vector<ime::TextSuggestion>& suggestions) override;
  SuggestionStatus HandleKeyEvent(const ui::KeyEvent& event) override;
  bool Suggest(const std::u16string& text,
               size_t cursor_pos,
               size_t anchor_pos) override;
  bool AcceptSuggestion(size_t index = 0) override;
  void DismissSuggestion() override;
  AssistiveType GetProposeActionType() override;
  bool HasSuggestions() override;
  std::vector<ime::TextSuggestion> GetSuggestions() override;

  void OnSurroundingTextChanged(const std::u16string& text,
                                int cursor_pos,
                                int anchor_pos);

 private:
  void DisplaySuggestion(const ime::TextSuggestion& suggestion);

  // The currently focused input (zero if none are focused)
  int focused_context_id_ = 0;

  bool suggestion_shown_ = false;

  // The last known state of text in the focused text input
  std::u16string last_known_text_;
  int last_known_cursor_pos_ = 0;
  int last_known_anchor_pos_ = 0;

  // Not owned by this class
  SuggestionHandlerInterface* suggestion_handler_;
};

}  // namespace chromeos

#endif  // CHROME_BROWSER_CHROMEOS_INPUT_METHOD_MULTI_WORD_SUGGESTER_H_
