// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_CHROMEOS_INPUT_METHOD_MULTI_WORD_SUGGESTER_H_
#define CHROME_BROWSER_CHROMEOS_INPUT_METHOD_MULTI_WORD_SUGGESTER_H_

#include "chrome/browser/chromeos/input_method/suggester.h"
#include "chrome/browser/chromeos/input_method/suggestion_enums.h"
#include "chrome/browser/chromeos/input_method/suggestion_handler_interface.h"
#include "chromeos/services/ime/public/cpp/suggestions.h"
#include "third_party/abseil-cpp/absl/types/optional.h"

namespace chromeos {

struct LastKnownTextState {
  std::u16string text;
  bool cursor_at_end_of_text;
};

struct LastKnownSuggestionState {
  size_t start_pos;
  std::u16string text;
};

// Integrates multi word suggestions produced by the system with the assistive
// framework. Handles showing / accepting / dismissing any multi word
// suggestions generated by the system.
class MultiWordSuggester : public Suggester {
 public:
  // `suggestion_handler` needs to exist longer than the lifetime of this
  // object.
  explicit MultiWordSuggester(SuggestionHandlerInterface* suggestion_handler);
  ~MultiWordSuggester() override;

  // Suggester overrides:
  void OnFocus(int context_id) override;
  void OnBlur() override;
  void OnExternalSuggestionsUpdated(
      const std::vector<ime::TextSuggestion>& suggestions) override;
  SuggestionStatus HandleKeyEvent(const ui::KeyEvent& event) override;
  bool Suggest(const std::u16string& text,
               size_t cursor_pos,
               size_t anchor_pos) override;
  bool AcceptSuggestion(size_t index = 0) override;
  void DismissSuggestion() override;
  AssistiveType GetProposeActionType() override;
  bool HasSuggestions() override;
  std::vector<ime::TextSuggestion> GetSuggestions() override;

  void OnSurroundingTextChanged(const std::u16string& text,
                                size_t cursor_pos,
                                size_t anchor_pos);

 private:
  void DisplaySuggestion(const std::u16string& text, int confirmed_length);
  void ResetSuggestionState();
  void ResetTextState();

  // The currently focused input (zero if none are focused)
  int focused_context_id_ = 0;

  // Previous suggestion details
  absl::optional<LastKnownSuggestionState> suggestion_state_;

  // The last known state of text in the focused text input
  LastKnownTextState text_state_;

  // Not owned by this class
  SuggestionHandlerInterface* suggestion_handler_;
};

}  // namespace chromeos

#endif  // CHROME_BROWSER_CHROMEOS_INPUT_METHOD_MULTI_WORD_SUGGESTER_H_
