// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/chromeos/input_method/multi_word_suggester.h"

#include <vector>

#include "chrome/browser/chromeos/input_method/fake_suggestion_handler.h"
#include "chromeos/services/ime/public/cpp/suggestions.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "ui/events/base_event_utils.h"
#include "ui/events/keycodes/dom/dom_code.h"

namespace chromeos {
namespace {

using ::chromeos::ime::TextSuggestion;
using ::chromeos::ime::TextSuggestionMode;
using ::chromeos::ime::TextSuggestionType;

void SendKeyEvent(MultiWordSuggester* suggester, const ui::DomCode& code) {
  suggester->HandleKeyEvent(ui::KeyEvent(ui::ET_KEY_PRESSED, ui::VKEY_UNKNOWN,
                                         code, ui::EF_NONE, ui::DomKey::NONE,
                                         ui::EventTimeForNow()));
}

}  // namespace

TEST(MultiWordSuggesterTest, IgnoresIrrelevantExternalSuggestions) {
  FakeSuggestionHandler suggestion_handler;
  MultiWordSuggester suggester(&suggestion_handler);
  int focused_context_id = 5;

  std::vector<TextSuggestion> suggestions = {
      TextSuggestion{.mode = TextSuggestionMode::kPrediction,
                     .type = TextSuggestionType::kAssistivePersonalInfo,
                     .text = "my name is John Wayne"}};

  suggester.OnFocus(focused_context_id);
  suggester.OnExternalSuggestionsUpdated(suggestions);

  EXPECT_FALSE(suggestion_handler.GetShowingSuggestion());
  EXPECT_NE(suggestion_handler.GetContextId(), focused_context_id);
  EXPECT_EQ(suggestion_handler.GetSuggestionText(), u"");
}

TEST(MultiWordSuggesterTest, IgnoresEmpyExternalSuggestions) {
  FakeSuggestionHandler suggestion_handler;
  MultiWordSuggester suggester(&suggestion_handler);
  int focused_context_id = 5;

  suggester.OnFocus(focused_context_id);
  suggester.OnExternalSuggestionsUpdated({});

  EXPECT_FALSE(suggestion_handler.GetShowingSuggestion());
  EXPECT_NE(suggestion_handler.GetContextId(), focused_context_id);
  EXPECT_EQ(suggestion_handler.GetSuggestionText(), u"");
}

TEST(MultiWordSuggesterTest, DisplaysRelevantExternalSuggestions) {
  FakeSuggestionHandler suggestion_handler;
  MultiWordSuggester suggester(&suggestion_handler);
  int focused_context_id = 5;

  std::vector<TextSuggestion> suggestions = {
      TextSuggestion{.mode = TextSuggestionMode::kPrediction,
                     .type = TextSuggestionType::kMultiWord,
                     .text = "hello there!"}};

  suggester.OnFocus(focused_context_id);
  suggester.OnExternalSuggestionsUpdated(suggestions);

  EXPECT_TRUE(suggestion_handler.GetShowingSuggestion());
  EXPECT_EQ(suggestion_handler.GetContextId(), focused_context_id);
  EXPECT_EQ(suggestion_handler.GetSuggestionText(), u"hello there!");
}

TEST(MultiWordSuggesterTest, AcceptsSuggestionOnTabPress) {
  FakeSuggestionHandler suggestion_handler;
  MultiWordSuggester suggester(&suggestion_handler);
  int focused_context_id = 5;

  std::vector<TextSuggestion> suggestions = {
      TextSuggestion{.mode = TextSuggestionMode::kPrediction,
                     .type = TextSuggestionType::kMultiWord,
                     .text = "hi there!"},
  };

  suggester.OnFocus(focused_context_id);
  suggester.OnExternalSuggestionsUpdated(suggestions);
  SendKeyEvent(&suggester, ui::DomCode::TAB);

  EXPECT_FALSE(suggestion_handler.GetShowingSuggestion());
  EXPECT_FALSE(suggestion_handler.GetDismissedSuggestion());
  EXPECT_TRUE(suggestion_handler.GetAcceptedSuggestion());
  EXPECT_EQ(suggestion_handler.GetSuggestionText(), u"");
}

TEST(MultiWordSuggesterTest, DoesNotAcceptSuggestionOnNonTabKeypress) {
  FakeSuggestionHandler suggestion_handler;
  MultiWordSuggester suggester(&suggestion_handler);
  int focused_context_id = 5;

  std::vector<TextSuggestion> suggestions = {
      TextSuggestion{.mode = TextSuggestionMode::kPrediction,
                     .type = TextSuggestionType::kMultiWord,
                     .text = "hi there!"},
  };

  suggester.OnFocus(focused_context_id);
  suggester.OnExternalSuggestionsUpdated(suggestions);
  SendKeyEvent(&suggester, ui::DomCode::ARROW_DOWN);

  EXPECT_TRUE(suggestion_handler.GetShowingSuggestion());
  EXPECT_FALSE(suggestion_handler.GetAcceptedSuggestion());
  EXPECT_EQ(suggestion_handler.GetSuggestionText(), u"hi there!");
}

TEST(MultiWordSuggesterTest, CalculatesConfirmedLengthForOneWord) {
  FakeSuggestionHandler suggestion_handler;
  MultiWordSuggester suggester(&suggestion_handler);
  int focused_context_id = 5;

  std::vector<TextSuggestion> suggestions = {
      TextSuggestion{.mode = TextSuggestionMode::kCompletion,
                     .type = TextSuggestionType::kMultiWord,
                     .text = "how are you going"},
  };

  suggester.OnFocus(focused_context_id);
  suggester.OnSurroundingTextChanged(u"ho", /*cursor_pos=*/2, /*anchor_pos=*/2);
  suggester.OnExternalSuggestionsUpdated(suggestions);

  EXPECT_TRUE(suggestion_handler.GetShowingSuggestion());
  EXPECT_EQ(suggestion_handler.GetSuggestionText(), u"how are you going");
  EXPECT_EQ(suggestion_handler.GetConfirmedLength(), 2);  // ho
}

TEST(MultiWordSuggesterTest, CalculatesConfirmedLengthForManyWords) {
  FakeSuggestionHandler suggestion_handler;
  MultiWordSuggester suggester(&suggestion_handler);
  int focused_context_id = 5;

  std::vector<TextSuggestion> suggestions = {
      TextSuggestion{.mode = TextSuggestionMode::kCompletion,
                     .type = TextSuggestionType::kMultiWord,
                     .text = "where are you going"},
  };

  suggester.OnFocus(focused_context_id);
  suggester.OnSurroundingTextChanged(u"hey there sam whe",
                                     /*cursor_pos=*/17, /*anchor_pos=*/17);
  suggester.OnExternalSuggestionsUpdated(suggestions);

  EXPECT_TRUE(suggestion_handler.GetShowingSuggestion());
  EXPECT_EQ(suggestion_handler.GetSuggestionText(), u"where are you going");
  EXPECT_EQ(suggestion_handler.GetConfirmedLength(), 3);  // whe
}

TEST(MultiWordSuggesterTest, TracksLastSuggestionOnSurroundingTextChange) {
  FakeSuggestionHandler suggestion_handler;
  MultiWordSuggester suggester(&suggestion_handler);
  int focused_context_id = 5;

  std::vector<TextSuggestion> suggestions = {
      TextSuggestion{.mode = TextSuggestionMode::kCompletion,
                     .type = TextSuggestionType::kMultiWord,
                     .text = "where are you going"},
  };

  suggester.OnFocus(focused_context_id);
  suggester.OnSurroundingTextChanged(u"hey there sam whe", 17, 17);
  suggester.OnExternalSuggestionsUpdated(suggestions);
  suggester.OnSurroundingTextChanged(u"hey there sam wher", 18, 18);
  suggester.Suggest(u"hey there sam wher", 18, 18);
  suggester.OnSurroundingTextChanged(u"hey there sam where", 19, 19);
  suggester.Suggest(u"hey there sam where", 19, 19);
  suggester.OnSurroundingTextChanged(u"hey there sam where ", 20, 20);
  suggester.Suggest(u"hey there sam where ", 20, 20);
  suggester.OnSurroundingTextChanged(u"hey there sam where a", 21, 21);
  suggester.Suggest(u"hey there sam where a", 21, 21);
  suggester.OnSurroundingTextChanged(u"hey there sam where ar", 22, 22);
  suggester.Suggest(u"hey there sam where ar", 22, 22);
  suggester.OnSurroundingTextChanged(u"hey there sam where are", 23, 23);
  suggester.Suggest(u"hey there sam where are", 23, 23);

  EXPECT_TRUE(suggestion_handler.GetShowingSuggestion());
  EXPECT_EQ(suggestion_handler.GetSuggestionText(), u"where are you going");
  EXPECT_EQ(suggestion_handler.GetConfirmedLength(), 9);  // where are
}

TEST(MultiWordSuggesterTest,
     TracksLastSuggestionOnSurroundingTextChangeAtBeginningText) {
  FakeSuggestionHandler suggestion_handler;
  MultiWordSuggester suggester(&suggestion_handler);
  int focused_context_id = 5;

  std::vector<TextSuggestion> suggestions = {
      TextSuggestion{.mode = TextSuggestionMode::kCompletion,
                     .type = TextSuggestionType::kMultiWord,
                     .text = "how are you"},
  };

  suggester.OnFocus(focused_context_id);
  suggester.OnSurroundingTextChanged(u"h", 1, 1);
  suggester.OnExternalSuggestionsUpdated(suggestions);
  suggester.OnSurroundingTextChanged(u"ho", 2, 2);
  suggester.Suggest(u"ho", 2, 2);
  suggester.OnSurroundingTextChanged(u"how", 3, 3);
  suggester.Suggest(u"how", 3, 3);

  EXPECT_TRUE(suggestion_handler.GetShowingSuggestion());
  EXPECT_EQ(suggestion_handler.GetSuggestionText(), u"how are you");
  EXPECT_EQ(suggestion_handler.GetConfirmedLength(), 3);  // how
}

TEST(MultiWordSuggesterTest, TracksLastSuggestionOnLargeSurroundingTextChange) {
  FakeSuggestionHandler suggestion_handler;
  MultiWordSuggester suggester(&suggestion_handler);
  int focused_context_id = 5;

  std::vector<TextSuggestion> suggestions = {
      TextSuggestion{.mode = TextSuggestionMode::kCompletion,
                     .type = TextSuggestionType::kMultiWord,
                     .text = "how are you"},
  };

  suggester.OnFocus(focused_context_id);
  suggester.OnSurroundingTextChanged(u"h", 1, 1);
  suggester.OnExternalSuggestionsUpdated(suggestions);
  suggester.OnSurroundingTextChanged(u"how ar", 6, 6);
  suggester.Suggest(u"how ar", 6, 6);
  suggester.OnSurroundingTextChanged(u"how are yo", 10, 10);
  suggester.Suggest(u"how are yo", 10, 10);

  EXPECT_TRUE(suggestion_handler.GetShowingSuggestion());
  EXPECT_EQ(suggestion_handler.GetSuggestionText(), u"how are you");
  EXPECT_EQ(suggestion_handler.GetConfirmedLength(), 10);  // how are yo
}

TEST(MultiWordSuggesterTest,
     DoesNotTrackLastSuggestionIfSurroundingTextChange) {
  FakeSuggestionHandler suggestion_handler;
  MultiWordSuggester suggester(&suggestion_handler);
  int focused_context_id = 5;

  std::vector<TextSuggestion> suggestions = {
      TextSuggestion{.mode = TextSuggestionMode::kCompletion,
                     .type = TextSuggestionType::kMultiWord,
                     .text = "how are you"},
  };

  suggester.OnFocus(focused_context_id);
  suggester.OnSurroundingTextChanged(u"h", 1, 1);
  suggester.OnExternalSuggestionsUpdated(suggestions);
  suggester.OnSurroundingTextChanged(u"how ar", 6, 6);
  suggester.Suggest(u"how ar", 6, 6);
  suggester.OnSurroundingTextChanged(u"how yo", 6, 6);

  // The consumer will handle dismissing the suggestion
  EXPECT_FALSE(suggester.Suggest(u"how yo", 6, 6));
}

}  // namespace chromeos
