// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
//
// This file contains utility functions for lacros_chrome_browsertests.

#ifndef CHROME_BROWSER_LACROS_BROWSER_TEST_UTIL_H_
#define CHROME_BROWSER_LACROS_BROWSER_TEST_UTIL_H_

#include <string>

namespace aura {
class Window;
}  // namespace aura

namespace browser_test_util {

// Returns the id associated with a root window. This id is generated by lacros
// and shared with ash.
std::string GetWindowId(aura::Window* window);

// Some crosapi methods rely on the assumption that ash/exo are aware of the
// existence of a Lacros Window. Wayland is an async protocol that uses a
// different communication channel than crosapi. This method provides a
// synchronization mechanism for window creation and destruction.
//
// Waits for the Window to be created. |id| comes from |GetWindowId|.
void WaitForWindowCreation(const std::string& id);

// Waits for the window to be destroyed. |id| comes from |GetWindowId|.
void WaitForWindowDestruction(const std::string& id);

// This function relies on |window| already being available in ash. It prompts
// ash to send the Wayland events associated with a mouse click to the |window|.
// It waits for the events to be seen by Aura.
//
// This function is necessary to prime the Wayland serial_id. The serial_id is
// generated by the Wayland server and sent to Wayland client along-side user
// events. This serial_id is a required input to Wayland clipboard APIs.
//
// |window| must be a root window.
void SendAndWaitForMouseClick(aura::Window* window);

}  // namespace browser_test_util

#endif  // CHROME_BROWSER_LACROS_BROWSER_TEST_UTIL_H_
