// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/sync/trusted_vault_client_android.h"

#include <utility>

#include "base/android/jni_android.h"
#include "base/check_op.h"
#include "base/notreached.h"
#include "chrome/android/chrome_jni_headers/TrustedVaultClient_jni.h"
#include "content/public/browser/browser_thread.h"

TrustedVaultClientAndroid::OngoingFetchKeys::OngoingFetchKeys(
    const CoreAccountInfo& account_info,
    base::OnceCallback<void(const std::vector<std::vector<uint8_t>>&)> callback)
    : account_info(account_info), callback(std::move(callback)) {}

TrustedVaultClientAndroid::OngoingFetchKeys::OngoingFetchKeys(
    OngoingFetchKeys&&) = default;

TrustedVaultClientAndroid::OngoingFetchKeys::~OngoingFetchKeys() = default;

TrustedVaultClientAndroid::OngoingMarkKeysAsStale::OngoingMarkKeysAsStale(
    base::OnceCallback<void(bool)> callback)
    : callback(std::move(callback)) {}

TrustedVaultClientAndroid::OngoingMarkKeysAsStale::OngoingMarkKeysAsStale(
    OngoingMarkKeysAsStale&&) = default;

TrustedVaultClientAndroid::OngoingMarkKeysAsStale::~OngoingMarkKeysAsStale() =
    default;

TrustedVaultClientAndroid::OngoingGetIsRecoverabilityDegraded::
    OngoingGetIsRecoverabilityDegraded(base::OnceCallback<void(bool)> callback)
    : callback(std::move(callback)) {}

TrustedVaultClientAndroid::OngoingGetIsRecoverabilityDegraded::
    OngoingGetIsRecoverabilityDegraded(OngoingGetIsRecoverabilityDegraded&&) =
        default;

TrustedVaultClientAndroid::OngoingGetIsRecoverabilityDegraded::
    ~OngoingGetIsRecoverabilityDegraded() = default;

TrustedVaultClientAndroid::TrustedVaultClientAndroid() {
  JNIEnv* const env = base::android::AttachCurrentThread();
  Java_TrustedVaultClient_registerNative(env, reinterpret_cast<intptr_t>(this));
}

TrustedVaultClientAndroid::~TrustedVaultClientAndroid() {
  JNIEnv* const env = base::android::AttachCurrentThread();
  Java_TrustedVaultClient_unregisterNative(env,
                                           reinterpret_cast<intptr_t>(this));
}

void TrustedVaultClientAndroid::FetchKeysCompleted(
    JNIEnv* env,
    jint request_id,
    const base::android::JavaParamRef<jstring>& gaia_id,
    const base::android::JavaParamRef<jobjectArray>& keys) {
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);

  OngoingRequest ongoing_request = GetAndUnregisterOngoingRequest(request_id);
  OngoingFetchKeys& ongoing_fetch_keys =
      absl::get<OngoingFetchKeys>(ongoing_request);

  DCHECK_EQ(ongoing_fetch_keys.account_info.gaia,
            base::android::ConvertJavaStringToUTF8(env, gaia_id))
      << "User mismatch in FetchKeys() response";

  std::vector<std::vector<uint8_t>> converted_keys;
  JavaArrayOfByteArrayToBytesVector(env, keys, &converted_keys);
  std::move(ongoing_fetch_keys.callback).Run(converted_keys);
}

void TrustedVaultClientAndroid::MarkKeysAsStaleCompleted(JNIEnv* env,
                                                         jint request_id,
                                                         jboolean result) {
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);

  OngoingRequest ongoing_request = GetAndUnregisterOngoingRequest(request_id);

  std::move(absl::get<OngoingMarkKeysAsStale>(ongoing_request).callback)
      .Run(!!result);
}

void TrustedVaultClientAndroid::GetIsRecoverabilityDegradedCompleted(
    JNIEnv* env,
    jint request_id,
    jboolean result) {
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);

  OngoingRequest ongoing_request = GetAndUnregisterOngoingRequest(request_id);

  std::move(
      absl::get<OngoingGetIsRecoverabilityDegraded>(ongoing_request).callback)
      .Run(!!result);
}

void TrustedVaultClientAndroid::NotifyKeysChanged(JNIEnv* env) {
  for (Observer& observer : observer_list_) {
    observer.OnTrustedVaultKeysChanged();
  }
}

void TrustedVaultClientAndroid::AddObserver(Observer* observer) {
  observer_list_.AddObserver(observer);
}

void TrustedVaultClientAndroid::RemoveObserver(Observer* observer) {
  observer_list_.RemoveObserver(observer);
}

void TrustedVaultClientAndroid::FetchKeys(
    const CoreAccountInfo& account_info,
    base::OnceCallback<void(const std::vector<std::vector<uint8_t>>&)> cb) {
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);

  // Store for later completion when Java invokes FetchKeysCompleted().
  const RequestId request_id =
      RegisterNewOngoingRequest(OngoingFetchKeys(account_info, std::move(cb)));

  JNIEnv* const env = base::android::AttachCurrentThread();
  const base::android::ScopedJavaLocalRef<jobject> java_account_info =
      ConvertToJavaCoreAccountInfo(env, account_info);

  // Trigger the fetching keys from the implementation in Java, which will
  // eventually call FetchKeysCompleted().
  Java_TrustedVaultClient_fetchKeys(env, reinterpret_cast<intptr_t>(this),
                                    request_id, java_account_info);
}

void TrustedVaultClientAndroid::StoreKeys(
    const std::string& gaia_id,
    const std::vector<std::vector<uint8_t>>& keys,
    int last_key_version) {
  // Not supported on Android, where keys are fetched outside the browser.
  NOTREACHED();
}

void TrustedVaultClientAndroid::MarkKeysAsStale(
    const CoreAccountInfo& account_info,
    base::OnceCallback<void(bool)> cb) {
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);
  DCHECK(cb);

  // Store for later completion when Java invokes MarkKeysAsStaleCompleted().
  const RequestId request_id =
      RegisterNewOngoingRequest(OngoingMarkKeysAsStale(std::move(cb)));

  JNIEnv* const env = base::android::AttachCurrentThread();
  const base::android::ScopedJavaLocalRef<jobject> java_account_info =
      ConvertToJavaCoreAccountInfo(env, account_info);

  // The Java implementation will eventually call MarkKeysAsStaleCompleted().
  Java_TrustedVaultClient_markKeysAsStale(env, reinterpret_cast<intptr_t>(this),
                                          request_id, java_account_info);
}

void TrustedVaultClientAndroid::GetIsRecoverabilityDegraded(
    const CoreAccountInfo& account_info,
    base::OnceCallback<void(bool)> cb) {
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);
  DCHECK(cb);

  // Store for later completion when Java invokes
  // GetIsRecoverabilityDegradedCompleted().
  const RequestId request_id = RegisterNewOngoingRequest(
      OngoingGetIsRecoverabilityDegraded(std::move(cb)));

  JNIEnv* const env = base::android::AttachCurrentThread();
  const base::android::ScopedJavaLocalRef<jobject> java_account_info =
      ConvertToJavaCoreAccountInfo(env, account_info);

  // The Java implementation will eventually call MarkKeysAsStaleCompleted().
  Java_TrustedVaultClient_getIsRecoverabilityDegraded(
      env, reinterpret_cast<intptr_t>(this), request_id, java_account_info);
}

void TrustedVaultClientAndroid::AddTrustedRecoveryMethod(
    const std::string& gaia_id,
    const std::vector<uint8_t>& public_key,
    int method_type_hint,
    base::OnceClosure cb) {
  // TODO(crbug.com/1100279): Needs implementation.
  NOTIMPLEMENTED();
  std::move(cb).Run();
}

TrustedVaultClientAndroid::RequestId
TrustedVaultClientAndroid::RegisterNewOngoingRequest(OngoingRequest request) {
  ongoing_requests_.emplace(++last_request_id_, std::move(request));
  return last_request_id_;
}

TrustedVaultClientAndroid::OngoingRequest
TrustedVaultClientAndroid::GetAndUnregisterOngoingRequest(RequestId id) {
  DCHECK_CURRENTLY_ON(content::BrowserThread::UI);

  auto it = ongoing_requests_.find(id);
  DCHECK(it != ongoing_requests_.end());

  OngoingRequest request = std::move(it->second);
  ongoing_requests_.erase(it);
  return request;
}
