// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_CHROMEOS_POLICY_INVALIDATION_AFFILIATED_INVALIDATION_SERVICE_PROVIDER_H_
#define CHROME_BROWSER_CHROMEOS_POLICY_INVALIDATION_AFFILIATED_INVALIDATION_SERVICE_PROVIDER_H_

#include "base/macros.h"

namespace invalidation {
class InvalidationService;
}

namespace policy {

// This class provides access to an |InvalidationService| that can be used to
// subscribe to invalidations generated by the device's enrollment domain, e.g.
// policy pushing and remote commands for:
// * the device itself
// * device-local accounts
// * other users affiliated with the enrollment domain
//
// If an affiliated user with a connected invalidation service is logged in,
// that invalidation service will be used to conserve server resources. If there
// are no logged-in users matching these criteria, a device-global
// |TiclInvalidationService| is spun up.
// The class monitors the status of the invalidation services and switches
// between them whenever the service currently in use disconnects or the
// device-global invalidation service can be replaced with another service that
// just connected.
class AffiliatedInvalidationServiceProvider {
 public:
  class Consumer {
   public:
    Consumer();

    // This method is called when the invalidation service that the consumer
    // should use changes:
    // * If |invalidation_service| is a nullptr, no invalidation service is
    //   currently available for use.
    // * Otherwise, |invalidation_service| is the invalidation service that the
    //   consumer should use. It is guaranteed to be connected. Any previously
    //   provided invalidation services must no longer be used.
    virtual void OnInvalidationServiceSet(
        invalidation::InvalidationService* invalidation_service) = 0;

   protected:
    virtual ~Consumer();

    DISALLOW_ASSIGN(Consumer);
  };

  AffiliatedInvalidationServiceProvider();
  virtual ~AffiliatedInvalidationServiceProvider();

  // Indicates that |consumer| is interested in using the shared
  // |InvalidationService|. The consumer's OnInvalidationServiceSet() method
  // will be called back when a connected invalidation service becomes
  // available. If an invalidation service is available already, the callback
  // will occur synchronously. The |consumer| must be unregistered before |this|
  // is destroyed.
  virtual void RegisterConsumer(Consumer* consumer) = 0;

  // Indicates that |consumer| is no longer interested in using the
  // shared |InvalidationService|.
  virtual void UnregisterConsumer(Consumer* consumer) = 0;

  // Shuts down the provider. Once the provider is shut down, it no longer makes
  // any invalidation service available to consumers, no longer observes any
  // per-profile invalidation services and no longer maintains a device-global
  // invalidation service.
  virtual void Shutdown() = 0;

 private:
  DISALLOW_ASSIGN(AffiliatedInvalidationServiceProvider);
};

}  // namespace policy

#endif  // CHROME_BROWSER_CHROMEOS_POLICY_INVALIDATION_AFFILIATED_INVALIDATION_SERVICE_PROVIDER_H_
