// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/ash/holding_space/holding_space_browsertest_base.h"

#include <set>
#include <unordered_map>
#include <vector>

#include "ash/constants/ash_features.h"
#include "ash/public/cpp/capture_mode_test_api.h"
#include "ash/public/cpp/holding_space/holding_space_client.h"
#include "ash/public/cpp/holding_space/holding_space_constants.h"
#include "ash/public/cpp/holding_space/holding_space_controller.h"
#include "ash/public/cpp/holding_space/holding_space_item.h"
#include "ash/public/cpp/holding_space/holding_space_metrics.h"
#include "ash/public/cpp/holding_space/holding_space_model.h"
#include "ash/public/cpp/holding_space/holding_space_model_observer.h"
#include "ash/public/cpp/holding_space/holding_space_prefs.h"
#include "ash/public/cpp/holding_space/holding_space_test_api.h"
#include "ash/public/cpp/holding_space/mock_holding_space_client.h"
#include "ash/public/cpp/holding_space/mock_holding_space_model_observer.h"
#include "base/callback_helpers.h"
#include "base/containers/contains.h"
#include "base/files/file_util.h"
#include "base/scoped_observation.h"
#include "base/test/bind.h"
#include "base/test/scoped_locale.h"
#include "chrome/browser/ash/profiles/profile_helper.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/profiles/profile_manager.h"
#include "chrome/browser/ui/ash/holding_space/holding_space_downloads_delegate.h"
#include "chrome/browser/ui/ash/holding_space/holding_space_util.h"
#include "chrome/browser/ui/browser_window.h"
#include "components/download/public/common/mock_download_item.h"
#include "content/public/test/browser_test.h"
#include "content/public/test/mock_download_manager.h"
#include "testing/gmock/include/gmock/gmock.h"
#include "ui/aura/window.h"
#include "ui/base/clipboard/custom_data_helper.h"
#include "ui/base/dragdrop/drag_drop_types.h"
#include "ui/base/dragdrop/mojom/drag_drop_types.mojom.h"
#include "ui/base/dragdrop/os_exchange_data_provider_factory.h"
#include "ui/compositor/layer.h"
#include "ui/compositor/scoped_animation_duration_scale_mode.h"
#include "ui/events/base_event_utils.h"
#include "ui/events/test/event_generator.h"
#include "ui/gfx/image/image_unittest_util.h"
#include "ui/views/controls/label.h"
#include "ui/views/controls/menu/menu_controller.h"
#include "ui/views/controls/menu/menu_item_view.h"
#include "ui/views/drag_controller.h"
#include "ui/views/view.h"
#include "ui/views/widget/widget.h"
#include "ui/views/widget/widget_delegate.h"
#include "ui/wm/public/activation_change_observer.h"
#include "ui/wm/public/activation_client.h"

namespace ash {

namespace {

// Helpers ---------------------------------------------------------------------

// Returns all holding space item types.
std::vector<HoldingSpaceItem::Type> GetHoldingSpaceItemTypes() {
  std::vector<HoldingSpaceItem::Type> types;
  for (int i = 0; i <= static_cast<int>(HoldingSpaceItem::Type::kMaxValue); ++i)
    types.push_back(static_cast<HoldingSpaceItem::Type>(i));
  return types;
}

// Flushes the message loop by posting a task and waiting for it to run.
void FlushMessageLoop() {
  base::RunLoop run_loop;
  base::SequencedTaskRunnerHandle::Get()->PostTask(FROM_HERE,
                                                   run_loop.QuitClosure());
  run_loop.Run();
}

// Performs a click on `view` with optional `flags`.
void Click(const views::View* view, int flags = ui::EF_NONE) {
  auto* root_window = HoldingSpaceBrowserTestBase::GetRootWindowForNewWindows();
  ui::test::EventGenerator event_generator(root_window);
  event_generator.set_flags(flags);
  event_generator.MoveMouseTo(view->GetBoundsInScreen().CenterPoint());
  event_generator.ClickLeftButton();
}

// Performs a double click on `view`.
void DoubleClick(const views::View* view) {
  auto* root_window = HoldingSpaceBrowserTestBase::GetRootWindowForNewWindows();
  ui::test::EventGenerator event_generator(root_window);
  event_generator.MoveMouseTo(view->GetBoundsInScreen().CenterPoint());
  event_generator.DoubleClickLeftButton();
}

using DragUpdateCallback =
    base::RepeatingCallback<void(const gfx::Point& screen_location)>;

// Performs a gesture drag between `from` and `to`.
void GestureDrag(const views::View* from,
                 const views::View* to,
                 DragUpdateCallback drag_update_callback = base::DoNothing(),
                 base::OnceClosure before_release_callback = base::DoNothing(),
                 base::OnceClosure after_release_callback = base::DoNothing()) {
  auto* root_window = HoldingSpaceBrowserTestBase::GetRootWindowForNewWindows();
  ui::test::EventGenerator event_generator(root_window);
  event_generator.PressTouch(from->GetBoundsInScreen().CenterPoint());

  // Gesture drag is initiated only after an `ui::ET_GESTURE_LONG_PRESS` event.
  ui::GestureEvent long_press(
      event_generator.current_screen_location().x(),
      event_generator.current_screen_location().y(), ui::EF_NONE,
      ui::EventTimeForNow(),
      ui::GestureEventDetails(ui::ET_GESTURE_LONG_PRESS));
  event_generator.Dispatch(&long_press);

  // Generate multiple interpolated touch move events.
  // NOTE: The `ash::DragDropController` applies a vertical offset when
  // determining the target view for touch initiated dragging so that needs to
  // be compensated for here.
  constexpr int kNumberOfTouchMoveEvents = 25;
  constexpr gfx::Vector2d offset(0, 25);
  const gfx::Point endpoint = to->GetBoundsInScreen().CenterPoint() + offset;
  const gfx::Point origin = event_generator.current_screen_location();
  const gfx::Vector2dF diff(endpoint - origin);
  for (int i = 1; i <= kNumberOfTouchMoveEvents; ++i) {
    gfx::Vector2dF step(diff);
    step.Scale(i / static_cast<float>(kNumberOfTouchMoveEvents));
    event_generator.MoveTouch(origin + gfx::ToRoundedVector2d(step));
    drag_update_callback.Run(event_generator.current_screen_location());
  }

  std::move(before_release_callback).Run();
  event_generator.ReleaseTouch();
  std::move(after_release_callback).Run();
}

// Performs a gesture tap on `view`.
void GestureTap(const views::View* view) {
  auto* root_window = HoldingSpaceBrowserTestBase::GetRootWindowForNewWindows();
  ui::test::EventGenerator event_generator(root_window);
  event_generator.GestureTapAt(view->GetBoundsInScreen().CenterPoint());
}

// Performs a mouse drag between `from` and `to`.
void MouseDrag(const views::View* from,
               const views::View* to,
               DragUpdateCallback drag_update_callback = base::DoNothing(),
               base::OnceClosure before_release_callback = base::DoNothing(),
               base::OnceClosure after_release_callback = base::DoNothing()) {
  auto* root_window = HoldingSpaceBrowserTestBase::GetRootWindowForNewWindows();
  ui::test::EventGenerator event_generator(root_window);
  event_generator.MoveMouseTo(from->GetBoundsInScreen().CenterPoint());
  event_generator.PressLeftButton();

  // Generate multiple interpolated mouse move events so that views are notified
  // of mouse enter/exit as they would be in production.
  constexpr int kNumberOfMouseMoveEvents = 25;
  const gfx::Point origin = event_generator.current_screen_location();
  const gfx::Vector2dF diff(to->GetBoundsInScreen().CenterPoint() - origin);
  for (int i = 1; i <= kNumberOfMouseMoveEvents; ++i) {
    gfx::Vector2dF step(diff);
    step.Scale(i / static_cast<float>(kNumberOfMouseMoveEvents));
    event_generator.MoveMouseTo(origin + gfx::ToRoundedVector2d(step));
    drag_update_callback.Run(event_generator.current_screen_location());
  }

  std::move(before_release_callback).Run();
  event_generator.ReleaseLeftButton();
  std::move(after_release_callback).Run();
}

// Moves mouse to `view` over `count` number of events.
void MoveMouseTo(const views::View* view, size_t count = 1u) {
  auto* root_window = HoldingSpaceBrowserTestBase::GetRootWindowForNewWindows();
  ui::test::EventGenerator event_generator(root_window);
  event_generator.MoveMouseTo(view->GetBoundsInScreen().CenterPoint(), count);
}

// Performs a press and release of the specified `key_code` with `flags`.
void PressAndReleaseKey(ui::KeyboardCode key_code, int flags = ui::EF_NONE) {
  auto* root_window = HoldingSpaceBrowserTestBase::GetRootWindowForNewWindows();
  ui::test::EventGenerator event_generator(root_window);
  event_generator.PressKey(key_code, flags);
  event_generator.ReleaseKey(key_code, flags);
}

// Performs a right click on `view` with the specified `flags`.
void RightClick(const views::View* view, int flags = ui::EF_NONE) {
  auto* root_window = view->GetWidget()->GetNativeWindow()->GetRootWindow();
  ui::test::EventGenerator event_generator(root_window);
  event_generator.MoveMouseTo(view->GetBoundsInScreen().CenterPoint());
  event_generator.set_flags(flags);
  event_generator.ClickRightButton();
}

// Selects the menu item with the specified command ID. Returns the selected
// menu item if successful, `nullptr` otherwise.
views::MenuItemView* SelectMenuItemWithCommandId(
    HoldingSpaceCommandId command_id) {
  auto* menu_controller = views::MenuController::GetActiveInstance();
  if (!menu_controller)
    return nullptr;

  PressAndReleaseKey(ui::KeyboardCode::VKEY_DOWN);
  auto* const first_selected_menu_item = menu_controller->GetSelectedMenuItem();
  if (!first_selected_menu_item)
    return nullptr;

  auto* selected_menu_item = first_selected_menu_item;
  do {
    if (selected_menu_item->GetCommand() == static_cast<int>(command_id))
      return selected_menu_item;

    PressAndReleaseKey(ui::KeyboardCode::VKEY_DOWN);
    selected_menu_item = menu_controller->GetSelectedMenuItem();

    // It is expected that context menus loop selection traversal. If the
    // currently `selected_menu_item` is the `first_selected_menu_item` then the
    // context menu has been completely traversed.
  } while (selected_menu_item != first_selected_menu_item);

  // If this LOC is reached the menu has been completely traversed without
  // finding a menu item for the desired `command_id`.
  return nullptr;
}

// Mocks -----------------------------------------------------------------------

class MockActivationChangeObserver : public wm::ActivationChangeObserver {
 public:
  MOCK_METHOD(void,
              OnWindowActivated,
              (wm::ActivationChangeObserver::ActivationReason reason,
               aura::Window* gained_active,
               aura::Window* lost_active),
              (override));
};

// DropSenderView --------------------------------------------------------------

class DropSenderView : public views::WidgetDelegateView,
                       public views::DragController {
 public:
  DropSenderView(const DropSenderView&) = delete;
  DropSenderView& operator=(const DropSenderView&) = delete;
  ~DropSenderView() override = default;

  static DropSenderView* Create(aura::Window* context) {
    return new DropSenderView(context);
  }

  void ClearFilenamesData() { filenames_data_.reset(); }

  void SetFilenamesData(const std::vector<base::FilePath> file_paths) {
    std::vector<ui::FileInfo> filenames;
    for (const base::FilePath& file_path : file_paths)
      filenames.emplace_back(file_path, /*display_name=*/base::FilePath());
    filenames_data_.emplace(std::move(filenames));
  }

  void ClearFileSystemSourcesData() { file_system_sources_data_.reset(); }

  void SetFileSystemSourcesData(const std::vector<GURL>& file_system_urls) {
    constexpr char16_t kFileSystemSourcesType[] = u"fs/sources";

    std::stringstream file_system_sources;
    for (const GURL& file_system_url : file_system_urls)
      file_system_sources << file_system_url.spec() << "\n";

    base::Pickle pickle;
    ui::WriteCustomDataToPickle(
        std::unordered_map<std::u16string, std::u16string>(
            {{kFileSystemSourcesType,
              base::UTF8ToUTF16(file_system_sources.str())}}),
        &pickle);

    file_system_sources_data_.emplace(std::move(pickle));
  }

 private:
  explicit DropSenderView(aura::Window* context) {
    InitWidget(context);
    set_drag_controller(this);
  }

  // views::DragController:
  bool CanStartDragForView(views::View* sender,
                           const gfx::Point& press_pt,
                           const gfx::Point& current_pt) override {
    DCHECK_EQ(sender, this);
    return true;
  }

  int GetDragOperationsForView(views::View* sender,
                               const gfx::Point& press_pt) override {
    DCHECK_EQ(sender, this);
    return ui::DragDropTypes::DRAG_COPY;
  }

  void WriteDragDataForView(views::View* sender,
                            const gfx::Point& press_pt,
                            ui::OSExchangeData* data) override {
    // Drag image.
    // NOTE: Gesture drag is only enabled if a drag image is specified.
    data->provider().SetDragImage(
        /*image=*/gfx::test::CreateImageSkia(/*width=*/10, /*height=*/10),
        /*cursor_offset=*/gfx::Vector2d());

    // Payload.
    if (filenames_data_)
      data->provider().SetFilenames(filenames_data_.value());
    if (file_system_sources_data_) {
      data->provider().SetPickledData(
          ui::ClipboardFormatType::GetWebCustomDataType(),
          file_system_sources_data_.value());
    }
  }

  void InitWidget(aura::Window* context) {
    views::Widget::InitParams params;
    params.accept_events = true;
    params.activatable = views::Widget::InitParams::Activatable::kNo;
    params.context = context;
    params.delegate = this;
    params.type = views::Widget::InitParams::TYPE_WINDOW_FRAMELESS;
    params.wants_mouse_events_when_inactive = true;

    views::Widget* widget = new views::Widget();
    widget->Init(std::move(params));
  }

  absl::optional<std::vector<ui::FileInfo>> filenames_data_;
  absl::optional<base::Pickle> file_system_sources_data_;
};

// DropTargetView --------------------------------------------------------------

class DropTargetView : public views::WidgetDelegateView {
 public:
  DropTargetView(const DropTargetView&) = delete;
  DropTargetView& operator=(const DropTargetView&) = delete;
  ~DropTargetView() override = default;

  static DropTargetView* Create(aura::Window* context) {
    return new DropTargetView(context);
  }

  const base::FilePath& copied_file_path() const { return copied_file_path_; }

 private:
  explicit DropTargetView(aura::Window* context) { InitWidget(context); }

  // views::WidgetDelegateView:
  bool GetDropFormats(
      int* formats,
      std::set<ui::ClipboardFormatType>* format_types) override {
    *formats = ui::OSExchangeData::FILE_NAME;
    return true;
  }

  bool CanDrop(const ui::OSExchangeData& data) override { return true; }

  int OnDragUpdated(const ui::DropTargetEvent& event) override {
    return ui::DragDropTypes::DRAG_COPY;
  }

  ui::mojom::DragOperation OnPerformDrop(
      const ui::DropTargetEvent& event) override {
    EXPECT_TRUE(event.data().GetFilename(&copied_file_path_));
    return ui::mojom::DragOperation::kCopy;
  }

  void InitWidget(aura::Window* context) {
    views::Widget::InitParams params;
    params.accept_events = true;
    params.activatable = views::Widget::InitParams::Activatable::kNo;
    params.context = context;
    params.delegate = this;
    params.type = views::Widget::InitParams::TYPE_WINDOW_FRAMELESS;
    params.wants_mouse_events_when_inactive = true;

    views::Widget* widget = new views::Widget();
    widget->Init(std::move(params));
  }

  base::FilePath copied_file_path_;
};

// ViewDrawnWaiter -------------------------------------------------------------

class ViewDrawnWaiter : public views::ViewObserver {
 public:
  ViewDrawnWaiter() = default;
  ViewDrawnWaiter(const ViewDrawnWaiter&) = delete;
  ViewDrawnWaiter& operator=(const ViewDrawnWaiter&) = delete;
  ~ViewDrawnWaiter() override = default;

  void Wait(views::View* view) {
    if (IsDrawn(view))
      return;

    DCHECK(!wait_loop_);
    DCHECK(!view_observer_.IsObserving());

    view_observer_.Observe(view);

    wait_loop_ = std::make_unique<base::RunLoop>();
    wait_loop_->Run();
    wait_loop_.reset();

    view_observer_.Reset();
  }

 private:
  // views::ViewObserver:
  void OnViewVisibilityChanged(views::View* view,
                               views::View* starting_view) override {
    if (IsDrawn(view))
      wait_loop_->Quit();
  }

  void OnViewBoundsChanged(views::View* view) override {
    if (IsDrawn(view))
      wait_loop_->Quit();
  }

  bool IsDrawn(views::View* view) {
    return view->IsDrawn() && !view->size().IsEmpty();
  }

  std::unique_ptr<base::RunLoop> wait_loop_;
  base::ScopedObservation<views::View, views::ViewObserver> view_observer_{
      this};
};

// HoldingSpaceUiBrowserTest ---------------------------------------------------

// Base class for holding space UI browser tests.
class HoldingSpaceUiBrowserTest : public HoldingSpaceBrowserTestBase {
 public:
  // HoldingSpaceBrowserTestBase:
  void SetUpOnMainThread() override {
    HoldingSpaceBrowserTestBase::SetUpOnMainThread();

    ui::ScopedAnimationDurationScaleMode scoped_animation_duration_scale_mode(
        ui::ScopedAnimationDurationScaleMode::ZERO_DURATION);

    // The holding space tray will not show until the user has added a file to
    // holding space. Holding space UI browser tests don't need to assert that
    // behavior since it is already asserted in ash_unittests. As a convenience,
    // add and remove a holding space item so that the holding space tray will
    // already be showing during test execution.
    ASSERT_FALSE(test_api().IsShowingInShelf());
    RemoveItem(AddDownloadFile());
    ASSERT_TRUE(test_api().IsShowingInShelf());

    // Confirm that holding space model has been emptied for test execution.
    ASSERT_TRUE(HoldingSpaceController::Get()->model()->items().empty());
  }
};

}  // namespace

// Tests -----------------------------------------------------------------------

using PerformDragAndDropCallback =
    base::RepeatingCallback<void(const views::View* from,
                                 const views::View* to,
                                 DragUpdateCallback drag_update_callback,
                                 base::OnceClosure before_release_callback,
                                 base::OnceClosure after_release_callback)>;

enum StorageLocationFlag : uint32_t {
  kFilenames = 1 << 0,
  kFileSystemSources = 1 << 1,
};

using StorageLocationFlags = uint32_t;

// Base class for holding space UI browser tests that test drag-and-drop.
// Parameterized by:
//   [0] - callback to invoke to perform a drag-and-drop.
//   [1] - storage location(s) on `ui::OSExchangeData` at which to store files.
class HoldingSpaceUiDragAndDropBrowserTest
    : public HoldingSpaceUiBrowserTest,
      public testing::WithParamInterface<
          std::tuple<PerformDragAndDropCallback, StorageLocationFlags>> {
 public:
  // Asserts expectations that the holding space tray is or isn't a drop target.
  void ExpectTrayIsDropTarget(bool is_drop_target) {
    EXPECT_EQ(
        test_api().GetTrayDropTargetOverlay()->layer()->GetTargetOpacity(),
        is_drop_target ? 1.f : 0.f);
    EXPECT_EQ(test_api().GetDefaultTrayIcon()->layer()->GetTargetOpacity(),
              is_drop_target ? 0.f : 1.f);
    EXPECT_EQ(test_api().GetPreviewsTrayIcon()->layer()->GetTargetOpacity(),
              is_drop_target ? 0.f : 1.f);

    // Cache a reference to preview layers.
    const ui::Layer* previews_container_layer =
        test_api().GetPreviewsTrayIcon()->layer()->children()[0];
    const std::vector<ui::Layer*>& preview_layers =
        previews_container_layer->children();

    // Iterate over the layers for each preview.
    for (size_t i = 0; i < preview_layers.size(); ++i) {
      const ui::Layer* preview_layer = preview_layers[i];
      const float preview_width = preview_layer->size().width();

      // Previews layers are expected to be translated w/ incremental offset.
      gfx::Vector2dF expected_translation(i * preview_width / 2.f, 0.f);

      // When the holding space tray is a drop target, preview layers are
      // expected to be translated by a fixed amount in addition to the standard
      // incremental offset.
      if (is_drop_target) {
        constexpr int kPreviewIndexOffsetForDropTarget = 3;
        expected_translation += gfx::Vector2dF(
            kPreviewIndexOffsetForDropTarget * preview_width / 2.f, 0.f);
      }

      EXPECT_EQ(preview_layer->transform().To2dTranslation(),
                expected_translation);
    }
  }

  // Returns true if `screen_location` is within sufficient range of the holding
  // space tray so as to make it present itself as a drop target.
  bool IsWithinTrayDropTargetRange(const gfx::Point& screen_location) {
    constexpr int kProximityThreshold = 20;
    gfx::Rect screen_bounds(test_api().GetTray()->GetBoundsInScreen());
    screen_bounds.Inset(gfx::Insets(-kProximityThreshold));
    return screen_bounds.Contains(screen_location);
  }

  // Performs a drag-and-drop between `from` and `to`.
  void PerformDragAndDrop(
      const views::View* from,
      const views::View* to,
      DragUpdateCallback drag_update_callback = base::DoNothing(),
      base::OnceClosure before_release_callback = base::DoNothing(),
      base::OnceClosure after_release_callback = base::DoNothing()) {
    GetPerformDragAndDropCallback().Run(
        from, to, std::move(drag_update_callback),
        std::move(before_release_callback), std::move(after_release_callback));
  }

  // Sets data on `sender()` at the storage location specified by test params.
  void SetSenderData(const std::vector<base::FilePath>& file_paths) {
    if (ShouldStoreDataIn(StorageLocationFlag::kFilenames))
      sender()->SetFilenamesData(file_paths);
    else
      sender()->ClearFilenamesData();

    if (!ShouldStoreDataIn(StorageLocationFlag::kFileSystemSources)) {
      sender()->ClearFileSystemSourcesData();
      return;
    }

    std::vector<GURL> file_system_urls;
    for (const base::FilePath& file_path : file_paths) {
      file_system_urls.push_back(
          holding_space_util::ResolveFileSystemUrl(GetProfile(), file_path));
    }

    sender()->SetFileSystemSourcesData(file_system_urls);
  }

  // Returns the view serving as the drop sender for tests.
  DropSenderView* sender() { return drop_sender_view_; }

  // Returns the view serving as the drop target for tests.
  const DropTargetView* target() const { return drop_target_view_; }

 private:
  // HoldingSpaceUiBrowserTest:
  void SetUpOnMainThread() override {
    HoldingSpaceUiBrowserTest::SetUpOnMainThread();

    // Initialize `drop_sender_view_`.
    drop_sender_view_ = DropSenderView::Create(GetRootWindowForNewWindows());
    drop_sender_view_->GetWidget()->SetBounds(gfx::Rect(0, 0, 100, 100));
    drop_sender_view_->GetWidget()->ShowInactive();

    // Initialize `drop_target_view_`.
    drop_target_view_ = DropTargetView::Create(GetRootWindowForNewWindows());
    drop_target_view_->GetWidget()->SetBounds(gfx::Rect(100, 100, 100, 100));
    drop_target_view_->GetWidget()->ShowInactive();
  }

  void TearDownOnMainThread() override {
    drop_sender_view_->GetWidget()->Close();
    drop_target_view_->GetWidget()->Close();
    HoldingSpaceUiBrowserTest::TearDownOnMainThread();
  }

  PerformDragAndDropCallback GetPerformDragAndDropCallback() {
    return std::get<0>(GetParam());
  }

  StorageLocationFlags GetStorageLocationFlags() const {
    return std::get<1>(GetParam());
  }

  bool ShouldStoreDataIn(StorageLocationFlag flag) const {
    return GetStorageLocationFlags() & flag;
  }

  DropSenderView* drop_sender_view_ = nullptr;
  DropTargetView* drop_target_view_ = nullptr;
};

// Verifies that drag-and-drop of holding space items works.
IN_PROC_BROWSER_TEST_P(HoldingSpaceUiDragAndDropBrowserTest, DragAndDrop) {
  ui::ScopedAnimationDurationScaleMode scoped_animation_duration_scale_mode(
      ui::ScopedAnimationDurationScaleMode::ZERO_DURATION);

  // Verify drag-and-drop of download items.
  HoldingSpaceItem* const download_file = AddDownloadFile();

  test_api().Show();
  ASSERT_TRUE(test_api().IsShowing());

  // Let the message loop run so that resize of status tray takes effect.
  base::RunLoop().RunUntilIdle();

  std::vector<views::View*> download_chips = test_api().GetDownloadChips();
  ASSERT_EQ(1u, download_chips.size());

  PerformDragAndDrop(/*from=*/download_chips[0], /*to=*/target());
  EXPECT_EQ(download_file->file_path(), target()->copied_file_path());

  // Drag-and-drop should close holding space UI.
  FlushMessageLoop();
  ASSERT_FALSE(test_api().IsShowing());

  // Verify drag-and-drop of pinned file items.
  // NOTE: Dragging a pinned file from a non-top row of the pinned files
  // container grid previously resulted in a crash (crbug.com/1143426). To
  // explicitly test against this case we will add and drag a second row item.
  HoldingSpaceItem* const pinned_file = AddPinnedFile();
  AddPinnedFile();
  AddPinnedFile();

  test_api().Show();
  ASSERT_TRUE(test_api().IsShowing());

  std::vector<views::View*> pinned_file_chips = test_api().GetPinnedFileChips();
  ASSERT_EQ(3u, pinned_file_chips.size());

  PerformDragAndDrop(/*from=*/pinned_file_chips.back(), /*to=*/target());
  EXPECT_EQ(pinned_file->file_path(), target()->copied_file_path());

  // Drag-and-drop should close holding space UI.
  FlushMessageLoop();
  ASSERT_FALSE(test_api().IsShowing());

  // Verify drag-and-drop of screenshot items.
  HoldingSpaceItem* const screenshot_file = AddScreenshotFile();

  test_api().Show();
  ASSERT_TRUE(test_api().IsShowing());

  std::vector<views::View*> screen_capture_views =
      test_api().GetScreenCaptureViews();
  ASSERT_EQ(1u, screen_capture_views.size());

  PerformDragAndDrop(/*from=*/screen_capture_views[0], /*to=*/target());
  EXPECT_EQ(screenshot_file->file_path(), target()->copied_file_path());

  // Drag-and-drop should close holding space UI.
  FlushMessageLoop();
  ASSERT_FALSE(test_api().IsShowing());
}

// Verifies that drag-and-drop to pin holding space items works.
IN_PROC_BROWSER_TEST_P(HoldingSpaceUiDragAndDropBrowserTest, DragAndDropToPin) {
  ui::ScopedAnimationDurationScaleMode scoped_animation_duration_scale_mode(
      ui::ScopedAnimationDurationScaleMode::ZERO_DURATION);

  // Add an item to holding space to cause the holding space tray to appear.
  AddDownloadFile();
  ASSERT_TRUE(test_api().IsShowingInShelf());

  // Let the message loop run so that resize of status tray takes effect.
  base::RunLoop().RunUntilIdle();

  // Bind an observer to watch for updates to the holding space model.
  testing::NiceMock<MockHoldingSpaceModelObserver> mock;
  base::ScopedObservation<HoldingSpaceModel, HoldingSpaceModelObserver>
      observer{&mock};
  observer.Observe(HoldingSpaceController::Get()->model());

  // Create a file to be dragged into the holding space.
  std::vector<base::FilePath> file_paths;
  file_paths.push_back(CreateFile());
  SetSenderData(file_paths);

  // Expect no events have been recorded to histograms.
  base::HistogramTester histogram_tester;
  histogram_tester.ExpectBucketCount(
      "HoldingSpace.Pod.Action.All",
      holding_space_metrics::PodAction::kDragAndDropToPin, 0);

  {
    base::RunLoop run_loop;
    EXPECT_CALL(mock, OnHoldingSpaceItemsAdded)
        .WillOnce([&](const std::vector<const HoldingSpaceItem*>& items) {
          ASSERT_EQ(items.size(), 1u);
          ASSERT_EQ(items[0]->type(), HoldingSpaceItem::Type::kPinnedFile);
          ASSERT_EQ(items[0]->file_path(), file_paths[0]);
          run_loop.Quit();
        });

    // Perform and verify the ability to pin a file via drag-and-drop.
    ExpectTrayIsDropTarget(false);
    PerformDragAndDrop(
        /*from=*/sender(), /*to=*/test_api().GetTray(),
        /*drag_update_callback=*/
        base::BindRepeating(
            &HoldingSpaceUiDragAndDropBrowserTest::IsWithinTrayDropTargetRange,
            base::Unretained(this))
            .Then(base::BindRepeating(
                &HoldingSpaceUiDragAndDropBrowserTest::ExpectTrayIsDropTarget,
                base::Unretained(this))),
        /*before_release_callback=*/
        base::BindOnce(
            &HoldingSpaceUiDragAndDropBrowserTest::ExpectTrayIsDropTarget,
            base::Unretained(this), true),
        /*after_release_callback=*/
        base::BindOnce(
            &HoldingSpaceUiDragAndDropBrowserTest::ExpectTrayIsDropTarget,
            base::Unretained(this), false));
    run_loop.Run();

    // Expect the event has been recorded to histograms.
    histogram_tester.ExpectBucketCount(
        "HoldingSpace.Pod.Action.All",
        holding_space_metrics::PodAction::kDragAndDropToPin, 1);
  }

  // Create a few more files to be dragged into the holding space.
  file_paths.push_back(CreateFile());
  file_paths.push_back(CreateFile());
  SetSenderData(file_paths);

  {
    base::RunLoop run_loop;
    EXPECT_CALL(mock, OnHoldingSpaceItemsAdded)
        .WillOnce([&](const std::vector<const HoldingSpaceItem*>& items) {
          ASSERT_EQ(items.size(), 2u);
          ASSERT_EQ(items[0]->type(), HoldingSpaceItem::Type::kPinnedFile);
          ASSERT_EQ(items[0]->file_path(), file_paths[1]);
          ASSERT_EQ(items[1]->type(), HoldingSpaceItem::Type::kPinnedFile);
          ASSERT_EQ(items[1]->file_path(), file_paths[2]);
          run_loop.Quit();
        });

    // Perform and verify the ability to pin multiple files via drag-and-drop.
    // Note that any already pinned files in the drop payload are ignored.
    ExpectTrayIsDropTarget(false);
    PerformDragAndDrop(
        /*from=*/sender(), /*to=*/test_api().GetTray(),
        /*drag_update_callback=*/
        base::BindRepeating(
            &HoldingSpaceUiDragAndDropBrowserTest::IsWithinTrayDropTargetRange,
            base::Unretained(this))
            .Then(base::BindRepeating(
                &HoldingSpaceUiDragAndDropBrowserTest::ExpectTrayIsDropTarget,
                base::Unretained(this))),
        /*before_release_callback=*/
        base::BindOnce(
            &HoldingSpaceUiDragAndDropBrowserTest::ExpectTrayIsDropTarget,
            base::Unretained(this), true),
        /*after_release_callback=*/
        base::BindOnce(
            &HoldingSpaceUiDragAndDropBrowserTest::ExpectTrayIsDropTarget,
            base::Unretained(this), false));
    run_loop.Run();

    // Expect the event has been recorded to histograms.
    histogram_tester.ExpectBucketCount(
        "HoldingSpace.Pod.Action.All",
        holding_space_metrics::PodAction::kDragAndDropToPin, 2);
  }

  // Swap out the registered holding space client with a mock.
  testing::NiceMock<MockHoldingSpaceClient> client;
  HoldingSpaceController::Get()->RegisterClientAndModelForUser(
      ProfileHelper::Get()->GetUserByProfile(GetProfile())->GetAccountId(),
      &client, HoldingSpaceController::Get()->model());
  ASSERT_EQ(&client, HoldingSpaceController::Get()->client());

  {
    // Verify that attempting to drag-and-drop a payload which contains only
    // files that are already pinned will not result in a client interaction.
    EXPECT_CALL(client, PinFiles).Times(0);
    ExpectTrayIsDropTarget(false);
    PerformDragAndDrop(
        /*from=*/sender(), /*to=*/test_api().GetTray(),
        /*drag_update_callback=*/
        base::BindRepeating(
            [](HoldingSpaceUiDragAndDropBrowserTest* test,
               const gfx::Point& screen_location) {
              // The drag payload cannot be handled by holding space so the tray
              // should never indicate it is a drop target regardless of drag
              // update `screen_location`.
              test->ExpectTrayIsDropTarget(false);
            },
            base::Unretained(this)),
        /*before_release_callback=*/
        base::BindOnce(
            &HoldingSpaceUiDragAndDropBrowserTest::ExpectTrayIsDropTarget,
            base::Unretained(this), false),
        /*after_release_callback=*/
        base::BindOnce(
            &HoldingSpaceUiDragAndDropBrowserTest::ExpectTrayIsDropTarget,
            base::Unretained(this), false));
    testing::Mock::VerifyAndClearExpectations(&client);

    // Expect no event has been recorded to histograms.
    histogram_tester.ExpectBucketCount(
        "HoldingSpace.Pod.Action.All",
        holding_space_metrics::PodAction::kDragAndDropToPin, 2);
  }
}

INSTANTIATE_TEST_SUITE_P(
    All,
    HoldingSpaceUiDragAndDropBrowserTest,
    testing::Combine(testing::ValuesIn({
                         base::BindRepeating(&MouseDrag),
                         base::BindRepeating(&GestureDrag),
                     }),
                     testing::ValuesIn(std::vector<StorageLocationFlags>({
                         StorageLocationFlag::kFilenames,
                         StorageLocationFlag::kFileSystemSources,
                         StorageLocationFlag::kFilenames |
                             StorageLocationFlag::kFileSystemSources,
                     }))));

// Verifies that the holding space tray does not appear on the lock screen.
IN_PROC_BROWSER_TEST_F(HoldingSpaceUiBrowserTest, LockScreen) {
  ui::ScopedAnimationDurationScaleMode scoped_animation_duration_scale_mode(
      ui::ScopedAnimationDurationScaleMode::ZERO_DURATION);

  ASSERT_TRUE(test_api().IsShowingInShelf());
  RequestAndAwaitLockScreen();
  ASSERT_FALSE(test_api().IsShowingInShelf());
}

// Verifies that pinning and unpinning holding space items works as intended.
IN_PROC_BROWSER_TEST_F(HoldingSpaceUiBrowserTest, PinAndUnpinItems) {
  ui::ScopedAnimationDurationScaleMode scoped_animation_duration_scale_mode(
      ui::ScopedAnimationDurationScaleMode::ZERO_DURATION);

  // Add an item of every type. For downloads, also add an in-progress item.
  for (HoldingSpaceItem::Type type : GetHoldingSpaceItemTypes())
    AddItem(GetProfile(), type, CreateFile());
  AddItem(GetProfile(), HoldingSpaceItem::Type::kDownload, CreateFile(),
          /*progress=*/0.f);

  // Show holding space UI.
  test_api().Show();
  ASSERT_TRUE(test_api().IsShowing());

  // Verify existence of views for pinned files, screen captures, and downloads.
  using ViewList = std::vector<views::View*>;
  ViewList pinned_file_chips = test_api().GetPinnedFileChips();
  ASSERT_EQ(pinned_file_chips.size(), 1u);
  ViewList screen_capture_views = test_api().GetScreenCaptureViews();
  ASSERT_GE(screen_capture_views.size(), 1u);
  ViewList download_chips = test_api().GetDownloadChips();
  ASSERT_GE(download_chips.size(), 2u);

  // Attempt to pin a screen capture via context menu.
  RightClick(screen_capture_views.front());
  ASSERT_TRUE(SelectMenuItemWithCommandId(HoldingSpaceCommandId::kPinItem));
  PressAndReleaseKey(ui::KeyboardCode::VKEY_RETURN);
  pinned_file_chips = test_api().GetPinnedFileChips();
  ASSERT_EQ(pinned_file_chips.size(), 2u);
  ASSERT_EQ(
      test_api().GetHoldingSpaceItemFilePath(pinned_file_chips.front()),
      test_api().GetHoldingSpaceItemFilePath(screen_capture_views.front()));

  // Attempt to pin a completed download via context menu. Note that the first
  // download is the in-progress download, so don't select that one.
  RightClick(download_chips.at(1));
  ASSERT_TRUE(SelectMenuItemWithCommandId(HoldingSpaceCommandId::kPinItem));
  PressAndReleaseKey(ui::KeyboardCode::VKEY_RETURN);
  pinned_file_chips = test_api().GetPinnedFileChips();
  ASSERT_EQ(pinned_file_chips.size(), 3u);
  ASSERT_EQ(test_api().GetHoldingSpaceItemFilePath(pinned_file_chips.front()),
            test_api().GetHoldingSpaceItemFilePath(download_chips.at(1)));

  // Attempt to pin an in-progress download via context menu. Because the
  // download is in-progress, it should neither be pin- or unpin-able.
  RightClick(download_chips.front());
  ASSERT_TRUE(views::MenuController::GetActiveInstance());
  ASSERT_FALSE(SelectMenuItemWithCommandId(HoldingSpaceCommandId::kPinItem));
  ASSERT_FALSE(SelectMenuItemWithCommandId(HoldingSpaceCommandId::kUnpinItem));
  PressAndReleaseKey(ui::KeyboardCode::VKEY_ESCAPE);

  // Attempt to unpin the pinned download via context menu without de-selecting
  // the in-progress download. Because the selection contains items which are
  // not in-progress and all of those items are pinned, the selection should be
  // unpin-able.
  RightClick(download_chips.at(1), ui::EF_CONTROL_DOWN);
  ASSERT_TRUE(SelectMenuItemWithCommandId(HoldingSpaceCommandId::kUnpinItem));
  PressAndReleaseKey(ui::KeyboardCode::VKEY_RETURN);
  pinned_file_chips = test_api().GetPinnedFileChips();
  ASSERT_EQ(pinned_file_chips.size(), 2u);
  ASSERT_EQ(
      test_api().GetHoldingSpaceItemFilePath(pinned_file_chips.front()),
      test_api().GetHoldingSpaceItemFilePath(screen_capture_views.front()));

  // Select the pinned file and again attempt to pin the completed download via
  // context menu, still without de-selecting the in-progress download. Because
  // the selection contains items which are not in-progress and at least one of
  // those items are unpinned, the selection should be pin-able.
  Click(pinned_file_chips.front(), ui::EF_CONTROL_DOWN);
  RightClick(download_chips.front());
  ASSERT_TRUE(SelectMenuItemWithCommandId(HoldingSpaceCommandId::kPinItem));
  PressAndReleaseKey(ui::KeyboardCode::VKEY_RETURN);
  pinned_file_chips = test_api().GetPinnedFileChips();
  ASSERT_EQ(pinned_file_chips.size(), 3u);
  ASSERT_EQ(test_api().GetHoldingSpaceItemFilePath(pinned_file_chips.front()),
            test_api().GetHoldingSpaceItemFilePath(download_chips.at(1)));
}

// Verifies that opening holding space items works.
IN_PROC_BROWSER_TEST_F(HoldingSpaceUiBrowserTest, OpenItem) {
  // Install the Media App, which we expect to open holding space items.
  WaitForTestSystemAppInstall();

  ui::ScopedAnimationDurationScaleMode scoped_animation_duration_scale_mode(
      ui::ScopedAnimationDurationScaleMode::ZERO_DURATION);

  auto* const activation_client = wm::GetActivationClient(
      HoldingSpaceBrowserTestBase::GetRootWindowForNewWindows());

  // Observe the `activation_client` so we can detect windows becoming active as
  // a result of opening holding space items.
  testing::NiceMock<MockActivationChangeObserver> mock;
  base::ScopedObservation<wm::ActivationClient, wm::ActivationChangeObserver>
      obs{&mock};
  obs.Observe(activation_client);

  // Create a holding space item.
  AddScreenshotFile();

  // We're going to verify we can open holding space items by interacting with
  // the view in a few ways as we expect a user to.
  std::vector<base::OnceCallback<void(const views::View*)>> user_interactions;
  user_interactions.push_back(base::BindOnce(&DoubleClick));
  user_interactions.push_back(base::BindOnce(&GestureTap));
  user_interactions.push_back(base::BindOnce([](const views::View* view) {
    while (!view->HasFocus())
      PressAndReleaseKey(ui::KeyboardCode::VKEY_TAB);
    PressAndReleaseKey(ui::KeyboardCode::VKEY_RETURN);
  }));

  for (auto& user_interaction : user_interactions) {
    // Show holding space UI and verify a holding space item view exists.
    test_api().Show();
    ASSERT_TRUE(test_api().IsShowing());
    std::vector<views::View*> screen_capture_views =
        test_api().GetScreenCaptureViews();
    ASSERT_EQ(1u, screen_capture_views.size());

    // Attempt to open the holding space item via user interaction on its view.
    std::move(user_interaction).Run(screen_capture_views[0]);

    // Expect and wait for a `Gallery` window to be activated since the holding
    // space item that we attempted to open was a screenshot.
    base::RunLoop run_loop;
    EXPECT_CALL(mock, OnWindowActivated)
        .WillOnce([&](wm::ActivationChangeObserver::ActivationReason reason,
                      aura::Window* gained_active, aura::Window* lost_active) {
          EXPECT_EQ("Gallery", base::UTF16ToUTF8(gained_active->GetTitle()));
          run_loop.Quit();
        });
    run_loop.Run();

    // Reset.
    testing::Mock::VerifyAndClearExpectations(&mock);
    activation_client->DeactivateWindow(activation_client->GetActiveWindow());
  }
}

// Verifies that removing holding space items works as intended.
IN_PROC_BROWSER_TEST_F(HoldingSpaceUiBrowserTest, RemoveItem) {
  ui::ScopedAnimationDurationScaleMode scoped_animation_duration_scale_mode(
      ui::ScopedAnimationDurationScaleMode::ZERO_DURATION);

  // Populate holding space with items of all types.
  for (HoldingSpaceItem::Type type : GetHoldingSpaceItemTypes())
    AddItem(GetProfile(), type, CreateFile());

  test_api().Show();
  ASSERT_TRUE(test_api().IsShowing());

  std::vector<views::View*> pinned_file_chips = test_api().GetPinnedFileChips();
  ASSERT_EQ(1u, pinned_file_chips.size());

  // Right clicking a pinned item should cause a context menu to show.
  ASSERT_FALSE(views::MenuController::GetActiveInstance());
  ViewDrawnWaiter().Wait(pinned_file_chips.front());
  RightClick(pinned_file_chips.front());
  ASSERT_TRUE(views::MenuController::GetActiveInstance());

  // There should be no `kRemoveItem` command for pinned items.
  ASSERT_FALSE(SelectMenuItemWithCommandId(HoldingSpaceCommandId::kRemoveItem));

  // Close the context menu.
  PressAndReleaseKey(ui::KeyboardCode::VKEY_ESCAPE);
  ASSERT_FALSE(views::MenuController::GetActiveInstance());

  std::vector<views::View*> download_chips = test_api().GetDownloadChips();
  ASSERT_GT(download_chips.size(), 1u);

  // Add a download item to the selection and show the context menu.
  ViewDrawnWaiter().Wait(download_chips.front());
  Click(download_chips.front(), ui::EF_CONTROL_DOWN);
  RightClick(download_chips.front());
  ASSERT_TRUE(views::MenuController::GetActiveInstance());

  // There should be no `kRemoveItem` command since a pinned item is selected.
  ASSERT_FALSE(SelectMenuItemWithCommandId(HoldingSpaceCommandId::kRemoveItem));

  // Close the context menu.
  PressAndReleaseKey(ui::KeyboardCode::VKEY_ESCAPE);
  ASSERT_FALSE(views::MenuController::GetActiveInstance());

  // Unselect the pinned item and right click show the context menu.
  Click(pinned_file_chips.front(), ui::EF_CONTROL_DOWN);
  RightClick(download_chips.front());
  ASSERT_TRUE(views::MenuController::GetActiveInstance());

  // There should be a `kRemoveItem` command in the context menu.
  ASSERT_TRUE(SelectMenuItemWithCommandId(HoldingSpaceCommandId::kRemoveItem));

  // Bind an observer to watch for updates to the holding space model.
  testing::NiceMock<MockHoldingSpaceModelObserver> mock;
  base::ScopedObservation<HoldingSpaceModel, HoldingSpaceModelObserver>
      observer{&mock};
  observer.Observe(HoldingSpaceController::Get()->model());

  {
    // Cache `item_id` of the download item to be removed.
    const std::string item_id =
        test_api().GetHoldingSpaceItemId(download_chips.front());
    EXPECT_EQ(test_api().GetHoldingSpaceItemView(download_chips, item_id),
              download_chips.front());

    base::RunLoop run_loop;
    EXPECT_CALL(mock, OnHoldingSpaceItemsRemoved)
        .WillOnce([&](const std::vector<const HoldingSpaceItem*>& items) {
          ASSERT_EQ(items.size(), 1u);
          EXPECT_EQ(items[0]->id(), item_id);
          run_loop.Quit();
        });

    // Press `ENTER` to remove the selected download item.
    PressAndReleaseKey(ui::KeyboardCode::VKEY_RETURN);
    run_loop.Run();

    // Verify the download chip has been removed.
    download_chips = test_api().GetDownloadChips();
    EXPECT_FALSE(test_api().GetHoldingSpaceItemView(download_chips, item_id));
  }

  std::vector<views::View*> screen_capture_views =
      test_api().GetScreenCaptureViews();
  ASSERT_GT(screen_capture_views.size(), 1u);

  // Select a screen capture item and show the context menu.
  ViewDrawnWaiter().Wait(screen_capture_views.front());
  Click(screen_capture_views.front());
  RightClick(screen_capture_views.front());
  ASSERT_TRUE(views::MenuController::GetActiveInstance());

  // There should be a `kRemoveItem` command in the context menu.
  ASSERT_TRUE(SelectMenuItemWithCommandId(HoldingSpaceCommandId::kRemoveItem));

  {
    // Cache `item_id` of the screen capture item to be removed.
    const std::string item_id =
        test_api().GetHoldingSpaceItemId(screen_capture_views.front());
    EXPECT_EQ(test_api().GetHoldingSpaceItemView(screen_capture_views, item_id),
              screen_capture_views.front());

    base::RunLoop run_loop;
    EXPECT_CALL(mock, OnHoldingSpaceItemsRemoved)
        .WillOnce([&](const std::vector<const HoldingSpaceItem*>& items) {
          ASSERT_EQ(items.size(), 1u);
          EXPECT_EQ(items[0]->id(), item_id);
          run_loop.Quit();
        });

    // Press `ENTER` to remove the selected screen capture item.
    PressAndReleaseKey(ui::KeyboardCode::VKEY_RETURN);
    run_loop.Run();

    // Verify the screen capture view has been removed.
    screen_capture_views = test_api().GetScreenCaptureViews();
    EXPECT_FALSE(
        test_api().GetHoldingSpaceItemView(screen_capture_views, item_id));
  }

  // Remove all items in the recent files bubble. Note that not all download
  // items or screen capture items that exist may be visible at the same time
  // due to max visibility count restrictions.
  while (!download_chips.empty() || !screen_capture_views.empty()) {
    // Select all visible download items.
    for (views::View* download_chip : download_chips) {
      ViewDrawnWaiter().Wait(download_chip);
      Click(download_chip, ui::EF_CONTROL_DOWN);
    }

    // Select all visible screen capture items.
    for (views::View* screen_capture_view : screen_capture_views) {
      ViewDrawnWaiter().Wait(screen_capture_view);
      Click(screen_capture_view, ui::EF_CONTROL_DOWN);
    }

    // Show the context menu. There should be a `kRemoveItem` command.
    RightClick(download_chips.size() ? download_chips.front()
                                     : screen_capture_views.front());
    ASSERT_TRUE(views::MenuController::GetActiveInstance());
    ASSERT_TRUE(
        SelectMenuItemWithCommandId(HoldingSpaceCommandId::kRemoveItem));

    {
      // Cache `item_ids` of download and screen capture items to be removed.
      std::set<std::string> item_ids;
      for (const views::View* download_chip : download_chips) {
        auto it =
            item_ids.insert(test_api().GetHoldingSpaceItemId(download_chip));
        EXPECT_EQ(test_api().GetHoldingSpaceItemView(download_chips, *it.first),
                  download_chip);
      }
      for (const views::View* screen_capture_view : screen_capture_views) {
        auto it = item_ids.insert(
            test_api().GetHoldingSpaceItemId(screen_capture_view));
        EXPECT_EQ(
            test_api().GetHoldingSpaceItemView(screen_capture_views, *it.first),
            screen_capture_view);
      }

      base::RunLoop run_loop;
      EXPECT_CALL(mock, OnHoldingSpaceItemsRemoved)
          .WillOnce([&](const std::vector<const HoldingSpaceItem*>& items) {
            ASSERT_EQ(items.size(), item_ids.size());
            for (const HoldingSpaceItem* item : items)
              ASSERT_TRUE(base::Contains(item_ids, item->id()));
            run_loop.Quit();
          });

      // Press `ENTER` to remove the selected items.
      PressAndReleaseKey(ui::KeyboardCode::VKEY_RETURN);
      run_loop.Run();

      // Verify all previously visible download chips and screen capture views
      // have been removed.
      download_chips = test_api().GetDownloadChips();
      screen_capture_views = test_api().GetScreenCaptureViews();
      for (const std::string& item_id : item_ids) {
        EXPECT_FALSE(
            test_api().GetHoldingSpaceItemView(download_chips, item_id));
        EXPECT_FALSE(
            test_api().GetHoldingSpaceItemView(screen_capture_views, item_id));
      }
    }
  }

  // The recent files bubble should be empty and therefore hidden.
  ASSERT_FALSE(test_api().RecentFilesBubbleShown());
}

// Verifies that unpinning a pinned holding space item works as intended.
IN_PROC_BROWSER_TEST_F(HoldingSpaceUiBrowserTest, UnpinItem) {
  ui::ScopedAnimationDurationScaleMode scoped_animation_duration_scale_mode(
      ui::ScopedAnimationDurationScaleMode::ZERO_DURATION);

  // Add enough pinned items for there to be multiple rows in the section.
  constexpr size_t kNumPinnedItems = 3u;
  for (size_t i = 0; i < kNumPinnedItems; ++i)
    AddPinnedFile();

  test_api().Show();
  ASSERT_TRUE(test_api().IsShowing());

  std::vector<views::View*> pinned_file_chips = test_api().GetPinnedFileChips();
  ASSERT_EQ(kNumPinnedItems, pinned_file_chips.size());

  // Operate on the last `pinned_file_chip` as there was an easy to reproduce
  // bug in which unpinning a chip *not* in the top row resulted in a crash on
  // destruction due to its ink drop layer attempting to be reordered.
  views::View* pinned_file_chip = pinned_file_chips.back();

  // The pin button is only visible after mousing over the `pinned_file_chip`,
  // so move the mouse and wait for the pin button to be drawn. Note that the
  // mouse is moved over multiple events to ensure that the appropriate mouse
  // enter event is also generated.
  MoveMouseTo(pinned_file_chip, /*count=*/10);
  auto* pin_btn = pinned_file_chip->GetViewByID(kHoldingSpaceItemPinButtonId);
  ViewDrawnWaiter().Wait(pin_btn);

  Click(pin_btn);

  pinned_file_chips = test_api().GetPinnedFileChips();
  ASSERT_EQ(kNumPinnedItems - 1, pinned_file_chips.size());
}

// Verifies that previews can be toggled via context menu.
IN_PROC_BROWSER_TEST_F(HoldingSpaceUiBrowserTest, TogglePreviews) {
  ui::ScopedAnimationDurationScaleMode scoped_animation_duration_scale_mode(
      ui::ScopedAnimationDurationScaleMode::ZERO_DURATION);

  ASSERT_TRUE(test_api().IsShowingInShelf());

  // Initially, the default icon should be shown.
  auto* default_tray_icon = test_api().GetDefaultTrayIcon();
  ASSERT_TRUE(default_tray_icon);
  EXPECT_TRUE(default_tray_icon->GetVisible());

  auto* previews_tray_icon = test_api().GetPreviewsTrayIcon();
  ASSERT_TRUE(previews_tray_icon);
  ASSERT_TRUE(previews_tray_icon->layer());
  ASSERT_EQ(1u, previews_tray_icon->layer()->children().size());
  auto* previews_container_layer = previews_tray_icon->layer()->children()[0];
  EXPECT_FALSE(previews_tray_icon->GetVisible());

  // After pinning a file, we should have a single preview in the tray icon.
  AddPinnedFile();
  FlushMessageLoop();

  EXPECT_FALSE(default_tray_icon->GetVisible());
  EXPECT_TRUE(previews_tray_icon->GetVisible());

  EXPECT_EQ(1u, previews_container_layer->children().size());
  EXPECT_EQ(gfx::Size(32, 32), previews_tray_icon->size());

  // After downloading a file, we should have two previews in the tray icon.
  AddDownloadFile();
  FlushMessageLoop();

  EXPECT_FALSE(default_tray_icon->GetVisible());
  EXPECT_TRUE(previews_tray_icon->GetVisible());
  EXPECT_EQ(2u, previews_container_layer->children().size());
  EXPECT_EQ(gfx::Size(48, 32), previews_tray_icon->size());

  // After taking a screenshot, we should have three previews in the tray icon.
  AddScreenshotFile();
  FlushMessageLoop();

  EXPECT_FALSE(default_tray_icon->GetVisible());
  EXPECT_TRUE(previews_tray_icon->GetVisible());
  EXPECT_EQ(3u, previews_container_layer->children().size());
  EXPECT_EQ(gfx::Size(64, 32), previews_tray_icon->size());

  // Right click the tray icon, and expect a context menu to be shown which will
  // allow the user to hide previews.
  ViewDrawnWaiter().Wait(previews_tray_icon);
  RightClick(previews_tray_icon);
  ASSERT_TRUE(views::MenuController::GetActiveInstance());

  // Use the keyboard to select the context menu item to hide previews. Doing so
  // should dismiss the context menu.
  PressAndReleaseKey(ui::KeyboardCode::VKEY_DOWN);
  PressAndReleaseKey(ui::KeyboardCode::VKEY_RETURN);
  EXPECT_FALSE(views::MenuController::GetActiveInstance());
  FlushMessageLoop();

  // The tray icon should now contain no previews, but have a single child which
  // contains the static image to show when previews are disabled.
  EXPECT_TRUE(default_tray_icon->GetVisible());
  EXPECT_FALSE(previews_tray_icon->GetVisible());

  EXPECT_EQ(gfx::Size(32, 32), default_tray_icon->size());

  // Right click the tray icon, and expect a context menu to be shown which will
  // allow the user to show previews.
  ViewDrawnWaiter().Wait(default_tray_icon);
  RightClick(default_tray_icon);
  ASSERT_TRUE(views::MenuController::GetActiveInstance());

  // Use the keyboard to select the context menu item to show previews. Doing so
  // should dismiss the context menu.
  PressAndReleaseKey(ui::KeyboardCode::VKEY_DOWN);
  PressAndReleaseKey(ui::KeyboardCode::VKEY_RETURN);
  EXPECT_FALSE(views::MenuController::GetActiveInstance());
  FlushMessageLoop();

  // The tray icon should once again show three previews.
  EXPECT_FALSE(default_tray_icon->GetVisible());
  EXPECT_TRUE(previews_tray_icon->GetVisible());

  EXPECT_EQ(3u, previews_container_layer->children().size());
  EXPECT_EQ(gfx::Size(64, 32), previews_tray_icon->size());
}

// Base class for holding space UI browser tests that require in-progress
// downloads integration. NOTE: This test suite will swap out the production
// download manager with a mock instance.
class HoldingSpaceUiInProgressDownloadsBrowserTest
    : public HoldingSpaceUiBrowserTest {
 public:
  HoldingSpaceUiInProgressDownloadsBrowserTest() {
    // Enable in-progress downloads integration.
    scoped_feature_list_.InitAndEnableFeature(
        features::kHoldingSpaceInProgressDownloadsIntegration);

    // Mock `content::DownloadManager::IsManagerInitialized()`.
    ON_CALL(download_manager_, IsManagerInitialized())
        .WillByDefault(testing::Return(true));

    // Mock `content::DownloadManager::AddObserver()`.
    ON_CALL(download_manager_, AddObserver)
        .WillByDefault(testing::Invoke(
            &download_manager_observers_,
            &base::ObserverList<
                content::DownloadManager::Observer>::Unchecked::AddObserver));

    // Mock `content::DownloadManager::RemoveObserver()`.
    ON_CALL(download_manager_, RemoveObserver)
        .WillByDefault(testing::Invoke(
            &download_manager_observers_,
            &base::ObserverList<content::DownloadManager::Observer>::Unchecked::
                RemoveObserver));

    // Swap out the production download manager with the mock.
    HoldingSpaceDownloadsDelegate::SetDownloadManagerForTesting(
        &download_manager_);
  }

  ~HoldingSpaceUiInProgressDownloadsBrowserTest() override {
    for (auto& observer : download_manager_observers_)
      observer.ManagerGoingDown(&download_manager_);
  }

  // Creates and returns a mock download item with the specified `state`,
  // `file_path`, `target_file_path`, and `percent_complete`.
  std::unique_ptr<testing::NiceMock<download::MockDownloadItem>>
  CreateMockDownloadItem(download::DownloadItem::DownloadState state,
                         const base::FilePath& file_path,
                         const base::FilePath& target_file_path,
                         int percent_complete) {
    auto mock_download_item =
        std::make_unique<testing::NiceMock<download::MockDownloadItem>>();

    // Mock `download::DownloadItem::Cancel()`.
    ON_CALL(*mock_download_item, Cancel(/*from_user=*/testing::Eq(true)))
        .WillByDefault(testing::InvokeWithoutArgs(
            [mock_download_item = mock_download_item.get()]() {
              // When a download is cancelled, the underlying file is deleted.
              const auto& file_path = mock_download_item->GetFullPath();
              if (!file_path.empty()) {
                base::ScopedAllowBlockingForTesting allow_blocking;
                ASSERT_TRUE(base::DeleteFile(file_path));
              }
            }));

    // Mock `download::DownloadItem::GetFullPath()`.
    ON_CALL(*mock_download_item, GetFullPath)
        .WillByDefault(testing::Invoke(
            [mock_download_item = mock_download_item.get(),
             file_path = base::FilePath(file_path)]() -> const base::FilePath& {
              return mock_download_item->GetState() ==
                             download::DownloadItem::COMPLETE
                         ? mock_download_item->GetTargetFilePath()
                         : file_path;
            }));

    // Mock `download::DownloadItem::GetState()`.
    ON_CALL(*mock_download_item, GetState)
        .WillByDefault(testing::Return(state));

    // Mock `download::DownloadItem::GetTargetFilePath()`.
    ON_CALL(*mock_download_item, GetTargetFilePath)
        .WillByDefault(testing::ReturnRefOfCopy(target_file_path));

    // Mock `download::DownloadItem::GetTotalBytes()`.
    ON_CALL(*mock_download_item, GetTotalBytes)
        .WillByDefault(testing::Return(-1));

    // Mock `download::DownloadItem::IsPaused()`.
    auto paused = std::make_unique<bool>(false);
    ON_CALL(*mock_download_item, IsPaused)
        .WillByDefault(testing::Invoke(
            [paused_ptr = paused.get()]() { return *paused_ptr; }));

    // Create a callback which can be run to set `paused` state and which
    // mirrors production behavior by notifying observers on change.
    auto set_paused = base::BindRepeating(
        [](download::MockDownloadItem* mock_download_item, bool* paused,
           bool new_paused) {
          if (*paused != new_paused) {
            *paused = new_paused;
            mock_download_item->NotifyObserversDownloadUpdated();
          }
        },
        base::Unretained(mock_download_item.get()),
        base::Owned(std::move(paused)));

    // Mock `download::DownloadItem::Pause()`.
    ON_CALL(*mock_download_item, Pause).WillByDefault([set_paused]() {
      set_paused.Run(true);
    });

    // Mock `download::DownloadItem::PercentComplete()`.
    ON_CALL(*mock_download_item, PercentComplete)
        .WillByDefault(testing::Return(percent_complete));

    // Mock `download::DownloadItem::Resume()`.
    ON_CALL(*mock_download_item, Resume(/*from_user=*/testing::Eq(true)))
        .WillByDefault([set_paused]() { set_paused.Run(false); });

    // Notify observers of the created download.
    for (auto& observer : download_manager_observers_)
      observer.OnDownloadCreated(&download_manager_, mock_download_item.get());

    return mock_download_item;
  }

 private:
  base::test::ScopedFeatureList scoped_feature_list_;
  testing::NiceMock<content::MockDownloadManager> download_manager_;
  base::ObserverList<content::DownloadManager::Observer>::Unchecked
      download_manager_observers_;
};

// Verifies that primary and secondary text are displayed as intended.
IN_PROC_BROWSER_TEST_F(HoldingSpaceUiInProgressDownloadsBrowserTest,
                       PrimaryAndSecondaryText) {
  ui::ScopedAnimationDurationScaleMode scoped_animation_duration_scale_mode(
      ui::ScopedAnimationDurationScaleMode::ZERO_DURATION);

  // Force locale since strings are being verified.
  base::ScopedLocale scoped_locale("en_US.UTF-8");

  // Create an in-progress download.
  const base::FilePath target_file_path(CreateFile());
  auto in_progress_download = CreateMockDownloadItem(
      download::DownloadItem::IN_PROGRESS, /*file_path=*/CreateFile(),
      target_file_path, /*percent_complete=*/0);
  in_progress_download->NotifyObserversDownloadUpdated();

  // Show holding space UI.
  test_api().Show();
  ASSERT_TRUE(test_api().IsShowing());

  // Verify the existence of a single download chip.
  std::vector<views::View*> download_chips = test_api().GetDownloadChips();
  ASSERT_EQ(download_chips.size(), 1u);

  // Cache pointers to the `primary_label` and `secondary_label`.
  auto* primary_label = static_cast<views::Label*>(
      download_chips[0]->GetViewByID(kHoldingSpaceItemPrimaryChipLabelId));
  auto* secondary_label = static_cast<views::Label*>(
      download_chips[0]->GetViewByID(kHoldingSpaceItemSecondaryChipLabelId));

  // The `primary_label` should always be visible and should always show the
  // lossy display name of the download's target file path.
  const auto target_file_name = target_file_path.BaseName().LossyDisplayName();
  EXPECT_TRUE(primary_label->GetVisible());
  EXPECT_EQ(primary_label->GetText(), target_file_name);

  // Initially, no bytes have been received so `secondary_label` should display
  // `0 B` as there is no knowledge of the total number of bytes expected.
  EXPECT_TRUE(secondary_label->GetVisible());
  EXPECT_EQ(secondary_label->GetText(), u"0 B");

  // Pause the download.
  RightClick(download_chips.at(0));
  EXPECT_TRUE(SelectMenuItemWithCommandId(HoldingSpaceCommandId::kPauseItem));
  PressAndReleaseKey(ui::VKEY_RETURN);

  // When paused with no bytes received, the `secondary_label` should display
  // "Paused, 0 B" as there is still no knowledge of the total number of
  // bytes expected.
  EXPECT_TRUE(primary_label->GetVisible());
  EXPECT_EQ(primary_label->GetText(), target_file_name);
  EXPECT_TRUE(secondary_label->GetVisible());
  EXPECT_EQ(secondary_label->GetText(), u"Paused, 0 B");

  // Mock `download::DownloadItem::GetReceivedBytes()`.
  ON_CALL(*in_progress_download, GetReceivedBytes)
      .WillByDefault(testing::Return(1024 * 1024));
  in_progress_download->NotifyObserversDownloadUpdated();

  // When paused with bytes received, the `secondary_label` should display both
  // the paused state and the number of bytes received with appropriate units.
  EXPECT_TRUE(primary_label->GetVisible());
  EXPECT_EQ(primary_label->GetText(), target_file_name);
  EXPECT_TRUE(secondary_label->GetVisible());
  EXPECT_EQ(secondary_label->GetText(), u"Paused, 1,024 KB");

  // Resume the download.
  RightClick(download_chips.at(0));
  EXPECT_TRUE(SelectMenuItemWithCommandId(HoldingSpaceCommandId::kResumeItem));
  PressAndReleaseKey(ui::VKEY_RETURN);

  // If resumed with bytes received, the `secondary_label` should display only
  // the number of bytes received with appropriate units.
  EXPECT_TRUE(primary_label->GetVisible());
  EXPECT_EQ(primary_label->GetText(), target_file_name);
  EXPECT_TRUE(secondary_label->GetVisible());
  EXPECT_EQ(secondary_label->GetText(), u"1,024 KB");

  // Mock `download::DownloadItem::GetTotalBytes()`.
  ON_CALL(*in_progress_download, GetTotalBytes)
      .WillByDefault(testing::Return(2 * 1024 * 1024));
  in_progress_download->NotifyObserversDownloadUpdated();

  // If both the number of bytes received and the total number of bytes expected
  // are known, the `secondary_label` should display both with appropriate
  // units.
  EXPECT_TRUE(primary_label->GetVisible());
  EXPECT_EQ(primary_label->GetText(), target_file_name);
  EXPECT_TRUE(secondary_label->GetVisible());
  EXPECT_EQ(secondary_label->GetText(), u"1.0/2.0 MB");

  // Pause the download.
  RightClick(download_chips.at(0));
  EXPECT_TRUE(SelectMenuItemWithCommandId(HoldingSpaceCommandId::kPauseItem));
  PressAndReleaseKey(ui::VKEY_RETURN);

  // If paused with both the number of bytes received and the total number of
  // bytes expected known, the `secondary_label` should display the paused state
  // and both received and total bytes with appropriate units.
  EXPECT_TRUE(primary_label->GetVisible());
  EXPECT_EQ(primary_label->GetText(), target_file_name);
  EXPECT_TRUE(secondary_label->GetVisible());
  EXPECT_EQ(secondary_label->GetText(), u"Paused, 1.0/2.0 MB");

  // Complete the download.
  ON_CALL(*in_progress_download, GetState())
      .WillByDefault(testing::Return(download::DownloadItem::COMPLETE));
  in_progress_download->NotifyObserversDownloadUpdated();

  // When no longer in progress, the `secondary_label` should be hidden.
  EXPECT_TRUE(primary_label->GetVisible());
  EXPECT_EQ(primary_label->GetText(), target_file_name);
  EXPECT_FALSE(secondary_label->GetVisible());
}

// Verifies that canceling holding space items works as intended.
IN_PROC_BROWSER_TEST_F(HoldingSpaceUiInProgressDownloadsBrowserTest,
                       CancelItem) {
  ui::ScopedAnimationDurationScaleMode scoped_animation_duration_scale_mode(
      ui::ScopedAnimationDurationScaleMode::ZERO_DURATION);

  // Create an in-progress download.
  auto in_progress_download = CreateMockDownloadItem(
      download::DownloadItem::IN_PROGRESS, /*file_path=*/CreateFile(),
      /*target_file_path=*/CreateFile(),
      /*percent_complete=*/0);
  in_progress_download->NotifyObserversDownloadUpdated();

  // Create a completed download.
  // NOTE: In production, the download manager will create COMPLETE download
  // items from previous sessions during initialization, so we ignore them. To
  // match production behavior, create an IN_PROGRESS download item and only
  // then update it to COMPLETE state.
  auto completed_download = CreateMockDownloadItem(
      download::DownloadItem::IN_PROGRESS, /*file_path=*/CreateFile(),
      /*target_file_path=*/CreateFile(), /*percent_complete=*/0);
  ON_CALL(*completed_download, GetState())
      .WillByDefault(testing::Return(download::DownloadItem::COMPLETE));
  ON_CALL(*completed_download, PercentComplete())
      .WillByDefault(testing::Return(100));
  completed_download->NotifyObserversDownloadUpdated();

  // Show holding space UI.
  test_api().Show();
  ASSERT_TRUE(test_api().IsShowing());

  // Expect two download chips, one for each created download item.
  std::vector<views::View*> download_chips = test_api().GetDownloadChips();
  ASSERT_EQ(download_chips.size(), 2u);

  // Cache download chips. NOTE: Chips are displayed in reverse order of their
  // underlying holding space item creation.
  views::View* const completed_download_chip = download_chips.at(0);
  views::View* const in_progress_download_chip = download_chips.at(1);

  // Right click the `completed_download_chip`. Because the underlying download
  // is completed, the context menu should *not* contain a "Cancel" command.
  RightClick(completed_download_chip);
  ASSERT_FALSE(SelectMenuItemWithCommandId(HoldingSpaceCommandId::kCancelItem));

  // Close the context menu and control-right click the
  // `in_progress_download_chip`. Because the `completed_download_chip` is still
  // selected and its underlying download is completed, the context menu should
  // *not* contain a "Cancel" command.
  PressAndReleaseKey(ui::KeyboardCode::VKEY_ESCAPE);
  RightClick(in_progress_download_chip, ui::EF_CONTROL_DOWN);
  ASSERT_FALSE(SelectMenuItemWithCommandId(HoldingSpaceCommandId::kCancelItem));

  // Close the context menu, press the `in_progress_download_chip` and then
  // right click it. Because the `in_progress_download_chip` is the only chip
  // selected and its underlying download is in-progress, the context menu
  // should contain a "Cancel" command.
  PressAndReleaseKey(ui::KeyboardCode::VKEY_ESCAPE);
  Click(in_progress_download_chip);
  RightClick(in_progress_download_chip);
  ASSERT_TRUE(SelectMenuItemWithCommandId(HoldingSpaceCommandId::kCancelItem));

  // Cache the holding space item IDs associated with the two download chips.
  const std::string completed_download_id =
      test_api().GetHoldingSpaceItemId(completed_download_chip);
  const std::string in_progress_download_id =
      test_api().GetHoldingSpaceItemId(in_progress_download_chip);

  // Bind an observer to watch for updates to the holding space model.
  testing::NiceMock<MockHoldingSpaceModelObserver> mock;
  base::ScopedObservation<HoldingSpaceModel, HoldingSpaceModelObserver>
      observer{&mock};
  observer.Observe(HoldingSpaceController::Get()->model());

  // Press ENTER to execute the "Cancel" command, expecting and waiting for
  // the in-progress download item to be removed from the holding space model.
  base::RunLoop run_loop;
  EXPECT_CALL(mock, OnHoldingSpaceItemsRemoved)
      .WillOnce([&](const std::vector<const HoldingSpaceItem*>& items) {
        ASSERT_EQ(items.size(), 1u);
        ASSERT_EQ(items[0]->id(), in_progress_download_id);
        run_loop.Quit();
      });
  PressAndReleaseKey(ui::KeyboardCode::VKEY_RETURN);
  run_loop.Run();

  // Verify that there is now only a single download chip.
  download_chips = test_api().GetDownloadChips();
  EXPECT_EQ(download_chips.size(), 1u);

  // Because the in-progress download was canceled, only the completed download
  // chip should still be present in the UI.
  EXPECT_TRUE(test_api().GetHoldingSpaceItemView(download_chips,
                                                 completed_download_id));
  EXPECT_FALSE(test_api().GetHoldingSpaceItemView(download_chips,
                                                  in_progress_download_id));
}

// Verifies that canceling holding space items via primary action is WAI.
IN_PROC_BROWSER_TEST_F(HoldingSpaceUiInProgressDownloadsBrowserTest,
                       CancelItemViaPrimaryAction) {
  ui::ScopedAnimationDurationScaleMode scoped_animation_duration_scale_mode(
      ui::ScopedAnimationDurationScaleMode::ZERO_DURATION);

  // Create an in-progress download.
  auto in_progress_download = CreateMockDownloadItem(
      download::DownloadItem::IN_PROGRESS, /*file_path=*/CreateFile(),
      /*target_file_path=*/CreateFile(), /*percent_complete=*/0);
  in_progress_download->NotifyObserversDownloadUpdated();

  // Create a completed download.
  // NOTE: In production, the download manager will create COMPLETE download
  // items from previous sessions during initialization, so we ignore them. To
  // match production behavior, create an IN_PROGRESS download item and only
  // then update it to COMPLETE state.
  auto completed_download = CreateMockDownloadItem(
      download::DownloadItem::IN_PROGRESS, /*file_path=*/CreateFile(),
      /*target_file_path=*/CreateFile(), /*percent_complete=*/0);
  ON_CALL(*completed_download, GetState())
      .WillByDefault(testing::Return(download::DownloadItem::COMPLETE));
  ON_CALL(*completed_download, PercentComplete())
      .WillByDefault(testing::Return(100));
  completed_download->NotifyObserversDownloadUpdated();

  // Show holding space UI.
  test_api().Show();
  ASSERT_TRUE(test_api().IsShowing());

  // Expect two download chips, one for each created download item.
  std::vector<views::View*> download_chips = test_api().GetDownloadChips();
  ASSERT_EQ(download_chips.size(), 2u);

  // Cache download chips. NOTE: Chips are displayed in reverse order of their
  // underlying holding space item creation.
  views::View* const completed_download_chip = download_chips.at(0);
  views::View* const in_progress_download_chip = download_chips.at(1);

  // Hover over the `completed_download_chip`. Because the underlying download
  // is completed, the chip should contain a visible primary action for "Pin".
  MoveMouseTo(completed_download_chip, /*count=*/10);
  auto* primary_action_container = completed_download_chip->GetViewByID(
      kHoldingSpaceItemPrimaryActionContainerId);
  auto* primary_action_cancel =
      primary_action_container->GetViewByID(kHoldingSpaceItemCancelButtonId);
  auto* primary_action_pin =
      primary_action_container->GetViewByID(kHoldingSpaceItemPinButtonId);
  ViewDrawnWaiter().Wait(primary_action_container);
  EXPECT_FALSE(primary_action_cancel->GetVisible());
  EXPECT_TRUE(primary_action_pin->GetVisible());

  // Hover over the `in_progress_download_chip`. Because the underlying download
  // is in-progress, the chip should contain a visible primary action for
  // "Cancel".
  MoveMouseTo(in_progress_download_chip, /*count=*/10);
  primary_action_container = in_progress_download_chip->GetViewByID(
      kHoldingSpaceItemPrimaryActionContainerId);
  primary_action_cancel =
      primary_action_container->GetViewByID(kHoldingSpaceItemCancelButtonId);
  primary_action_pin =
      primary_action_container->GetViewByID(kHoldingSpaceItemPinButtonId);
  ViewDrawnWaiter().Wait(primary_action_container);
  EXPECT_TRUE(primary_action_cancel->GetVisible());
  EXPECT_FALSE(primary_action_pin->GetVisible());

  // Cache the holding space item IDs associated with the two download chips.
  const std::string completed_download_id =
      test_api().GetHoldingSpaceItemId(completed_download_chip);
  const std::string in_progress_download_id =
      test_api().GetHoldingSpaceItemId(in_progress_download_chip);

  // Bind an observer to watch for updates to the holding space model.
  testing::NiceMock<MockHoldingSpaceModelObserver> mock;
  base::ScopedObservation<HoldingSpaceModel, HoldingSpaceModelObserver>
      observer{&mock};
  observer.Observe(HoldingSpaceController::Get()->model());

  // Press the `primary_action_container` to execute "Cancel", expecting and
  // waiting for the in-progress download item to be removed from the holding
  // space model.
  base::RunLoop run_loop;
  EXPECT_CALL(mock, OnHoldingSpaceItemsRemoved)
      .WillOnce([&](const std::vector<const HoldingSpaceItem*>& items) {
        ASSERT_EQ(items.size(), 1u);
        ASSERT_EQ(items[0]->id(), in_progress_download_id);
        run_loop.Quit();
      });
  Click(primary_action_container);
  run_loop.Run();

  // Verify that there is now only a single download chip.
  download_chips = test_api().GetDownloadChips();
  EXPECT_EQ(download_chips.size(), 1u);

  // Because the in-progress download was canceled, only the completed download
  // chip should still be present in the UI.
  EXPECT_TRUE(test_api().GetHoldingSpaceItemView(download_chips,
                                                 completed_download_id));
  EXPECT_FALSE(test_api().GetHoldingSpaceItemView(download_chips,
                                                  in_progress_download_id));
}

// Base class for tests of the pause or resume commands, parameterized by which
// command to use. This will either be `kPauseItem` or `kResumeItem`.
class HoldingSpaceUiPauseOrResumeBrowserTest
    : public HoldingSpaceUiInProgressDownloadsBrowserTest,
      public testing::WithParamInterface<HoldingSpaceCommandId> {
 public:
  HoldingSpaceUiPauseOrResumeBrowserTest() {
    const HoldingSpaceCommandId command_id(GetPauseOrResumeCommandId());
    EXPECT_TRUE(command_id == HoldingSpaceCommandId::kPauseItem ||
                command_id == HoldingSpaceCommandId::kResumeItem);
  }

  // Returns either `kPauseItem` or `kResumeItem` depending on parameterization.
  HoldingSpaceCommandId GetPauseOrResumeCommandId() const { return GetParam(); }
};

INSTANTIATE_TEST_SUITE_P(
    All,
    HoldingSpaceUiPauseOrResumeBrowserTest,
    testing::ValuesIn({HoldingSpaceCommandId::kPauseItem,
                       HoldingSpaceCommandId::kResumeItem}));

// Verifies that pausing or resuming holding space items works as intended.
IN_PROC_BROWSER_TEST_P(HoldingSpaceUiPauseOrResumeBrowserTest,
                       PauseOrResumeItem) {
  // Use zero animation duration so that UI updates are immediate.
  ui::ScopedAnimationDurationScaleMode scoped_animation_duration_scale_mode(
      ui::ScopedAnimationDurationScaleMode::ZERO_DURATION);

  // Create an in-progress download which may or may not be paused depending
  // on parameterization.
  auto in_progress_download = CreateMockDownloadItem(
      download::DownloadItem::IN_PROGRESS, /*file_path=*/CreateFile(),
      /*target_file_path=*/CreateFile(), /*percent_complete=*/0);
  if (GetPauseOrResumeCommandId() == HoldingSpaceCommandId::kResumeItem)
    in_progress_download->Pause();
  in_progress_download->NotifyObserversDownloadUpdated();

  // Create a completed download.
  // NOTE: In production, the download manager will create COMPLETE download
  // items from previous sessions during initialization, so we ignore them. To
  // match production behavior, create an IN_PROGRESS download item and only
  // then update it to COMPLETE state.
  auto completed_download = CreateMockDownloadItem(
      download::DownloadItem::IN_PROGRESS, /*file_path=*/CreateFile(),
      /*target_file_path=*/CreateFile(), /*percent_complete=*/0);
  ON_CALL(*completed_download, GetState())
      .WillByDefault(testing::Return(download::DownloadItem::COMPLETE));
  ON_CALL(*completed_download, PercentComplete())
      .WillByDefault(testing::Return(100));
  completed_download->NotifyObserversDownloadUpdated();

  // Show holding space UI.
  test_api().Show();
  ASSERT_TRUE(test_api().IsShowing());

  // Expect two download chips, one for each created download item.
  std::vector<views::View*> download_chips = test_api().GetDownloadChips();
  ASSERT_EQ(download_chips.size(), 2u);

  // Cache download chips. NOTE: Chips are displayed in reverse order of their
  // underlying holding space item creation.
  views::View* const completed_download_chip = download_chips.at(0);
  views::View* const in_progress_download_chip = download_chips.at(1);

  // Right click the `completed_download_chip`. Because the underlying download
  // is completed, the context menu should *not* contain a "Pause" or "Resume"
  // command.
  RightClick(completed_download_chip);
  ASSERT_FALSE(SelectMenuItemWithCommandId(GetPauseOrResumeCommandId()));

  // Close the context menu and control-right click the
  // `in_progress_download_chip`. Because the `completed_download_chip` is still
  // selected and its underlying download is completed, the context menu should
  // *not* contain a "Pause" or "Resume" command.
  PressAndReleaseKey(ui::KeyboardCode::VKEY_ESCAPE);
  RightClick(in_progress_download_chip, ui::EF_CONTROL_DOWN);
  ASSERT_FALSE(SelectMenuItemWithCommandId(GetPauseOrResumeCommandId()));

  // Close the context menu, press the `in_progress_download_chip` and then
  // right click it. Because the `in_progress_download_chip` is the only chip
  // selected and its underlying download is in-progress, the context menu
  // should contain a "Pause" or "Resume" command.
  PressAndReleaseKey(ui::KeyboardCode::VKEY_ESCAPE);
  Click(in_progress_download_chip);
  RightClick(in_progress_download_chip);
  ASSERT_TRUE(SelectMenuItemWithCommandId(GetPauseOrResumeCommandId()));

  // Bind an observer to watch for updates to the holding space model.
  testing::NiceMock<MockHoldingSpaceModelObserver> mock;
  base::ScopedObservation<HoldingSpaceModel, HoldingSpaceModelObserver>
      observer{&mock};
  observer.Observe(HoldingSpaceController::Get()->model());

  // Press ENTER to execute the "Pause" or "Resume" command, expecting and
  // waiting for the in-progress download item to be updated in the holding
  // space model.
  base::RunLoop run_loop;
  const bool was_paused = in_progress_download->IsPaused();
  EXPECT_CALL(mock, OnHoldingSpaceItemUpdated)
      .WillOnce([&](const HoldingSpaceItem* item) {
        EXPECT_EQ(item->id(),
                  test_api().GetHoldingSpaceItemId(in_progress_download_chip));
        EXPECT_EQ(item->IsPaused(), !was_paused);
        run_loop.Quit();
      });
  PressAndReleaseKey(ui::KeyboardCode::VKEY_RETURN);
  run_loop.Run();

  // Verify that there are still two download chips.
  download_chips = test_api().GetDownloadChips();
  ASSERT_EQ(download_chips.size(), 2u);

  // The two download chips present should still be the original chips for the
  // completed download and the (now paused or resumed) in-progress download.
  EXPECT_EQ(download_chips.at(0), completed_download_chip);
  EXPECT_EQ(download_chips.at(1), in_progress_download_chip);
}

// Verifies that pausing or resuming holding space items via secondary action is
// working as intended.
IN_PROC_BROWSER_TEST_P(HoldingSpaceUiPauseOrResumeBrowserTest,
                       PauseOrResumeItemViaSecondaryAction) {
  // Use zero animation duration so that UI updates are immediate.
  ui::ScopedAnimationDurationScaleMode scoped_animation_duration_scale_mode(
      ui::ScopedAnimationDurationScaleMode::ZERO_DURATION);

  // Create an in-progress download which may or may not be paused depending
  // on parameterization.
  auto in_progress_download = CreateMockDownloadItem(
      download::DownloadItem::IN_PROGRESS, /*file_path=*/CreateFile(),
      /*target_file_path=*/CreateFile(), /*percent_complete=*/0);
  if (GetPauseOrResumeCommandId() == HoldingSpaceCommandId::kResumeItem)
    in_progress_download->Pause();
  in_progress_download->NotifyObserversDownloadUpdated();

  // Create a completed download.
  // NOTE: In production, the download manager will create COMPLETE download
  // items from previous sessions during initialization, so we ignore them. To
  // match production behavior, create an IN_PROGRESS download item and only
  // then update it to COMPLETE state.
  auto completed_download = CreateMockDownloadItem(
      download::DownloadItem::IN_PROGRESS, /*file_path=*/CreateFile(),
      /*target_file_path=*/CreateFile(), /*percent_complete=*/0);
  ON_CALL(*completed_download, GetState())
      .WillByDefault(testing::Return(download::DownloadItem::COMPLETE));
  ON_CALL(*completed_download, PercentComplete())
      .WillByDefault(testing::Return(100));
  completed_download->NotifyObserversDownloadUpdated();

  // Show holding space UI.
  test_api().Show();
  ASSERT_TRUE(test_api().IsShowing());

  // Expect two download chips, one for each created download item.
  std::vector<views::View*> download_chips = test_api().GetDownloadChips();
  ASSERT_EQ(download_chips.size(), 2u);

  // Cache download chips. NOTE: Chips are displayed in reverse order of their
  // underlying holding space item creation.
  views::View* const completed_download_chip = download_chips.at(0);
  views::View* const in_progress_download_chip = download_chips.at(1);

  // Hover over the `completed_download_chip`. Because the underlying download
  // is completed, the chip should not contain a visible secondary action.
  MoveMouseTo(completed_download_chip, /*count=*/10);
  ASSERT_FALSE(completed_download_chip
                   ->GetViewByID(kHoldingSpaceItemSecondaryActionContainerId)
                   ->GetVisible());

  // Hover over the `in_progress_download_chip`. Because the underlying download
  // is in-progress, the chip should contain a visible secondary action for
  // either "Pause" or "Resume", depending on test parameterization.
  MoveMouseTo(in_progress_download_chip, /*count=*/10);
  auto* secondary_action_container = in_progress_download_chip->GetViewByID(
      kHoldingSpaceItemSecondaryActionContainerId);
  auto* secondary_action_pause =
      secondary_action_container->GetViewByID(kHoldingSpaceItemPauseButtonId);
  auto* secondary_action_resume =
      secondary_action_container->GetViewByID(kHoldingSpaceItemResumeButtonId);
  ViewDrawnWaiter().Wait(secondary_action_container);
  EXPECT_EQ(secondary_action_pause->GetVisible(),
            GetPauseOrResumeCommandId() == HoldingSpaceCommandId::kPauseItem);
  EXPECT_EQ(secondary_action_resume->GetVisible(),
            GetPauseOrResumeCommandId() == HoldingSpaceCommandId::kResumeItem);

  // Bind an observer to watch for updates to the holding space model.
  testing::NiceMock<MockHoldingSpaceModelObserver> mock;
  base::ScopedObservation<HoldingSpaceModel, HoldingSpaceModelObserver>
      observer{&mock};
  observer.Observe(HoldingSpaceController::Get()->model());

  // Press the `secondary_action_container` to execute the "Pause" or "Resume"
  // command, expecting and waiting for the in-progress download item to be
  // updated in the holding space model.
  base::RunLoop run_loop;
  const bool was_paused = in_progress_download->IsPaused();
  EXPECT_CALL(mock, OnHoldingSpaceItemUpdated)
      .WillOnce([&](const HoldingSpaceItem* item) {
        EXPECT_EQ(item->id(),
                  test_api().GetHoldingSpaceItemId(in_progress_download_chip));
        EXPECT_EQ(item->IsPaused(), !was_paused);
        run_loop.Quit();
      });
  Click(secondary_action_container);
  run_loop.Run();

  // Verify that there are still two download chips.
  download_chips = test_api().GetDownloadChips();
  ASSERT_EQ(download_chips.size(), 2u);

  // The two download chips present should still be the original chips for the
  // completed download and the (now paused or resumed) in-progress download.
  EXPECT_EQ(download_chips.at(0), completed_download_chip);
  EXPECT_EQ(download_chips.at(1), in_progress_download_chip);
}

// Base class for holding space UI browser tests that take screenshots.
// Parameterized by whether or not `features::CaptureMode` is enabled.
class HoldingSpaceUiScreenshotBrowserTest
    : public HoldingSpaceUiBrowserTest,
      public testing::WithParamInterface<bool> {
 public:
  HoldingSpaceUiScreenshotBrowserTest() {
    scoped_feature_list_.InitWithFeatureState(features::kCaptureMode,
                                              GetParam());
  }

 private:
  base::test::ScopedFeatureList scoped_feature_list_;
};

// Verifies that taking a screenshot adds a screenshot holding space item.
IN_PROC_BROWSER_TEST_P(HoldingSpaceUiScreenshotBrowserTest, AddScreenshot) {
  // Verify that no screenshots exist in holding space UI.
  test_api().Show();
  ASSERT_TRUE(test_api().IsShowing());
  EXPECT_TRUE(test_api().GetScreenCaptureViews().empty());

  test_api().Close();
  ASSERT_FALSE(test_api().IsShowing());

  // Take a screenshot using the keyboard. If `features::kCaptureMode` is
  // enabled, the screenshot will be taken using the `CaptureModeController`.
  // Otherwise the screenshot will be taken using the `ChromeScreenshotGrabber`.
  PressAndReleaseKey(ui::VKEY_MEDIA_LAUNCH_APP1,
                     ui::EF_ALT_DOWN | ui::EF_CONTROL_DOWN);
  // Move the mouse over to the browser window. The reason for that is with
  // `features::kCaptureMode` enabled, the new capture mode implementation will
  // not automatically capture the topmost window unless the mouse is hovered
  // above it.
  aura::Window* browser_window = browser()->window()->GetNativeWindow();
  ui::test::EventGenerator event_generator(browser_window->GetRootWindow());
  event_generator.MoveMouseTo(
      browser_window->GetBoundsInScreen().CenterPoint());
  PressAndReleaseKey(ui::VKEY_RETURN);

  // Bind an observer to watch for updates to the holding space model.
  testing::NiceMock<MockHoldingSpaceModelObserver> mock;
  base::ScopedObservation<HoldingSpaceModel, HoldingSpaceModelObserver>
      observer{&mock};
  observer.Observe(HoldingSpaceController::Get()->model());

  // Expect and wait for a screenshot item to be added to holding space.
  base::RunLoop run_loop;
  EXPECT_CALL(mock, OnHoldingSpaceItemsAdded)
      .WillOnce([&](const std::vector<const HoldingSpaceItem*>& items) {
        ASSERT_EQ(items.size(), 1u);
        ASSERT_EQ(items[0]->type(), HoldingSpaceItem::Type::kScreenshot);
        run_loop.Quit();
      });
  run_loop.Run();

  // Verify that the screenshot appears in holding space UI.
  test_api().Show();
  ASSERT_TRUE(test_api().IsShowing());
  EXPECT_EQ(1u, test_api().GetScreenCaptureViews().size());
}

// Base class for holding space UI browser tests that take screen recordings.
class HoldingSpaceUiScreenCaptureBrowserTest
    : public HoldingSpaceUiBrowserTest {
 public:
  HoldingSpaceUiScreenCaptureBrowserTest() {
    scoped_feature_list_.InitAndEnableFeature(features::kCaptureMode);
  }

 private:
  base::test::ScopedFeatureList scoped_feature_list_;
};

// Verifies that taking a screen recording adds a screen recording holding space
// item.
IN_PROC_BROWSER_TEST_F(HoldingSpaceUiScreenCaptureBrowserTest,
                       AddScreenRecording) {
  // Verify that no screen recordings exist in holding space UI.
  test_api().Show();
  ASSERT_TRUE(test_api().IsShowing());
  EXPECT_TRUE(test_api().GetScreenCaptureViews().empty());

  test_api().Close();
  ASSERT_FALSE(test_api().IsShowing());
  ash::CaptureModeTestApi capture_mode_test_api;
  capture_mode_test_api.StartForFullscreen(/*for_video=*/true);
  capture_mode_test_api.PerformCapture();
  // Record a 100 ms long video.
  base::RunLoop video_recording_time;
  base::ThreadTaskRunnerHandle::Get()->PostDelayedTask(
      FROM_HERE, video_recording_time.QuitClosure(),
      base::TimeDelta::FromMilliseconds(100));
  video_recording_time.Run();
  capture_mode_test_api.StopVideoRecording();

  // Bind an observer to watch for updates to the holding space model.
  testing::NiceMock<MockHoldingSpaceModelObserver> mock;
  base::ScopedObservation<HoldingSpaceModel, HoldingSpaceModelObserver>
      observer{&mock};
  observer.Observe(HoldingSpaceController::Get()->model());

  base::RunLoop wait_for_item;
  // Expect and wait for a screen recording item to be added to holding space.
  EXPECT_CALL(mock, OnHoldingSpaceItemsAdded)
      .WillOnce([&](const std::vector<const HoldingSpaceItem*>& items) {
        ASSERT_EQ(items.size(), 1u);
        ASSERT_EQ(items[0]->type(), HoldingSpaceItem::Type::kScreenRecording);
        wait_for_item.Quit();
      });
  wait_for_item.Run();

  // Verify that the screen recording appears in holding space UI.
  test_api().Show();
  ASSERT_TRUE(test_api().IsShowing());
  EXPECT_EQ(1u, test_api().GetScreenCaptureViews().size());
}

INSTANTIATE_TEST_SUITE_P(All,
                         HoldingSpaceUiScreenshotBrowserTest,
                         testing::Bool());

}  // namespace ash
