// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
#include "chrome/browser/ui/hats/trust_safety_sentiment_service_factory.h"

#include "chrome/browser/ui/hats/hats_service.h"
#include "chrome/browser/ui/hats/hats_service_factory.h"
#include "chrome/common/chrome_features.h"
#include "components/keyed_service/content/browser_context_dependency_manager.h"
#include "content/public/browser/browser_context.h"

TrustSafetySentimentServiceFactory::TrustSafetySentimentServiceFactory()
    : BrowserContextKeyedServiceFactory(
          "TrustSafetySentimentService",
          BrowserContextDependencyManager::GetInstance()) {
  DependsOn(HatsServiceFactory::GetInstance());
}

TrustSafetySentimentServiceFactory*
TrustSafetySentimentServiceFactory::GetInstance() {
  return base::Singleton<TrustSafetySentimentServiceFactory>::get();
}

TrustSafetySentimentService* TrustSafetySentimentServiceFactory::GetForProfile(
    Profile* profile) {
  return static_cast<TrustSafetySentimentService*>(
      GetInstance()->GetServiceForBrowserContext(profile, /*create=*/true));
}

KeyedService* TrustSafetySentimentServiceFactory::BuildServiceInstanceFor(
    content::BrowserContext* context) const {
  if (context->IsOffTheRecord() ||
      !base::FeatureList::IsEnabled(features::kTrustSafetySentimentSurvey)) {
    return nullptr;
  }
  Profile* profile = Profile::FromBrowserContext(context);

  // If there is no HaTS service, or the HaTS service reports the user is not
  // eligible to be surveyed by HaTS, do not create the service. This state is
  // unlikely to change over the life of the profile (e.g. before closing
  // Chrome) and simply not creating the service avoids unnecessary work
  // tracking user interactions.
  auto* hats_service =
      HatsServiceFactory::GetForProfile(profile, /*create_if_necessary=*/true);
  if (!hats_service ||
      !hats_service->CanShowAnySurvey(/*user_prompted=*/false)) {
    return nullptr;
  }

  return new TrustSafetySentimentService(profile);
}
