// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/chromeos/policy/handlers/device_name_policy_handler_impl.h"

#include "base/test/task_environment.h"
#include "chrome/browser/ash/settings/scoped_testing_cros_settings.h"
#include "chrome/browser/ash/settings/stub_cros_settings_provider.h"
#include "chromeos/network/network_handler_test_helper.h"
#include "chromeos/system/fake_statistics_provider.h"
#include "chromeos/tpm/stub_install_attributes.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace policy {
namespace {

class FakeObserver : public DeviceNamePolicyHandler::Observer {
 public:
  FakeObserver() = default;
  ~FakeObserver() override = default;

  size_t num_calls() const { return num_calls_; }

  // DeviceNamePolicyHandler::Observer:
  void OnHostnamePolicyChanged() override { ++num_calls_; }

 private:
  size_t num_calls_ = 0;
};

}  // namespace

class DeviceNamePolicyHandlerImplTest : public testing::Test {
 public:
  DeviceNamePolicyHandlerImplTest() = default;

  // testing::Test
  void SetUp() override {
    testing::Test::SetUp();
    network_handler_test_helper_ =
        std::make_unique<chromeos::NetworkHandlerTestHelper>();
  }

  void TearDown() override {
    handler_->RemoveObserver(&fake_observer_);
    handler_.reset();
  }

  // Sets kDeviceHostnameTemplate policy which will eventually cause
  // OnDeviceHostnamePropertyChangedAndMachineStatisticsLoaded() to run
  // and set the hostname_.
  void SetTemplate(const std::string& hostname_template) {
    scoped_testing_cros_settings_.device_settings()->Set(
        chromeos::kDeviceHostnameTemplate, base::Value(hostname_template));
    // Makes sure that template is set before continuing.
    base::RunLoop().RunUntilIdle();
  }

  void UnsetTemplate() {
    scoped_testing_cros_settings_.device_settings()->Set(
        chromeos::kDeviceHostnameTemplate, base::Value());
    // Makes sure that template is set before continuing.
    base::RunLoop().RunUntilIdle();
  }

  void InitializeHandler() {
    handler_ = base::WrapUnique(new DeviceNamePolicyHandlerImpl(
        ash::CrosSettings::Get(), &fake_statistics_provider_));
    handler_->AddObserver(&fake_observer_);
    base::RunLoop().RunUntilIdle();
  }

  size_t GetNumObserverCalls() const { return fake_observer_.num_calls(); }

  std::unique_ptr<DeviceNamePolicyHandler> handler_;

 private:
  base::test::TaskEnvironment task_environment_;
  std::unique_ptr<chromeos::NetworkHandlerTestHelper>
      network_handler_test_helper_;
  ash::ScopedTestingCrosSettings scoped_testing_cros_settings_;
  ash::ScopedStubInstallAttributes test_install_attributes_;
  chromeos::system::ScopedFakeStatisticsProvider fake_statistics_provider_;
  FakeObserver fake_observer_;
};

// Check that device name policy set to kNoPolicy by default.
TEST_F(DeviceNamePolicyHandlerImplTest, DeviceNamePolicyUnmanaged) {
  InitializeHandler();
  DeviceNamePolicyHandler::DeviceNamePolicy initial =
      handler_->GetDeviceNamePolicy();
  EXPECT_EQ(DeviceNamePolicyHandler::DeviceNamePolicy::kNoPolicy, initial);

  // GetHostnameChosenByAdministrator() should therefore return null.
  const absl::optional<std::string> hostname =
      handler_->GetHostnameChosenByAdministrator();
  EXPECT_FALSE(hostname);
}

// Check outputs are correct when device name policy is set to
// kPolicyHostnameChosenByAdmin.
TEST_F(DeviceNamePolicyHandlerImplTest, HostnameChosenByAdministrator) {
  InitializeHandler();
  // Check that DeviceNamePolicy changes from kNoPolicy to
  // kPolicyHostnameChosenByAdmin on setting template.
  DeviceNamePolicyHandler::DeviceNamePolicy initial =
      handler_->GetDeviceNamePolicy();
  EXPECT_EQ(DeviceNamePolicyHandler::DeviceNamePolicy::kNoPolicy, initial);
  const std::string hostname_template = "chromebook";
  SetTemplate(hostname_template);
  DeviceNamePolicyHandler::DeviceNamePolicy after =
      handler_->GetDeviceNamePolicy();
  EXPECT_EQ(
      DeviceNamePolicyHandler::DeviceNamePolicy::kPolicyHostnameChosenByAdmin,
      after);

  // Check that GetDeviceHostname() and GetHostnameChosenByAdministrator()
  // both return the expected hostname value.
  const absl::optional<std::string> hostname_chosen_by_administrator =
      handler_->GetHostnameChosenByAdministrator();
  const std::string device_hostname = handler_->GetDeviceHostname();
  EXPECT_EQ(hostname_chosen_by_administrator, "chromebook");
  EXPECT_EQ(device_hostname, "chromebook");
}

// Check that OnHostnamePolicyChanged() correctly notifies observer when
// hostname and/or policy changes.
TEST_F(DeviceNamePolicyHandlerImplTest, HostnameAndOrPolicyChanged) {
  InitializeHandler();
  // Neither hostname or policy changes on initialization of handler
  EXPECT_EQ(0u, GetNumObserverCalls());

  // Both hostname and policy change, hence observer should be notified once
  EXPECT_EQ(DeviceNamePolicyHandler::DeviceNamePolicy::kNoPolicy,
            handler_->GetDeviceNamePolicy());
  EXPECT_FALSE(handler_->GetHostnameChosenByAdministrator());
  std::string hostname_template = "template1";
  SetTemplate(hostname_template);
  EXPECT_EQ(
      DeviceNamePolicyHandler::DeviceNamePolicy::kPolicyHostnameChosenByAdmin,
      handler_->GetDeviceNamePolicy());
  EXPECT_EQ(hostname_template, handler_->GetHostnameChosenByAdministrator());
  EXPECT_EQ(1u, GetNumObserverCalls());

  // Hostname changes every time, hence observer should be notified each time.
  hostname_template = "template2";
  SetTemplate(hostname_template);
  EXPECT_EQ(hostname_template, handler_->GetHostnameChosenByAdministrator());
  EXPECT_EQ(2u, GetNumObserverCalls());
  hostname_template = "template3";
  SetTemplate(hostname_template);
  EXPECT_EQ(hostname_template, handler_->GetHostnameChosenByAdministrator());
  EXPECT_EQ(3u, GetNumObserverCalls());

  // Hostname is unchanged, hence observer should not be notified.
  const std::string const_template = "const_template";
  SetTemplate(const_template);
  EXPECT_EQ(const_template, handler_->GetHostnameChosenByAdministrator());
  EXPECT_EQ(4u, GetNumObserverCalls());
  SetTemplate(const_template);
  EXPECT_EQ(const_template, handler_->GetHostnameChosenByAdministrator());
  EXPECT_EQ(4u, GetNumObserverCalls());
  SetTemplate(const_template);
  EXPECT_EQ(const_template, handler_->GetHostnameChosenByAdministrator());
  EXPECT_EQ(4u, GetNumObserverCalls());

  // Policy changes every time, hence observer should be notified each time.
  UnsetTemplate();
  EXPECT_EQ(DeviceNamePolicyHandler::DeviceNamePolicy::kNoPolicy,
            handler_->GetDeviceNamePolicy());
  EXPECT_EQ(5u, GetNumObserverCalls());
  SetTemplate(hostname_template);
  EXPECT_EQ(
      DeviceNamePolicyHandler::DeviceNamePolicy::kPolicyHostnameChosenByAdmin,
      handler_->GetDeviceNamePolicy());
  EXPECT_EQ(6u, GetNumObserverCalls());
  UnsetTemplate();
  EXPECT_EQ(DeviceNamePolicyHandler::DeviceNamePolicy::kNoPolicy,
            handler_->GetDeviceNamePolicy());
  EXPECT_EQ(7u, GetNumObserverCalls());
}

}  // namespace policy
