// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "base/path_service.h"
#include "chrome/browser/ui/views/web_apps/web_app_integration_browsertest_base.h"
#include "chrome/browser/web_applications/components/os_integration_manager.h"
#include "chrome/test/base/in_process_browser_test.h"
#include "content/public/test/browser_test.h"
#include "services/network/public/cpp/network_switches.h"

namespace web_app {

namespace {

const std::string kTestCaseFileName =
    "web_app_integration_browsertest_cases.csv";

// Returns the path of the requested file in the test data directory.
base::FilePath GetTestFileDir() {
  base::FilePath file_path;
  base::PathService::Get(base::DIR_SOURCE_ROOT, &file_path);
  file_path = file_path.Append(FILE_PATH_LITERAL("chrome"));
  file_path = file_path.Append(FILE_PATH_LITERAL("test"));
  file_path = file_path.Append(FILE_PATH_LITERAL("data"));
  return file_path.Append(FILE_PATH_LITERAL("web_apps"));
}

std::vector<std::string> BuildAllPlatformTestCaseSet() {
  return WebAppIntegrationBrowserTestBase::BuildAllPlatformTestCaseSet(
      GetTestFileDir(), kTestCaseFileName);
}

}  // anonymous namespace

class WebAppIntegrationBrowserTest
    : public InProcessBrowserTest,
      public WebAppIntegrationBrowserTestBase::TestDelegate,
      public testing::WithParamInterface<std::string> {
 public:
  WebAppIntegrationBrowserTest() : helper_(this) {}

  // InProcessBrowserTest
  void SetUp() override {
    helper_.SetUp(GetChromeTestDataDir());
    InProcessBrowserTest::SetUp();
  }

  // BrowserTestBase
  void SetUpOnMainThread() override { helper_.SetUpOnMainThread(); }

  void SetUpCommandLine(base::CommandLine* command_line) override {
    ASSERT_TRUE(embedded_test_server()->Start());
    command_line->AppendSwitchASCII(
        network::switches::kUnsafelyTreatInsecureOriginAsSecure,
        helper_.GetInstallableAppURL("site_a").GetOrigin().spec());
    command_line->AppendSwitchASCII(
        network::switches::kUnsafelyTreatInsecureOriginAsSecure,
        helper_.GetInstallableAppURL("site_b").GetOrigin().spec());
    command_line->AppendSwitchASCII(
        network::switches::kUnsafelyTreatInsecureOriginAsSecure,
        helper_.GetInstallableAppURL("site_c").GetOrigin().spec());
    command_line->AppendSwitchASCII(
        network::switches::kUnsafelyTreatInsecureOriginAsSecure,
        helper_.GetInstallableAppURL("site_a_foo").GetOrigin().spec());
    command_line->AppendSwitchASCII(
        network::switches::kUnsafelyTreatInsecureOriginAsSecure,
        helper_.GetInstallableAppURL("site_a_bar").GetOrigin().spec());
  }

  // WebAppIntegrationBrowserTestBase::TestDelegate
  Browser* CreateBrowser(Profile* profile) override {
    return InProcessBrowserTest::CreateBrowser(profile);
  }

  void AddBlankTabAndShow(Browser* browser) override {
    InProcessBrowserTest::AddBlankTabAndShow(browser);
  }

  net::EmbeddedTestServer* EmbeddedTestServer() override {
    return embedded_test_server();
  }

  std::vector<Profile*> GetAllProfiles() override {
    return std::vector<Profile*>{browser()->profile()};
  }

  bool IsSyncTest() override { return false; }

  bool UserSigninInternal() override {
    NOTREACHED();
    return false;
  }
  void TurnSyncOff() override { NOTREACHED(); }
  void TurnSyncOn() override { NOTREACHED(); }

  WebAppIntegrationBrowserTestBase helper_;
};

// This test is a part of the web app integration test suite, which is
// documented in //chrome/browser/ui/views/web_apps/README.md. For information
// about diagnosing, debugging and/or disabling tests, please look to the
// README file.
IN_PROC_BROWSER_TEST_P(WebAppIntegrationBrowserTest, Default) {
  helper_.ParseParams(GetParam());
  // Since this test framework differs from traditional browser tests, print
  // some useful information for sheriffs and developers to help identify,
  // diagnose, and disable failing tests.
  LOG(INFO) << helper_.BuildLogForTest(helper_.testing_actions(), IsSyncTest());

  for (auto& action : helper_.testing_actions()) {
    helper_.ExecuteAction(action);
  }
  helper_.ResetRegistrarObserver();
}

INSTANTIATE_TEST_SUITE_P(All,
                         WebAppIntegrationBrowserTest,
                         testing::ValuesIn(BuildAllPlatformTestCaseSet()));

// TODO(jarrydg@chromium.org): Remove the macro disabling the following tests
// when they can compile. https://crbug.com/1215791
#if false
namespace {

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstOmniboxSiteA_WindowCreated_InstPlcyTabShctSiteA_NavSiteA_LaunchIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallOmniboxIcon("SiteA");
  CheckWindowCreated();
  InstallPolicyAppTabbedShortcut("SiteA");
  NavigateBrowser("SiteA");
  CheckLaunchIconShown();
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstOmniboxSiteA_WindowCreated_InstPlcyTabShctSiteA_UninstallPlcySiteA) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallOmniboxIcon("SiteA");
  CheckWindowCreated();
  InstallPolicyAppTabbedShortcut("SiteA");
  UninstallPolicyApp("SiteA");
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstOmniboxSiteA_WindowCreated_NavSiteA_InstIconNotShown_LaunchIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallOmniboxIcon("SiteA");
  CheckWindowCreated();
  NavigateBrowser("SiteA");
  CheckInstallIconNotShown();
  CheckLaunchIconShown();
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstOmniboxSiteA_WindowCreated_ClosePWA_MnfstUpdateDsplMinimalSiteA) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallOmniboxIcon("SiteA");
  CheckWindowCreated();
  ClosePwa();
  ManifestUpdateDisplayMinimal("SiteA");
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstOmniboxSiteA_WindowCreated_SetOpenTabSiteA_NavSiteA_InstIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallOmniboxIcon("SiteA");
  CheckWindowCreated();
  SetOpenInTab("SiteA");
  NavigateBrowser("SiteA");
  CheckInstallIconShown();
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstOmniboxSiteA_WindowCreated_NavSiteAFoo_InstIconNotShown_LaunchIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallOmniboxIcon("SiteA");
  CheckWindowCreated();
  NavigateBrowser("SiteAFoo");
  CheckInstallIconNotShown();
  CheckLaunchIconShown();
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstOmniboxSiteA_WindowCreated_InstPlcyWinShctSiteA_UninstallPlcySiteA) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallOmniboxIcon("SiteA");
  CheckWindowCreated();
  InstallPolicyAppWindowedShortcut("SiteA");
  UninstallPolicyApp("SiteA");
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstMenuOptionSiteA_WindowCreated_InstPlcyTabShctSiteA_NavSiteA_LaunchIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallMenuOption("SiteA");
  CheckWindowCreated();
  InstallPolicyAppTabbedShortcut("SiteA");
  NavigateBrowser("SiteA");
  CheckLaunchIconShown();
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstMenuOptionSiteA_WindowCreated_InstPlcyTabShctSiteA_UninstallPlcySiteA) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallMenuOption("SiteA");
  CheckWindowCreated();
  InstallPolicyAppTabbedShortcut("SiteA");
  UninstallPolicyApp("SiteA");
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstMenuOptionSiteA_WindowCreated_NavSiteA_InstIconNotShown_LaunchIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallMenuOption("SiteA");
  CheckWindowCreated();
  NavigateBrowser("SiteA");
  CheckInstallIconNotShown();
  CheckLaunchIconShown();
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstMenuOptionSiteA_WindowCreated_ClosePWA_MnfstUpdateDsplMinimalSiteA) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallMenuOption("SiteA");
  CheckWindowCreated();
  ClosePwa();
  ManifestUpdateDisplayMinimal("SiteA");
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstMenuOptionSiteA_WindowCreated_SetOpenTabSiteA_NavSiteA_InstIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallMenuOption("SiteA");
  CheckWindowCreated();
  SetOpenInTab("SiteA");
  NavigateBrowser("SiteA");
  CheckInstallIconShown();
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstMenuOptionSiteA_WindowCreated_NavSiteAFoo_InstIconNotShown_LaunchIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallMenuOption("SiteA");
  CheckWindowCreated();
  NavigateBrowser("SiteAFoo");
  CheckInstallIconNotShown();
  CheckLaunchIconShown();
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstMenuOptionSiteA_WindowCreated_InstPlcyWinShctSiteA_UninstallPlcySiteA) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallMenuOption("SiteA");
  CheckWindowCreated();
  InstallPolicyAppWindowedShortcut("SiteA");
  UninstallPolicyApp("SiteA");
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstPlcyWinShctSiteA_UninstallPlcySiteA_NotInListSiteA) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallPolicyAppWindowedShortcut("SiteA");
  UninstallPolicyApp("SiteA");
  CheckAppNotInList("SiteA");
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstPlcyWinShctSiteA_NavSiteA_InstIconNotShown_LaunchIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallPolicyAppWindowedShortcut("SiteA");
  NavigateBrowser("SiteA");
  CheckInstallIconNotShown();
  CheckLaunchIconShown();
}

IN_PROC_BROWSER_TEST(WebAppIntegrationBrowserTest,
                     WebAppIntegration_InstPlcyWinShctSiteA_SetOpenTabSiteA) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallPolicyAppWindowedShortcut("SiteA");
  SetOpenInTab("SiteA");
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstPlcyWinShctSiteA_NavSiteAFoo_InstIconNotShown_LaunchIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallPolicyAppWindowedShortcut("SiteA");
  NavigateBrowser("SiteAFoo");
  CheckInstallIconNotShown();
  CheckLaunchIconShown();
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstPlcyTabShctSiteA_UninstallPlcySiteA_NotInListSiteA_NavSiteA_InstIconShown_LaunchIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallPolicyAppTabbedShortcut("SiteA");
  UninstallPolicyApp("SiteA");
  CheckAppNotInList("SiteA");
  NavigateBrowser("SiteA");
  CheckInstallIconShown();
  CheckLaunchIconNotShown();
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstPlcyTabShctSiteA_InstOmniboxSiteA_WindowCreated_UninstallPlcySiteA) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallPolicyAppTabbedShortcut("SiteA");
  InstallOmniboxIcon("SiteA");
  CheckWindowCreated();
  UninstallPolicyApp("SiteA");
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstPlcyTabShctSiteA_InstMenuOptionSiteA_WindowCreated_UninstallPlcySiteA) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallPolicyAppTabbedShortcut("SiteA");
  InstallMenuOption("SiteA");
  CheckWindowCreated();
  UninstallPolicyApp("SiteA");
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstPlcyTabShctSiteA_NavSiteA_InstIconShown_LaunchIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallPolicyAppTabbedShortcut("SiteA");
  NavigateBrowser("SiteA");
  CheckInstallIconShown();
  CheckLaunchIconNotShown();
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstPlcyTabShctSiteA_SetOpenWindowSiteA) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallPolicyAppTabbedShortcut("SiteA");
  SetOpenInWindow("SiteA");
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstPlcyTabShctSiteC_NavSiteC_InstIconNotShown_LaunchIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallPolicyAppTabbedShortcut("SiteC");
  NavigateBrowser("SiteC");
  CheckInstallIconNotShown();
  CheckLaunchIconNotShown();
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstPlcyWinShctSiteC_NavSiteC_InstIconNotShown_LaunchIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallPolicyAppWindowedShortcut("SiteC");
  NavigateBrowser("SiteC");
  CheckInstallIconNotShown();
  CheckLaunchIconShown();
}

IN_PROC_BROWSER_TEST(WebAppIntegrationBrowserTest,
                     WebAppIntegration_InstOmniboxSiteAFoo_ClosePWA) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallOmniboxIcon("SiteAFoo");
  ClosePwa();
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstOmniboxSiteAFoo_NavSiteABar_InstIconShown_LaunchIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallOmniboxIcon("SiteAFoo");
  NavigateBrowser("SiteABar");
  CheckInstallIconShown();
  CheckLaunchIconNotShown();
}

IN_PROC_BROWSER_TEST(WebAppIntegrationBrowserTest,
                     WebAppIntegration_InstMenuOptionSiteAFoo_ClosePWA) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallMenuOption("SiteAFoo");
  ClosePwa();
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstMenuOptionSiteAFoo_NavSiteABar_InstIconShown_LaunchIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallMenuOption("SiteAFoo");
  NavigateBrowser("SiteABar");
  CheckInstallIconShown();
  CheckLaunchIconNotShown();
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_InstPlcyWinShctSiteAFoo_NavSiteABar_InstIconShown_LaunchIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallPolicyAppWindowedShortcut("SiteAFoo");
  NavigateBrowser("SiteABar");
  CheckInstallIconShown();
  CheckLaunchIconNotShown();
}

IN_PROC_BROWSER_TEST(WebAppIntegrationBrowserTest,
                     WebAppIntegration_NavSiteAFoo_InstIconShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  NavigateBrowser("SiteAFoo");
  CheckInstallIconShown();
}

IN_PROC_BROWSER_TEST(WebAppIntegrationBrowserTest,
                     WebAppIntegration_NavSiteA_NotInListSiteA) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  NavigateBrowser("SiteA");
  CheckAppNotInList("SiteA");
}

IN_PROC_BROWSER_TEST(
    WebAppIntegrationBrowserTest,
    WebAppIntegration_NavSiteC_NotInListSiteA_InstIconNotShown) {
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  NavigateBrowser("SiteC");
  CheckAppNotInList("SiteA");
  CheckInstallIconNotShown();
}

IN_PROC_BROWSERTEST_F(WebAppIntegrationBrowserTest, WebAppIntegration_InstOmniboxSiteA_WindowCreated_NavSiteB_InstIconShown_LaunchIconShown){
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallOmniboxIcon("SiteA");
  CheckWindowCreated();
  NavigateBrowser("SiteB");
  CheckInstallIconShown();
  CheckLaunchIconNotShown();
}

IN_PROC_BROWSERTEST_F(WebAppIntegrationBrowserTest, WebAppIntegration_InstMenuOptionSiteA_WindowCreated_NavSiteB_InstIconShown_LaunchIconShown){
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallMenuOption("SiteA");
  CheckWindowCreated();
  NavigateBrowser("SiteB");
  CheckInstallIconShown();
  CheckLaunchIconNotShown();
}

IN_PROC_BROWSERTEST_F(WebAppIntegrationBrowserTest, WebAppIntegration_InstOmniboxSiteB_NavSiteB_LaunchIconShown){
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallOmniboxIcon("SiteB");
  NavigateBrowser("SiteB");
  CheckLaunchIconShown();
}

IN_PROC_BROWSERTEST_F(WebAppIntegrationBrowserTest, WebAppIntegration_InstPlcyWinShctSiteB_NavSiteB_LaunchIconShown){
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallPolicyAppWindowedShortcut("SiteB");
  NavigateBrowser("SiteB");
  CheckLaunchIconShown();
}

IN_PROC_BROWSERTEST_F(WebAppIntegrationBrowserTest, WebAppIntegration_InstMenuOptionSiteB_NavSiteB_LaunchIconShown){
  // Test contents are generated by script. Please do not modify!
  // See `chrome/test/webapps/README.md` for more info.
  // Sheriffs: Disabling this test is supported.
  InstallMenuOption("SiteB");
  NavigateBrowser("SiteB");
  CheckLaunchIconShown();
}

}  // namespace
#endif

}  // namespace web_app
