// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

/**
 * @fileoverview Polymer element for displaying material design management
 * transition screen.
 */

(function() {

const UIState = {
  PROGRESS: 'progress',
  ERROR: 'error',
};

/**
 * Possible transition types. Must be in the same order as
 * ArcSupervisionTransition enum values.
 * @enum {number}
 */
const ARC_SUPERVISION_TRANSITION = {
  NO_TRANSITION: 0,
  CHILD_TO_REGULAR: 1,
  REGULAR_TO_CHILD: 2,
  UNMANAGED_TO_MANAGED: 3,
};

Polymer({
  is: 'management-transition-element',

  behaviors: [OobeI18nBehavior, LoginScreenBehavior, MultiStepBehavior],

  properties: {
    /**
     * Property that determines transition direction.
     */
    arcTransition_: Number,
    /**
     * String that represents management entity for the user. Can be domain or
     * admin name.
     */
    managementEntity_: String,
  },

  UI_STEPS: UIState,

  defaultUIStep() {
    return UIState.PROGRESS;
  },

  ready() {
    this.initializeLoginScreen('ManagementTransitionScreen', {
      resetAllowed: false,
    });
  },

  onBeforeShow(data) {
    this.setArcTransition(data['arcTransition']);
    this.setManagementEntity(data['managementEntity']);
  },

  setArcTransition(arc_transition) {
    switch (arc_transition) {
      case ARC_SUPERVISION_TRANSITION.CHILD_TO_REGULAR:
      case ARC_SUPERVISION_TRANSITION.REGULAR_TO_CHILD:
      case ARC_SUPERVISION_TRANSITION.UNMANAGED_TO_MANAGED:
        this.arcTransition_ = arc_transition;
        break;
      case ARC_SUPERVISION_TRANSITION.NO_TRANSITION:
        console.error(
            'Screen should not appear for ARC_SUPERIVISION_TRANSITION.NO_TRANSITION');
        break;
      default:
        console.error('Not handled transition type: ' + arc_transition);
    }
  },

  setManagementEntity(management_entity) {
    this.managementEntity_ = management_entity;
  },

  /** @override */
  attached() {
    cr.addWebUIListener(
        'management-transition-failed',
        this.showManagementTransitionFailedScreen_.bind(this));
  },

  /** @private */
  getDialogTitle_(locale, arcTransition, managementEntity) {
    switch (arcTransition) {
      case ARC_SUPERVISION_TRANSITION.CHILD_TO_REGULAR:
        return this.i18n('removingSupervisionTitle');
      case ARC_SUPERVISION_TRANSITION.REGULAR_TO_CHILD:
        return this.i18n('addingSupervisionTitle');
      case ARC_SUPERVISION_TRANSITION.UNMANAGED_TO_MANAGED:
        if (managementEntity) {
          return this.i18n('addingManagementTitle', managementEntity);
        } else {
          return this.i18n('addingManagementTitleUnknownAdmin');
        }
    }
  },

  /** @private */
  isChildTransition_(arcTransition) {
    return arcTransition != ARC_SUPERVISION_TRANSITION.UNMANAGED_TO_MANAGED;
  },

  /** @private */
  showManagementTransitionFailedScreen_() {
    this.setUIStep(UIState.ERROR);
  },

  /**
   * On-tap event handler for OK button.
   *
   * @private
   */
  onAcceptAndContinue_() {
    chrome.send('finishManagementTransition');
  },
});
})();
