// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_CHROMEOS_INPUT_METHOD_SUGGESTIONS_SOURCE_H_
#define CHROME_BROWSER_CHROMEOS_INPUT_METHOD_SUGGESTIONS_SOURCE_H_

#include <vector>

#include "base/callback.h"
#include "chromeos/services/ime/public/cpp/suggestions.h"

namespace chromeos {

// Represents a source of text based suggestions.
class SuggestionsSource {
 public:
  virtual ~SuggestionsSource() = default;

  // Get any suggestions that have been generated by the class. This may, or
  // may not, return suggestions.
  virtual std::vector<ime::TextSuggestion> GetSuggestions() = 0;
};

// As the name suggests, this also represents a source of text based
// suggestions, however the interface for fetching suggestions from this object
// is asynchronous.
class AsyncSuggestionsSource {
 public:
  virtual ~AsyncSuggestionsSource() = default;

  using RequestSuggestionsCallback =
      base::OnceCallback<void(const std::vector<ime::TextSuggestion>&)>;

  // Fetch the suggestions from this object, any suggestions fetched will be
  // returned in the callback passed.
  virtual void RequestSuggestions(
      const std::string& preceding_text,
      const ime::TextSuggestionMode& suggestion_mode,
      const std::vector<ime::TextCompletionCandidate>& completion_candidates,
      RequestSuggestionsCallback callback) = 0;

  // Is the source ready to produce suggestions?
  virtual bool IsAvailable() = 0;
};

}  // namespace chromeos

#endif  // CHROME_BROWSER_CHROMEOS_INPUT_METHOD_SUGGESTIONS_SOURCE_H_
