// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/sharing/click_to_call/click_to_call_utils.h"

#include <algorithm>
#include <cctype>

#include "base/strings/string_util.h"
#include "base/strings/utf_string_conversions.h"
#include "build/build_config.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/sharing/click_to_call/phone_number_regex.h"
#include "chrome/browser/sharing/sharing_service.h"
#include "chrome/browser/sharing/sharing_service_factory.h"
#include "chrome/common/pref_names.h"
#include "components/prefs/pref_service.h"
#include "content/public/browser/browser_context.h"
#include "third_party/abseil-cpp/absl/types/optional.h"
#include "third_party/re2/src/re2/re2.h"
#include "url/url_constants.h"
#include "url/url_util.h"

namespace {

// Upper bound on length of selection text to avoid performance issues. Take
// into account numbers with country code and spaces '+99 0 999 999 9999' when
// reducing the max length.
constexpr int kSelectionTextMaxLength = 30;

// Upper bound on digits in selected text to reduce false positives. This
// matches the maximum number of digits in phone numbers according to E.164 and
// showed a good tradeoff between false negatives vs. false positives.
constexpr int kSelectionTextMaxDigits = 15;

bool IsClickToCallEnabled(content::BrowserContext* browser_context) {
#if defined(OS_ANDROID)
  // We don't support sending phone numbers from Android.
  return false;
#else   // defined(OS_ANDROID)
  // Check Chrome enterprise policy for Click to Call.
  Profile* profile = Profile::FromBrowserContext(browser_context);
  if (profile && !profile->GetPrefs()->GetBoolean(prefs::kClickToCallEnabled))
    return false;

  SharingService* sharing_service =
      SharingServiceFactory::GetForBrowserContext(browser_context);
  return sharing_service != nullptr;
#endif  // defined(OS_ANDROID)
}

// Returns the first possible phone number in |selection_text| given the
// |regex_variant| to be used or absl::nullopt if the regex did not match.
absl::optional<std::string> ExtractPhoneNumber(
    const std::string& selection_text) {
  std::string parsed_number;

  const re2::RE2& regex = GetPhoneNumberRegex();
  if (!re2::RE2::PartialMatch(selection_text, regex, &parsed_number))
    return absl::nullopt;

  return base::UTF16ToUTF8(
      base::TrimWhitespace(base::UTF8ToUTF16(parsed_number), base::TRIM_ALL));
}

// Unescapes and returns the URL contents.
std::string GetUnescapedURLContent(const GURL& url) {
  std::string content_string(url.GetContent());
  url::RawCanonOutputT<char16_t> unescaped_content;
  url::DecodeURLEscapeSequences(content_string.data(), content_string.size(),
                                url::DecodeURLMode::kUTF8OrIsomorphic,
                                &unescaped_content);
  return base::UTF16ToUTF8(
      std::u16string(unescaped_content.data(), unescaped_content.length()));
}

}  // namespace

bool ShouldOfferClickToCallForURL(content::BrowserContext* browser_context,
                                  const GURL& url) {
  return !url.is_empty() && url.SchemeIs(url::kTelScheme) &&
         IsUrlSafeForClickToCall(url) && IsClickToCallEnabled(browser_context);
}

absl::optional<std::string> ExtractPhoneNumberForClickToCall(
    content::BrowserContext* browser_context,
    const std::string& selection_text) {
  DCHECK(!selection_text.empty());

  if (selection_text.size() > kSelectionTextMaxLength)
    return absl::nullopt;

  int digits = std::count_if(selection_text.begin(), selection_text.end(),
                             [](char c) { return std::isdigit(c); });
  if (digits > kSelectionTextMaxDigits)
    return absl::nullopt;

  if (!IsClickToCallEnabled(browser_context))
    return absl::nullopt;

  return ExtractPhoneNumber(selection_text);
}

bool IsUrlSafeForClickToCall(const GURL& url) {
  // Get the unescaped content as this is what we'll end up sending to the
  // Android dialer.
  std::string unescaped = GetUnescapedURLContent(url);
  // We don't allow any number that contains any of these characters as they
  // might be used to create USSD codes.
  return !unescaped.empty() &&
         std::none_of(unescaped.begin(), unescaped.end(),
                      [](char c) { return c == '#' || c == '*' || c == '%'; });
}
