// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/enterprise/connectors/file_system/signin_experience.h"

#include "chrome/browser/enterprise/connectors/common.h"
#include "chrome/browser/enterprise/connectors/connectors_service.h"
#include "chrome/browser/enterprise/connectors/file_system/account_info_utils.h"
#include "chrome/browser/enterprise/connectors/file_system/rename_handler.h"
#include "chrome/browser/enterprise/connectors/file_system/service_settings.h"
#include "chrome/browser/enterprise/connectors/file_system/signin_confirmation_modal.h"
#include "chrome/browser/ui/browser.h"
#include "chrome/browser/ui/browser_finder.h"
#include "chrome/browser/ui/browser_window.h"
#include "chrome/grit/generated_resources.h"
#include "content/public/browser/download_item_utils.h"
#include "google_apis/gaia/google_service_auth_error.h"
#include "ui/base/l10n/l10n_util.h"

namespace {

gfx::NativeWindow FindMostRelevantContextWindow(
    const content::WebContents* web_contents) {
  // Can't just use web_contents->GetNativeView(): it results in a dialog that
  // disappears upon browser going out of focus and cannot be re-activated or
  // closed by user.
  auto* browser =
      web_contents ? chrome::FindBrowserWithWebContents(web_contents) : nullptr;

  // Back up methods are needed to find a window to attach the dialog to,
  // because the |web_contents| from |download_item| is stored as a mapping
  // inside of it and is not guaranteed to always exist or be valid. Example: if
  // the original window got closed when download was still in progress; or if
  // we need to resume file upload upon browser restart.
  if (!browser) {
    LOG(ERROR) << "Can't find window from download item; using active window";
    browser = chrome::FindBrowserWithActiveWindow();
  }
  if (!browser) {
    LOG(ERROR) << "Can't find active window; using last active window";
    browser = chrome::FindLastActive();
  }
  DCHECK(browser);
  return browser->window()->GetNativeWindow();
}

namespace ec = enterprise_connectors;

bool MimeTypeMatches(const std::set<std::string>& mime_types,
                     const std::string& mime_type) {
  return mime_types.count(ec::kWildcardMimeType) != 0 ||
         mime_types.count(mime_type) != 0;
}

ec::ConnectorsService* GetConnectorsService(content::BrowserContext* context) {
  if (!base::FeatureList::IsEnabled(ec::kFileSystemConnectorEnabled))
    return nullptr;

  // Check to see if the download item matches any rules.  If the URL of the
  // download itself does not match then check the URL of site on which the
  // download is hosted.
  DCHECK(context);
  return ec::ConnectorsServiceFactory::GetForBrowserContext(context);
}

// These fields must match up fields used in
// chrome/browser/resources/settings/downloads_page/downloads_page.html.
constexpr char kAccountKey[] = "account";
// These fields must also match how base::DictionaryValue's are filled below:
constexpr char kFolderLinkKey[] = "folder.link";
constexpr char kFolderNameKey[] = "folder.name";

void AddFolderInfoToDictionary(PrefService* prefs,
                               std::string provider,
                               base::DictionaryValue* dict) {
  std::string link = ec::GetDefaultFolderLink(prefs, provider);
  std::string name = ec::GetDefaultFolderName(prefs, provider);
  DCHECK(name.size());
  base::DictionaryValue folder;
  folder.SetStringKey("name", std::move(name));
  folder.SetStringKey("link", std::move(link));
  dict->SetKey("folder", std::move(folder));
}

}  // namespace

namespace enterprise_connectors {

absl::optional<FileSystemSettings> GetFileSystemSettings(Profile* profile) {
  auto* service = GetConnectorsService(profile);
  if (!service)
    return absl::nullopt;
  return service->GetFileSystemGlobalSettings(
      FileSystemConnector::SEND_DOWNLOAD_TO_CLOUD);
}

absl::optional<FileSystemSettings> GetFileSystemSettings(
    download::DownloadItem* download_item) {
  auto* context = content::DownloadItemUtils::GetBrowserContext(download_item);
  auto* service = GetConnectorsService(context);
  if (!service)
    return absl::nullopt;

  auto settings = service->GetFileSystemSettings(
      download_item->GetURL(), FileSystemConnector::SEND_DOWNLOAD_TO_CLOUD);
  if (settings.has_value() &&
      MimeTypeMatches(settings->mime_types, download_item->GetMimeType())) {
    return settings;
  }

  settings = service->GetFileSystemSettings(
      download_item->GetTabUrl(), FileSystemConnector::SEND_DOWNLOAD_TO_CLOUD);
  if (settings.has_value() &&
      MimeTypeMatches(settings->mime_types, download_item->GetMimeType())) {
    return settings;
  }

  return absl::nullopt;
}

void OnConfirmationModalClosed(gfx::NativeWindow context,
                               content::BrowserContext* browser_context,
                               const FileSystemSettings& settings,
                               AuthorizationCompletedCallback callback,
                               SigninExperienceTestObserver* test_observer,
                               bool user_confirmed_to_proceed) {
  if (!user_confirmed_to_proceed) {
    return ReturnCancellation(std::move(callback));
  }

  std::unique_ptr<FileSystemSigninDialogDelegate> delegate =
      std::make_unique<FileSystemSigninDialogDelegate>(
          browser_context, settings, std::move(callback));
  content::WebContents* dialog_web_contents = delegate->web_contents();

  // We want a dialog whose lifetime is independent from that of |web_contents|,
  // therefore using FindMostRelevantContextWindow() as context, instead of
  // using web_contents->GetNativeView() as parent. This gives us a new
  // top-level window.
  auto* widget = views::DialogDelegate::CreateDialogWidget(
      std::move(delegate), context, /* parent = */ nullptr);

  if (test_observer)
    test_observer->OnSignInDialogCreated(dialog_web_contents, widget);

  widget->Show();
}

// Start the sign in experience as triggered by a download item.
void StartFileSystemConnectorSigninExperienceForDownloadItem(
    content::WebContents* web_contents,
    const FileSystemSettings& settings,
    AuthorizationCompletedCallback callback,
    SigninExperienceTestObserver* test_observer) {
  gfx::NativeWindow context = FindMostRelevantContextWindow(web_contents);
  DCHECK(context);

  DCHECK_EQ(settings.service_provider, kFileSystemServiceProviderPrefNameBox);
  std::u16string provider =
      l10n_util::GetStringUTF16(IDS_FILE_SYSTEM_CONNECTOR_BOX);

  base::OnceCallback<void(bool)> confirmed_to_sign_in = base::BindOnce(
      &OnConfirmationModalClosed, context, web_contents->GetBrowserContext(),
      settings, std::move(callback), test_observer);
  FileSystemConfirmationModal::Show(
      context,
      l10n_util::GetStringFUTF16(
          IDS_FILE_SYSTEM_CONNECTOR_SIGNIN_REQUIRED_TITLE, provider),
      l10n_util::GetStringFUTF16(
          IDS_FILE_SYSTEM_CONNECTOR_SIGNIN_REQUIRED_MESSAGE, provider),
      l10n_util::GetStringUTF16(
          IDS_FILE_SYSTEM_CONNECTOR_SIGNIN_REQUIRED_CANCEL_BUTTON),
      l10n_util::GetStringUTF16(
          IDS_FILE_SYSTEM_CONNECTOR_SIGNIN_REQUIRED_ACCEPT_BUTTON),
      std::move(confirmed_to_sign_in), test_observer);
}

void OnConfirmationModalClosedForSettingsPage(
    gfx::NativeWindow context,
    content::BrowserContext* browser_context,
    const FileSystemSettings& settings,
    base::OnceCallback<void(bool)> settings_page_callback,
    SigninExperienceTestObserver* test_observer,
    bool user_confirmed_to_proceed) {
  AuthorizationCompletedCallback converted_cb = base::BindOnce(
      [](base::OnceCallback<void(bool)> cb,
         const GoogleServiceAuthError& status, const std::string& access_token,
         const std::string& refresh_token) {
        std::move(cb).Run(status.state() ==
                          GoogleServiceAuthError::State::NONE);
      },
      std::move(settings_page_callback));
  OnConfirmationModalClosed(context, browser_context, settings,
                            std::move(converted_cb), test_observer,
                            user_confirmed_to_proceed);
}

// Start the sign in experience as triggered by the settings page. Similar to
// StartFileSystemConnectorSigninExperienceForDownloadItem() but with different
// displayed texts for FileSystemConfirmationModal::Show().
void StartFileSystemConnectorSigninExperienceForSettingsPage(
    Profile* profile,
    base::OnceCallback<void(bool)> callback,
    SigninExperienceTestObserver* test_observer) {
  gfx::NativeWindow context = FindMostRelevantContextWindow(nullptr);
  DCHECK(context);

  auto settings = GetFileSystemSettings(profile);
  if (!settings.has_value())
    return std::move(callback).Run(false);

  DCHECK_EQ(settings->service_provider, kFileSystemServiceProviderPrefNameBox);
  std::u16string provider =
      l10n_util::GetStringUTF16(IDS_FILE_SYSTEM_CONNECTOR_BOX);

  base::OnceCallback<void(bool)> confirmed_to_sign_in = base::BindOnce(
      &OnConfirmationModalClosedForSettingsPage, context, profile,
      settings.value(), std::move(callback), test_observer);
  FileSystemConfirmationModal::Show(
      context,
      l10n_util::GetStringFUTF16(IDS_FILE_SYSTEM_CONNECTOR_SIGNIN_CONFIRM_TITLE,
                                 provider),
      l10n_util::GetStringUTF16(
          IDS_FILE_SYSTEM_CONNECTOR_SIGNIN_CONFIRM_MESSAGE),
      l10n_util::GetStringUTF16(
          IDS_FILE_SYSTEM_CONNECTOR_SIGNIN_CONFIRM_CANCEL_BUTTON),
      l10n_util::GetStringUTF16(
          IDS_FILE_SYSTEM_CONNECTOR_SIGNIN_CONFIRM_ACCEPT_BUTTON),
      std::move(confirmed_to_sign_in));
}

// Clear authentication tokens and stored account info.
bool ClearFileSystemConnectorLinkedAccount(const FileSystemSettings& settings,
                                           PrefService* prefs) {
  return ClearFileSystemOAuth2Tokens(prefs, settings.service_provider) &&
         ClearFileSystemAccountInfo(prefs, settings.service_provider);
}

absl::optional<base::DictionaryValue>
GetFileSystemConnectorLinkedAccountInfoForSettingsPage(
    const FileSystemSettings& settings,
    PrefService* prefs) {
  std::string refresh_token;
  base::DictionaryValue info;
  base::Value* stored_account_info = nullptr;
  const std::string& provider = settings.service_provider;
  if (!(stored_account_info = info.SetKey(
            kAccountKey, GetFileSystemAccountInfo(prefs, provider))) ||
      stored_account_info->DictEmpty() ||
      !GetFileSystemOAuth2Tokens(prefs, provider, /* access_token = */ nullptr,
                                 &refresh_token) ||
      refresh_token.empty()) {
    return absl::nullopt;
  }

  DCHECK(refresh_token.size()) << "No refresh token for linked account";

  AddFolderInfoToDictionary(prefs, provider, &info);
  return absl::make_optional<base::DictionaryValue>(std::move(info));
}

void SetFileSystemConnectorAccountLinkForSettingsPage(
    bool enable_link,
    Profile* profile,
    base::OnceCallback<void(bool)> callback,
    SigninExperienceTestObserver* test_observer) {
  absl::optional<FileSystemSettings> settings = GetFileSystemSettings(profile);
  auto has_linked_account =
      settings.has_value() &&
      GetFileSystemConnectorLinkedAccountInfoForSettingsPage(
          settings.value(), profile->GetPrefs());

  // Early return if linked state already match the desired state.
  if (has_linked_account == enable_link) {
    std::move(callback).Run(true);
    return;
  }

  // Early return after a quick clearing function call.
  if (has_linked_account) {
    bool success = ClearFileSystemConnectorLinkedAccount(
        GetFileSystemSettings(profile).value(), profile->GetPrefs());
    std::move(callback).Run(success);
    return;
  }

  // This shows dialogs for the sign-in experience that the user needs to
  // interact with, so the process is async.
  StartFileSystemConnectorSigninExperienceForSettingsPage(
      profile, std::move(callback), test_observer);
}

void ReturnCancellation(AuthorizationCompletedCallback callback) {
  std::move(callback).Run(
      GoogleServiceAuthError{GoogleServiceAuthError::State::REQUEST_CANCELED},
      std::string(), std::string());
}

// Helper method for testing.
void ExtractAccountInfoFromDictionary(const base::DictionaryValue& dict,
                                      base::Value* account,
                                      std::string* folder_name,
                                      std::string* folder_link) {
  if (account && dict.FindPath(kAccountKey))
    *account = dict.FindPath(kAccountKey)->Clone();
  if (folder_name && dict.FindStringPath(kFolderNameKey))
    *folder_name = *dict.FindStringPath(kFolderNameKey);
  if (folder_link && dict.FindStringPath(kFolderLinkKey))
    *folder_link = *dict.FindStringPath(kFolderLinkKey);
}

// SigninExperienceTestObserver
SigninExperienceTestObserver::SigninExperienceTestObserver() = default;

void SigninExperienceTestObserver::InitForTesting(
    FileSystemRenameHandler* rename_handler) {
  if (!rename_handler)
    return;
  rename_handler_ =
      rename_handler->RegisterSigninObserverForTesting(this);  // IN-TEST
}

SigninExperienceTestObserver::~SigninExperienceTestObserver() {
  if (!rename_handler_)
    return;
  rename_handler_->UnregisterSigninObserverForTesting(this);  // IN-TEST
  rename_handler_.reset();
}

// TODO(https://crbug.com/1159185): add browser_tests for
// StartFileSystemConnectorSigninExperienceForXxx.

}  // namespace enterprise_connectors
