// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_NOTIFICATIONS_NOTIFICATION_DISPATCHER_MOJO_H_
#define CHROME_BROWSER_NOTIFICATIONS_NOTIFICATION_DISPATCHER_MOJO_H_

#include <memory>
#include <string>
#include <tuple>

#include "base/callback_forward.h"
#include "base/cancelable_callback.h"
#include "base/containers/flat_set.h"
#include "base/time/time.h"
#include "chrome/browser/notifications/displayed_notifications_dispatch_callback.h"
#include "chrome/browser/notifications/notification_dispatcher_mac.h"
#include "chrome/services/mac_notifications/public/mojom/mac_notifications.mojom.h"
#include "mojo/public/cpp/bindings/receiver.h"
#include "mojo/public/cpp/bindings/remote.h"

class MacNotificationProviderFactory;

// Connects to the macOS notification service via mojo to manage notifications.
class NotificationDispatcherMojo
    : public NotificationDispatcherMac,
      public mac_notifications::mojom::MacNotificationActionHandler {
 public:
  explicit NotificationDispatcherMojo(
      std::unique_ptr<MacNotificationProviderFactory> provider_factory);
  NotificationDispatcherMojo(const NotificationDispatcherMojo&) = delete;
  NotificationDispatcherMojo& operator=(const NotificationDispatcherMojo&) =
      delete;
  ~NotificationDispatcherMojo() override;

  // NotificationDispatcherMac:
  void DisplayNotification(
      NotificationHandler::Type notification_type,
      Profile* profile,
      const message_center::Notification& notification) override;
  void CloseNotificationWithId(
      const MacNotificationIdentifier& identifier) override;
  void CloseNotificationsWithProfileId(const std::string& profile_id,
                                       bool incognito) override;
  void CloseAllNotifications() override;
  void GetDisplayedNotificationsForProfileId(
      const std::string& profile_id,
      bool incognito,
      GetDisplayedNotificationsCallback callback) override;
  void GetAllDisplayedNotifications(
      GetAllDisplayedNotificationsCallback callback) override;

  // mac_notifications::mojom::MacNotificationActionHandler:
  void OnNotificationAction(
      mac_notifications::mojom::NotificationActionInfoPtr info) override;

 private:
  void CheckIfNotificationsRemaining();
  void OnServiceDisconnectedGracefully(bool gracefully);

  mac_notifications::mojom::MacNotificationService* GetOrCreateService();

  void DispatchGetNotificationsReply(
      GetDisplayedNotificationsCallback callback,
      std::vector<mac_notifications::mojom::NotificationIdentifierPtr>
          notifications);
  void DispatchGetAllNotificationsReply(
      GetAllDisplayedNotificationsCallback callback,
      std::vector<mac_notifications::mojom::NotificationIdentifierPtr>
          notifications);

  std::unique_ptr<MacNotificationProviderFactory> provider_factory_;
  mojo::Remote<mac_notifications::mojom::MacNotificationProvider> provider_;
  mojo::Remote<mac_notifications::mojom::MacNotificationService> service_;
  mojo::Receiver<mac_notifications::mojom::MacNotificationActionHandler>
      handler_{this};
  base::CancelableOnceClosure no_notifications_checker_;
  base::TimeTicks service_start_time_;
};

#endif  // CHROME_BROWSER_NOTIFICATIONS_NOTIFICATION_DISPATCHER_MOJO_H_
