// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/notifications/notification_dispatcher_mojo.h"

#include <memory>

#include "base/callback.h"
#include "base/process/process_handle.h"
#include "base/test/gmock_callback_support.h"
#include "base/test/metrics/histogram_tester.h"
#include "base/test/mock_callback.h"
#include "base/time/time.h"
#include "chrome/browser/notifications/mac_notification_provider_factory.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/services/mac_notifications/public/mojom/mac_notifications.mojom.h"
#include "chrome/test/base/testing_browser_process.h"
#include "chrome/test/base/testing_profile_manager.h"
#include "content/public/test/browser_task_environment.h"
#include "mojo/public/cpp/bindings/pending_receiver.h"
#include "mojo/public/cpp/bindings/pending_remote.h"
#include "mojo/public/cpp/bindings/receiver.h"
#include "mojo/public/cpp/bindings/remote.h"
#include "testing/gmock/include/gmock/gmock.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "third_party/abseil-cpp/absl/types/optional.h"

namespace {

class MockNotificationService
    : public mac_notifications::mojom::MacNotificationService {
 public:
  MOCK_METHOD(void,
              DisplayNotification,
              (mac_notifications::mojom::NotificationPtr),
              (override));
  MOCK_METHOD(void,
              GetDisplayedNotifications,
              (mac_notifications::mojom::ProfileIdentifierPtr,
               GetDisplayedNotificationsCallback),
              (override));
  MOCK_METHOD(void,
              CloseNotification,
              (mac_notifications::mojom::NotificationIdentifierPtr),
              (override));
  MOCK_METHOD(void,
              CloseNotificationsForProfile,
              (mac_notifications::mojom::ProfileIdentifierPtr),
              (override));
  MOCK_METHOD(void, CloseAllNotifications, (), (override));
};

class MockNotificationProvider
    : public mac_notifications::mojom::MacNotificationProvider {
 public:
  MOCK_METHOD(
      void,
      BindNotificationService,
      (mojo::PendingReceiver<mac_notifications::mojom::MacNotificationService>,
       mojo::PendingRemote<
           mac_notifications::mojom::MacNotificationActionHandler>),
      (override));
};

std::vector<mac_notifications::mojom::NotificationIdentifierPtr>
CreateOneNotificationList() {
  std::vector<mac_notifications::mojom::NotificationIdentifierPtr>
      notifications;
  auto profile_id = mac_notifications::mojom::ProfileIdentifier::New(
      "profileId", /*incognito=*/true);
  notifications.push_back(mac_notifications::mojom::NotificationIdentifier::New(
      "notificationId", std::move(profile_id)));
  return notifications;
}

class FakeMacNotificationProviderFactory
    : public MacNotificationProviderFactory,
      public mac_notifications::mojom::MacNotificationProvider {
 public:
  explicit FakeMacNotificationProviderFactory(base::OnceClosure on_disconnect)
      : MacNotificationProviderFactory(/*in_process=*/false),
        on_disconnect_(std::move(on_disconnect)) {}
  ~FakeMacNotificationProviderFactory() override = default;

  // MacNotificationProviderFactory:
  mojo::Remote<mac_notifications::mojom::MacNotificationProvider>
  LaunchProvider() override {
    mojo::Remote<mac_notifications::mojom::MacNotificationProvider> remote;
    provider_receiver_.Bind(remote.BindNewPipeAndPassReceiver());
    provider_receiver_.set_disconnect_handler(std::move(on_disconnect_));
    return remote;
  }

  // mac_notifications::mojom::MacNotificationProvider:
  void BindNotificationService(
      mojo::PendingReceiver<mac_notifications::mojom::MacNotificationService>
          service,
      mojo::PendingRemote<
          mac_notifications::mojom::MacNotificationActionHandler> handler)
      override {
    service_receiver_.Bind(std::move(service));
    handler_remote_.Bind(std::move(handler));
  }

  MockNotificationService& service() { return mock_service_; }

  mac_notifications::mojom::MacNotificationActionHandler* handler() {
    return handler_remote_.get();
  }

  void disconnect() {
    handler_remote_.reset();
    service_receiver_.reset();
    provider_receiver_.reset();
  }

 private:
  base::OnceClosure on_disconnect_;
  mojo::Receiver<mac_notifications::mojom::MacNotificationProvider>
      provider_receiver_{this};
  MockNotificationService mock_service_;
  mojo::Receiver<mac_notifications::mojom::MacNotificationService>
      service_receiver_{&mock_service_};
  mojo::Remote<mac_notifications::mojom::MacNotificationActionHandler>
      handler_remote_;
};

message_center::Notification CreateNotification() {
  return message_center::Notification(
      message_center::NOTIFICATION_TYPE_SIMPLE, "notification_id", u"title",
      u"message", /*icon=*/gfx::Image(),
      /*display_source=*/std::u16string(), /*origin_url=*/GURL(),
      message_center::NotifierId(), message_center::RichNotificationData(),
      base::MakeRefCounted<message_center::NotificationDelegate>());
}

mac_notifications::mojom::NotificationMetadataPtr CreateNotificationMetadata() {
  auto profile_identifier = mac_notifications::mojom::ProfileIdentifier::New(
      "profile", /*incognito=*/false);
  auto notification_identifier =
      mac_notifications::mojom::NotificationIdentifier::New(
          "notification_id", std::move(profile_identifier));
  return mac_notifications::mojom::NotificationMetadata::New(
      std::move(notification_identifier), /*notification_type=*/0,
      /*origin_url=*/GURL("https://example.com"), base::GetCurrentProcId());
}

mac_notifications::mojom::NotificationActionInfoPtr
CreateNotificationActionInfo() {
  auto meta = CreateNotificationMetadata();
  return mac_notifications::mojom::NotificationActionInfo::New(
      std::move(meta), NotificationOperation::NOTIFICATION_CLICK,
      /*button_index=*/-1, /*reply=*/absl::nullopt);
}

}  // namespace

class NotificationDispatcherMojoTest : public testing::Test {
 public:
  NotificationDispatcherMojoTest() {
    auto provider_factory =
        std::make_unique<FakeMacNotificationProviderFactory>(
            on_disconnect_.Get());
    provider_factory_ = provider_factory.get();
    notification_dispatcher_ = std::make_unique<NotificationDispatcherMojo>(
        std::move(provider_factory));
  }

  ~NotificationDispatcherMojoTest() override {
    provider_factory_->disconnect();
    task_environment_.RunUntilIdle();
  }

  // testing::Test:
  void SetUp() override {
    ASSERT_TRUE(testing_profile_manager_.SetUp());
    profile_ = testing_profile_manager_.CreateTestingProfile("profile");
  }

  MockNotificationService& service() { return provider_factory_->service(); }

  mac_notifications::mojom::MacNotificationActionHandler* handler() {
    return provider_factory_->handler();
  }

 protected:
  void ExpectDisconnect(base::OnceClosure callback) {
    EXPECT_CALL(on_disconnect_, Run())
        .WillOnce(base::test::RunOnceClosure(std::move(callback)));
  }

  void ExpectKeepConnected() {
    EXPECT_CALL(on_disconnect_, Run()).Times(0);
    // Run remaining tasks to see if we get disconnected.
    task_environment_.RunUntilIdle();
  }

  void EmulateNoNotifications() {
    EXPECT_CALL(service(), GetDisplayedNotifications)
        .WillOnce([](mac_notifications::mojom::ProfileIdentifierPtr profile,
                     MockNotificationService::GetDisplayedNotificationsCallback
                         callback) {
          // Emulate an empty list of notifications.
          std::move(callback).Run({});
        });
  }

  void EmulateOneNotification(base::OnceClosure callback) {
    EXPECT_CALL(service(), GetDisplayedNotifications)
        .WillOnce(testing::DoAll(
            base::test::RunOnceClosure(std::move(callback)),
            [](mac_notifications::mojom::ProfileIdentifierPtr profile,
               MockNotificationService::GetDisplayedNotificationsCallback
                   callback) {
              // Emulate one remaining notification.
              std::move(callback).Run(CreateOneNotificationList());
            }));
  }

  content::BrowserTaskEnvironment task_environment_{
      base::test::TaskEnvironment::TimeSource::MOCK_TIME};
  TestingProfileManager testing_profile_manager_{
      TestingBrowserProcess::GetGlobal()};
  Profile* profile_;
  base::MockOnceClosure on_disconnect_;
  std::unique_ptr<NotificationDispatcherMojo> notification_dispatcher_;
  FakeMacNotificationProviderFactory* provider_factory_ = nullptr;
};

TEST_F(NotificationDispatcherMojoTest, CloseNotificationAndDisconnect) {
  base::RunLoop run_loop;
  // Expect that we disconnect after closing the last notification.
  ExpectDisconnect(run_loop.QuitClosure());
  EXPECT_CALL(service(), CloseNotification)
      .WillOnce(
          [](mac_notifications::mojom::NotificationIdentifierPtr identifier) {
            EXPECT_EQ("notificationId", identifier->id);
            EXPECT_EQ("profileId", identifier->profile->id);
            EXPECT_TRUE(identifier->profile->incognito);
          });
  EmulateNoNotifications();
  notification_dispatcher_->CloseNotificationWithId(
      {"notificationId", "profileId", /*incognito=*/true});
  run_loop.Run();
}

TEST_F(NotificationDispatcherMojoTest, CloseNotificationAndKeepConnected) {
  base::RunLoop run_loop;
  // Expect that we continue running if there are remaining notifications.
  EXPECT_CALL(service(), CloseNotification);
  EmulateOneNotification(run_loop.QuitClosure());
  notification_dispatcher_->CloseNotificationWithId(
      {"notificationId", "profileId", /*incognito=*/true});
  run_loop.Run();
  ExpectKeepConnected();
}

TEST_F(NotificationDispatcherMojoTest,
       CloseThenDispatchNotificationAndKeepConnected) {
  base::RunLoop run_loop;
  // Expect that we continue running when showing a new notification just after
  // closing the last one.
  EXPECT_CALL(service(), CloseNotification);
  EmulateNoNotifications();
  notification_dispatcher_->CloseNotificationWithId(
      {"notificationId", "profileId", /*incognito=*/true});

  EXPECT_CALL(service(), DisplayNotification)
      .WillOnce(base::test::RunOnceClosure(run_loop.QuitClosure()));
  notification_dispatcher_->DisplayNotification(
      NotificationHandler::Type::WEB_PERSISTENT, profile_,
      CreateNotification());

  run_loop.Run();
  ExpectKeepConnected();

  base::RunLoop run_loop2;
  // Expect that we disconnect after closing all notifications.
  ExpectDisconnect(run_loop2.QuitClosure());
  EXPECT_CALL(service(), CloseAllNotifications);
  notification_dispatcher_->CloseAllNotifications();
  run_loop2.Run();
}

TEST_F(NotificationDispatcherMojoTest, CloseProfileNotificationsAndDisconnect) {
  base::RunLoop run_loop;
  // Expect that we disconnect after closing the last notification.
  ExpectDisconnect(run_loop.QuitClosure());
  EXPECT_CALL(service(), CloseNotificationsForProfile)
      .WillOnce([](mac_notifications::mojom::ProfileIdentifierPtr profile) {
        EXPECT_EQ("profileId", profile->id);
        EXPECT_TRUE(profile->incognito);
      });
  EmulateNoNotifications();
  notification_dispatcher_->CloseNotificationsWithProfileId("profileId",
                                                            /*incognito=*/true);
  run_loop.Run();
}

TEST_F(NotificationDispatcherMojoTest, CloseAndDisconnectTiming) {
  base::HistogramTester histograms;
  // Show a new notification.
  EXPECT_CALL(service(), DisplayNotification);
  notification_dispatcher_->DisplayNotification(
      NotificationHandler::Type::WEB_PERSISTENT, profile_,
      CreateNotification());

  // Wait for 30 seconds and close the notification.
  auto delay = base::TimeDelta::FromSeconds(30);
  task_environment_.FastForwardBy(delay);

  // Expect that we disconnect after closing the last notification.
  base::RunLoop run_loop;
  ExpectDisconnect(run_loop.QuitClosure());
  EmulateNoNotifications();
  EXPECT_CALL(service(), CloseNotification);
  notification_dispatcher_->CloseNotificationWithId(
      {"notificationId", "profileId", /*incognito=*/true});

  // Verify that we log the runtime length and no unexpected kill.
  run_loop.Run();
  histograms.ExpectUniqueTimeSample("Notifications.macOS.ServiceProcessRuntime",
                                    delay, /*expected_count=*/1);
  histograms.ExpectTotalCount("Notifications.macOS.ServiceProcessKilled",
                              /*count=*/0);
}

TEST_F(NotificationDispatcherMojoTest, KillServiceTiming) {
  base::HistogramTester histograms;
  // Show a new notification.
  EXPECT_CALL(service(), DisplayNotification);
  notification_dispatcher_->DisplayNotification(
      NotificationHandler::Type::WEB_PERSISTENT, profile_,
      CreateNotification());

  // Wait for 30 seconds and terminate the service.
  auto delay = base::TimeDelta::FromSeconds(30);
  task_environment_.FastForwardBy(delay);
  // Simulate a dying service process.
  provider_factory_->disconnect();

  // Run remaining tasks as we can't observe the disconnect callback.
  task_environment_.RunUntilIdle();
  // Verify that we log the runtime length and an unexpected kill.
  histograms.ExpectUniqueTimeSample("Notifications.macOS.ServiceProcessRuntime",
                                    delay, /*expected_count=*/1);
  histograms.ExpectUniqueTimeSample("Notifications.macOS.ServiceProcessKilled",
                                    delay, /*expected_count=*/1);
}

TEST_F(NotificationDispatcherMojoTest, DidActivateNotification) {
  base::HistogramTester histograms;
  // Show a new notification.
  EmulateNoNotifications();
  EXPECT_CALL(service(), DisplayNotification);
  notification_dispatcher_->DisplayNotification(
      NotificationHandler::Type::WEB_PERSISTENT, profile_,
      CreateNotification());

  // Wait until the action handler has been bound.
  task_environment_.RunUntilIdle();
  handler()->OnNotificationAction(CreateNotificationActionInfo());

  // Handling responses is async, make sure we wait for all tasks to complete.
  task_environment_.RunUntilIdle();

  histograms.ExpectUniqueSample("Notifications.macOS.ActionReceived.Alert",
                                /*sample=*/true,
                                /*expected_count=*/1);
}
