// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/views/page_info/page_info_bubble_view.h"

#include "base/json/json_reader.h"
#include "base/macros.h"
#include "base/strings/utf_string_conversions.h"
#include "build/build_config.h"
#include "chrome/browser/content_settings/host_content_settings_map_factory.h"
#include "chrome/browser/content_settings/page_specific_content_settings_delegate.h"
#include "chrome/browser/history/history_service_factory.h"
#include "chrome/browser/ssl/security_state_tab_helper.h"
#include "chrome/browser/ui/exclusive_access/exclusive_access_manager.h"
#include "chrome/browser/ui/hats/mock_trust_safety_sentiment_service.h"
#include "chrome/browser/ui/hats/trust_safety_sentiment_service_factory.h"
#include "chrome/browser/ui/views/hover_button.h"
#include "chrome/browser/ui/views/page_info/chosen_object_view.h"
#include "chrome/browser/ui/views/page_info/page_info_hover_button.h"
#include "chrome/browser/ui/views/page_info/page_info_main_view.h"
#include "chrome/browser/ui/views/page_info/page_info_new_bubble_view.h"
#include "chrome/browser/ui/views/page_info/page_info_row_view.h"
#include "chrome/browser/ui/views/page_info/page_info_security_content_view.h"
#include "chrome/browser/ui/views/page_info/page_info_view_factory.h"
#include "chrome/browser/ui/views/page_info/page_switcher_view.h"
#include "chrome/browser/ui/views/page_info/permission_selector_row.h"
#include "chrome/browser/ui/views/page_info/permission_toggle_row_view.h"
#include "chrome/browser/usb/usb_chooser_context.h"
#include "chrome/browser/usb/usb_chooser_context_factory.h"
#include "chrome/common/pref_names.h"
#include "chrome/common/url_constants.h"
#include "chrome/test/base/testing_browser_process.h"
#include "chrome/test/base/testing_profile.h"
#include "chrome/test/base/testing_profile_manager.h"
#include "chrome/test/views/chrome_test_views_delegate.h"
#include "components/content_settings/core/browser/host_content_settings_map.h"
#include "components/content_settings/core/common/pref_names.h"
#include "components/history/core/browser/history_service.h"
#include "components/page_info/features.h"
#include "components/permissions/permission_uma_util.h"
#include "components/strings/grit/components_strings.h"
#include "components/ukm/test_ukm_recorder.h"
#include "content/public/browser/ssl_status.h"
#include "content/public/test/browser_task_environment.h"
#include "content/public/test/navigation_simulator.h"
#include "content/public/test/test_renderer_host.h"
#include "content/public/test/test_web_contents_factory.h"
#include "mojo/public/cpp/bindings/pending_remote.h"
#include "net/cert/cert_status_flags.h"
#include "net/ssl/ssl_connection_status_flags.h"
#include "net/test/cert_test_util.h"
#include "net/test/test_certificate_data.h"
#include "net/test/test_data_directory.h"
#include "ppapi/buildflags/buildflags.h"
#include "services/device/public/cpp/test/fake_usb_device_manager.h"
#include "services/device/public/mojom/usb_device.mojom.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "ui/accessibility/ax_enums.mojom.h"
#include "ui/base/l10n/l10n_util.h"
#include "ui/base/ui_base_features.h"
#include "ui/events/event_utils.h"
#include "ui/views/controls/button/toggle_button.h"
#include "ui/views/controls/combobox/combobox.h"
#include "ui/views/controls/label.h"
#include "ui/views/controls/link.h"
#include "ui/views/test/button_test_api.h"
#include "ui/views/test/scoped_views_test_helper.h"
#include "ui/views/test/test_views_delegate.h"

#if BUILDFLAG(ENABLE_PLUGINS)
#include "chrome/browser/plugins/chrome_plugin_service_filter.h"
#endif

#if BUILDFLAG(IS_CHROMEOS_ASH)
#include "chrome/browser/ash/login/users/fake_chrome_user_manager.h"
#include "components/account_id/account_id.h"
#include "components/user_manager/scoped_user_manager.h"
#endif  // BUILDFLAG(IS_CHROMEOS_ASH)

const char* kUrl = "http://www.example.com/index.html";
const char* kSecureUrl = "https://www.example.com/index.html";
std::u16string kHostname = u"example.com";

namespace test {

class PageInfoBubbleViewTestApi {
 public:
  PageInfoBubbleViewTestApi(gfx::NativeView parent,
                            Profile* profile,
                            content::WebContents* web_contents,
                            bool is_version_two)
      : bubble_delegate_(nullptr),
        parent_(parent),
        profile_(profile),
        web_contents_(web_contents),
        is_version_two_(is_version_two) {
    CreateView();
  }

  void CreateView() {
    if (bubble_delegate_) {
      bubble_delegate_->GetWidget()->CloseNow();
    }

    views::View* anchor_view = nullptr;
    Profile* profile =
        use_off_the_record_profile_
            ? profile_->GetPrimaryOTRProfile(/*create_if_needed=*/true)
            : profile_;
    if (is_version_two_) {
      auto* bubble = new PageInfoNewBubbleView(
          anchor_view, gfx::Rect(), parent_, profile, web_contents_, GURL(kUrl),
          base::BindOnce(&PageInfoBubbleViewTestApi::OnPageInfoBubbleClosed,
                         base::Unretained(this), run_loop_.QuitClosure()));
      presenter_ = bubble->presenter_.get();
      navigation_handler_ = bubble;
      bubble_delegate_ = bubble;
      toggle_rows_ =
          &static_cast<PageInfoMainView*>(current_view())->selector_rows_;
    } else {
      auto* bubble = new PageInfoBubbleView(
          anchor_view, gfx::Rect(), parent_, profile, web_contents_, GURL(kUrl),
          base::BindOnce(&PageInfoBubbleViewTestApi::OnPageInfoBubbleClosed,
                         base::Unretained(this), run_loop_.QuitClosure()));
      presenter_ = bubble->presenter_.get();
      selector_rows_ = &bubble->selector_rows_;
      bubble_delegate_ = bubble;
    }
  }

  views::View* current_view() {
    return bubble_delegate_->GetViewByID(
        PageInfoViewFactory::VIEW_ID_PAGE_INFO_CURRENT_VIEW);
  }
  bool reload_prompt() const { return *reload_prompt_; }
  views::Widget::ClosedReason closed_reason() const { return *closed_reason_; }

  views::View* permissions_view() {
    return bubble_delegate_->GetViewByID(
        PageInfoViewFactory::VIEW_ID_PAGE_INFO_PERMISSION_VIEW);
  }

  views::View* cookie_button() {
    return bubble_delegate_->GetViewByID(
        PageInfoViewFactory::VIEW_ID_PAGE_INFO_LINK_OR_BUTTON_COOKIE_DIALOG);
  }

  PageInfoHoverButton* certificate_button() const {
    return static_cast<PageInfoHoverButton*>(bubble_delegate_->GetViewByID(
        PageInfoViewFactory::
            VIEW_ID_PAGE_INFO_LINK_OR_BUTTON_CERTIFICATE_VIEWER));
  }

  views::View* security_summary_label() {
    DCHECK(is_version_two_);
    return bubble_delegate_->GetViewByID(
        PageInfoViewFactory::VIEW_ID_PAGE_INFO_SECURITY_SUMMARY_LABEL);
  }

  views::StyledLabel* security_details_label() {
    return static_cast<views::StyledLabel*>(bubble_delegate_->GetViewByID(
        PageInfoViewFactory::VIEW_ID_PAGE_INFO_SECURITY_DETAILS_LABEL));
  }

  views::LabelButton* reset_permissions_button() {
    DCHECK(is_version_two_);
    return static_cast<views::LabelButton*>(bubble_delegate_->GetViewByID(
        PageInfoViewFactory::VIEW_ID_PAGE_INFO_RESET_PERMISSIONS_BUTTON));
  }

  PageInfoNavigationHandler* navigation_handler() {
    DCHECK(is_version_two_);
    return navigation_handler_;
  }

  std::u16string GetWindowTitle() { return bubble_delegate_->GetWindowTitle(); }

  PermissionSelectorRow* GetPermissionSelectorAt(int index) {
    DCHECK(!is_version_two_);
    return (*selector_rows_)[index].get();
  }

  PermissionToggleRowView* GetPermissionToggleRowAt(int index) {
    DCHECK(is_version_two_);
    return (*toggle_rows_)[index];
  }

  views::ToggleButton* GetToggleViewAt(int index) {
    DCHECK(is_version_two_);
    return GetPermissionToggleRowAt(index)->toggle_button_;
  }

  views::Label* GetStateLabelAt(int index) {
    DCHECK(is_version_two_);
    return GetPermissionToggleRowAt(index)->state_label_;
  }

  // Returns the number of cookies shown on the link or button to open the
  // collected cookies dialog. This should always be shown.
  std::u16string GetCookiesLinkText() {
    EXPECT_TRUE(cookie_button());
    ui::AXNodeData data;
    cookie_button()->GetAccessibleNodeData(&data);
    std::string name;
    data.GetStringAttribute(ax::mojom::StringAttribute::kName, &name);
    return base::ASCIIToUTF16(name);
  }

  std::u16string GetSecurityInformationButtonText() {
    auto* button = bubble_delegate_->GetViewByID(
        PageInfoViewFactory::
            VIEW_ID_PAGE_INFO_LINK_OR_BUTTON_SECURITY_INFORMATION);
    return static_cast<PageInfoHoverButton*>(button)->title()->GetText();
  }

  std::u16string GetSecuritySummaryText() {
    EXPECT_TRUE(security_summary_label());
    return static_cast<views::StyledLabel*>(security_summary_label())
        ->GetText();
  }

  std::u16string GetPermissionLabelTextAt(int index) {
    return is_version_two_
               ? GetPermissionToggleRowAt(index)->row_view_->title_->GetText()
               : GetPermissionSelectorAt(index)->label_->GetText();
  }

  std::u16string GetPermissionComboboxTextAt(int index) {
    DCHECK(!is_version_two_);
    auto* combobox = GetPermissionSelectorAt(index)->combobox_;
    return combobox->GetTextForRow(combobox->GetSelectedRow());
  }

  bool GetPermissionToggleIsOnAt(int index) {
    auto* toggle = GetToggleViewAt(index);
    return toggle->GetIsOn();
  }

  void SimulateUserSelectingComboboxItemAt(int selector_index, int menu_index) {
    DCHECK(!is_version_two_);
    auto* combobox = GetPermissionSelectorAt(selector_index)->combobox_;
    combobox->SetSelectedRow(menu_index);
  }

  void SimulateTogglingPermissionAt(int index) {
    auto* toggle = GetToggleViewAt(index);
    toggle->SetIsOn(!toggle->GetIsOn());
  }

  bool ValidatePermissionsChildrenCount(int expected_count) {
    // In page info v2, not-empty permission section has a reset all button
    // after all permission rows.
    const int kAdditionalViewsCount = 1;
    const int actual_count =
        permissions_view() ? permissions_view()->children().size() : 0;
    if (expected_count == 0 || !is_version_two_) {
      return expected_count == actual_count;
    } else {
      return expected_count + kAdditionalViewsCount == actual_count;
    }
  }

  // Simulates updating the number of cookies.
  void SetCookieInfo(const CookieInfoList& list) {
    presenter_->ui_for_testing()->SetCookieInfo(list);
  }

  // Simulates recreating the dialog with a new PermissionInfoList.
  void SetPermissionInfo(const PermissionInfoList& list) {
    for (const PageInfo::PermissionInfo& info : list) {
      presenter_->OnSitePermissionChanged(info.type, info.setting,
                                          /*is_one_time=*/false);
    }
    CreateView();
  }

  std::u16string GetCertificateButtonSubtitleText() const {
    EXPECT_TRUE(certificate_button());
    EXPECT_TRUE(certificate_button()->subtitle());
    return certificate_button()->subtitle()->GetText();
  }

  const views::View::Views& GetChosenObjectChildren() {
    const int object_view_index = 0;
    ChosenObjectView* object_view = static_cast<ChosenObjectView*>(
        permissions_view()->children()[object_view_index]);
    views::View* row_view = object_view->children()[0];
    return row_view->children();
  }

  void WaitForBubbleClose() { run_loop_.Run(); }

  void SetOffTheRecordProfile() {
    use_off_the_record_profile_ = true;
    CreateView();
  }

 private:
  void OnPageInfoBubbleClosed(base::RepeatingCallback<void()> quit_closure,
                              views::Widget::ClosedReason closed_reason,
                              bool reload_prompt) {
    closed_reason_ = closed_reason;
    reload_prompt_ = reload_prompt;
    quit_closure.Run();
  }

  views::BubbleDialogDelegateView* bubble_delegate_;
  PageInfo* presenter_ = nullptr;
  std::vector<std::unique_ptr<PermissionSelectorRow>>* selector_rows_ = nullptr;
  std::vector<PermissionToggleRowView*>* toggle_rows_ = nullptr;

  PageInfoNavigationHandler* navigation_handler_ = nullptr;

  // For recreating the view.
  gfx::NativeView parent_;
  Profile* profile_;
  content::WebContents* web_contents_;
  base::RunLoop run_loop_;
  absl::optional<bool> reload_prompt_;
  absl::optional<views::Widget::ClosedReason> closed_reason_;
  bool is_version_two_;
  bool use_off_the_record_profile_ = false;

  DISALLOW_COPY_AND_ASSIGN(PageInfoBubbleViewTestApi);
};

}  // namespace test

namespace {

constexpr char kTestUserEmail[] = "user@example.com";

// Helper class that wraps a TestingProfile and a TestWebContents for a test
// harness. Inspired by RenderViewHostTestHarness, but doesn't use inheritance
// so the helper can be composed with other helpers in the test harness.
class ScopedWebContentsTestHelper {
 public:
  ScopedWebContentsTestHelper()
      : testing_profile_manager_(TestingBrowserProcess::GetGlobal()) {
#if BUILDFLAG(IS_CHROMEOS_ASH)
    auto fake_user_manager = std::make_unique<ash::FakeChromeUserManager>();
    auto* fake_user_manager_ptr = fake_user_manager.get();
    scoped_user_manager_ = std::make_unique<user_manager::ScopedUserManager>(
        std::move(fake_user_manager));

    constexpr char kTestUserGaiaId[] = "1111111111";
    auto account_id =
        AccountId::FromUserEmailGaiaId(kTestUserEmail, kTestUserGaiaId);
    fake_user_manager_ptr->AddUserWithAffiliation(account_id,
                                                  /*is_affiliated=*/true);
    fake_user_manager_ptr->LoginUser(account_id);
#endif  // BUILDFLAG(IS_CHROMEOS_ASH)

    EXPECT_TRUE(testing_profile_manager_.SetUp());
    profile_ = testing_profile_manager_.CreateTestingProfile(
        kTestUserEmail, {{HistoryServiceFactory::GetInstance(),
                          HistoryServiceFactory::GetDefaultFactory()}});
    EXPECT_TRUE(profile_);

    web_contents_ = factory_.CreateWebContents(profile_);
  }

  Profile* profile() { return profile_; }
  content::WebContents* web_contents() { return web_contents_; }
  TestingPrefServiceSimple* local_state() {
    return testing_profile_manager_.local_state()->Get();
  }

 private:
  content::BrowserTaskEnvironment task_environment_;

#if BUILDFLAG(IS_CHROMEOS_ASH)
  std::unique_ptr<user_manager::ScopedUserManager> scoped_user_manager_;
#endif

  TestingProfileManager testing_profile_manager_;
  TestingProfile* profile_ = nullptr;
  content::TestWebContentsFactory factory_;
  content::WebContents* web_contents_;  // Weak. Owned by factory_.

  DISALLOW_COPY_AND_ASSIGN(ScopedWebContentsTestHelper);
};

class PageInfoBubbleViewTest : public testing::Test,
                               public ::testing::WithParamInterface<bool> {
 public:
  PageInfoBubbleViewTest() {
    feature_list_.InitWithFeatureState(page_info::kPageInfoV2Desktop,
                                       is_page_info_v2_enabled());
    web_contents_helper_ = std::make_unique<ScopedWebContentsTestHelper>();
    views_helper_ = std::make_unique<views::ScopedViewsTestHelper>(
        std::make_unique<ChromeTestViewsDelegate<>>());
  }

  PageInfoBubbleViewTest(const PageInfoBubbleViewTest& chip) = delete;
  PageInfoBubbleViewTest& operator=(const PageInfoBubbleViewTest& chip) =
      delete;

  // testing::Test:
  void SetUp() override {
    views::Widget::InitParams parent_params;
    parent_params.context = views_helper_->GetContext();
    parent_window_ = new views::Widget();
    parent_window_->Init(std::move(parent_params));

    mock_sentiment_service_ = static_cast<MockTrustSafetySentimentService*>(
        TrustSafetySentimentServiceFactory::GetInstance()
            ->SetTestingFactoryAndUse(
                web_contents_helper_->profile(),
                base::BindRepeating(&BuildMockTrustSafetySentimentService)));

    content::WebContents* web_contents = web_contents_helper_->web_contents();
    content_settings::PageSpecificContentSettings::CreateForWebContents(
        web_contents,
        std::make_unique<chrome::PageSpecificContentSettingsDelegate>(
            web_contents));
    api_ = std::make_unique<test::PageInfoBubbleViewTestApi>(
        parent_window_->GetNativeView(), web_contents_helper_->profile(),
        web_contents, is_page_info_v2_enabled());
  }

  void TearDown() override {
    parent_window_->CloseNow();
  }

 protected:
  bool is_page_info_v2_enabled() const { return GetParam(); }

  base::test::ScopedFeatureList feature_list_;
  std::unique_ptr<ScopedWebContentsTestHelper> web_contents_helper_;
  std::unique_ptr<views::ScopedViewsTestHelper> views_helper_;
  MockTrustSafetySentimentService* mock_sentiment_service_;

  views::Widget* parent_window_ = nullptr;  // Weak. Owned by the NativeWidget.
  std::unique_ptr<test::PageInfoBubbleViewTestApi> api_;
};

base::Value ReadJson(base::StringPiece json) {
  base::JSONReader::ValueWithError result =
      base::JSONReader::ReadAndReturnValueWithError(json);
  EXPECT_TRUE(result.value) << result.error_message;
  return result.value ? std::move(*result.value) : base::Value();
}

views::Label* GetChosenObjectTitle(const views::View::Views& children) {
  views::View* labels_container = children[1];
  return static_cast<views::Label*>(labels_container->children()[0]);
}

views::Button* GetChosenObjectButton(const views::View::Views& children) {
  return static_cast<views::Button*>(children[2]);
}

views::Label* GetChosenObjectDescriptionLabel(
    const views::View::Views& children) {
  views::View* labels_container = children[1];
  return static_cast<views::Label*>(labels_container->children()[1]);
}

}  // namespace

// Each permission selector row is like this: [icon] [label] [selector]
constexpr size_t kViewsPerPermissionRow = 3;

TEST_P(PageInfoBubbleViewTest, NotificationPermissionRevokeUkm) {
  GURL origin_url = GURL(kUrl).GetOrigin();
  TestingProfile* profile =
      static_cast<TestingProfile*>(web_contents_helper_->profile());
  ukm::TestAutoSetUkmRecorder ukm_recorder;
  auto* history_service = HistoryServiceFactory::GetForProfile(
      profile, ServiceAccessType::EXPLICIT_ACCESS);
  history_service->AddPage(origin_url, base::Time::Now(),
                           history::SOURCE_BROWSED);
  base::RunLoop origin_queried_waiter;
  history_service->set_origin_queried_closure_for_testing(
      origin_queried_waiter.QuitClosure());

  PermissionInfoList list(1);
  list.back().type = ContentSettingsType::NOTIFICATIONS;
  list.back().source = content_settings::SETTING_SOURCE_USER;

  list.back().setting = CONTENT_SETTING_ALLOW;
  api_->SetPermissionInfo(list);

  list.back().setting = CONTENT_SETTING_BLOCK;
  api_->SetPermissionInfo(list);

  origin_queried_waiter.Run();

  auto entries = ukm_recorder.GetEntriesByName("Permission");
  EXPECT_EQ(1u, entries.size());
  auto* entry = entries.front();

  ukm_recorder.ExpectEntrySourceHasUrl(entry, origin_url);
  EXPECT_EQ(*ukm_recorder.GetEntryMetric(entry, "Source"),
            static_cast<int64_t>(permissions::PermissionSourceUI::OIB));
  size_t num_values = 0;
  EXPECT_EQ(*ukm_recorder.GetEntryMetric(entry, "PermissionType"),
            ContentSettingTypeToHistogramValue(
                ContentSettingsType::NOTIFICATIONS, &num_values));
  EXPECT_EQ(*ukm_recorder.GetEntryMetric(entry, "Action"),
            static_cast<int64_t>(permissions::PermissionAction::REVOKED));
}

// Test UI construction and reconstruction via
// PageInfoBubbleView::SetPermissionInfo().
TEST_P(PageInfoBubbleViewTest, SetPermissionInfo) {
  // This test exercises PermissionSelectorRow in a way that it is not used in
  // practice. In practice, every setting in PermissionSelectorRow starts off
  // "set", so there is always one option checked in the resulting MenuModel.
  // This test creates settings that are left at their defaults, leading to zero
  // checked options, and checks that the text on the MenuButtons is right.

  PermissionInfoList list(1);
  list.back().type = ContentSettingsType::GEOLOCATION;
  list.back().source = content_settings::SETTING_SOURCE_USER;
  list.back().setting = CONTENT_SETTING_BLOCK;

  // Initially, no permissions are shown because they are all set to default.
  size_t num_expected_children = 0;
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(num_expected_children));
  if (is_page_info_v2_enabled()) {
    EXPECT_FALSE(api_->reset_permissions_button());
  }

  num_expected_children += is_page_info_v2_enabled()
                               ? list.size()
                               : kViewsPerPermissionRow * list.size();
  list.back().setting = CONTENT_SETTING_ALLOW;
  api_->SetPermissionInfo(list);
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(num_expected_children));

  if (is_page_info_v2_enabled()) {
    EXPECT_TRUE(api_->reset_permissions_button()->GetVisible());
    EXPECT_TRUE(api_->reset_permissions_button()->GetEnabled());
    EXPECT_EQ(u"Reset permission", api_->reset_permissions_button()->GetText());
    PermissionToggleRowView* toggle_view = api_->GetPermissionToggleRowAt(0);
    EXPECT_TRUE(toggle_view);
  } else {
    PermissionSelectorRow* selector = api_->GetPermissionSelectorAt(0);
    EXPECT_TRUE(selector);
  }

  // Verify labels match the settings on the PermissionInfoList.
  EXPECT_EQ(u"Location", api_->GetPermissionLabelTextAt(0));
  if (is_page_info_v2_enabled()) {
    EXPECT_TRUE(api_->GetPermissionToggleIsOnAt(0));
  } else {
    EXPECT_EQ(u"Allow", api_->GetPermissionComboboxTextAt(0));
  }

  // Verify calling SetPermissionInfo() directly updates the UI.
  list.back().setting = CONTENT_SETTING_BLOCK;
  api_->SetPermissionInfo(list);
  if (is_page_info_v2_enabled()) {
    EXPECT_FALSE(api_->GetPermissionToggleIsOnAt(0));
  } else {
    EXPECT_EQ(u"Block", api_->GetPermissionComboboxTextAt(0));
  }

  // Simulate a user selection via the UI. Note this will also cover logic in
  // PageInfo to update the pref.
  if (is_page_info_v2_enabled()) {
    api_->SimulateTogglingPermissionAt(0);
  } else {
    api_->SimulateUserSelectingComboboxItemAt(0, 1);
  }
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(num_expected_children));
  if (is_page_info_v2_enabled()) {
    EXPECT_TRUE(api_->GetPermissionToggleIsOnAt(0));
  } else {
    EXPECT_EQ(u"Allow", api_->GetPermissionComboboxTextAt(0));
  }

  if (is_page_info_v2_enabled()) {
    const ui::MouseEvent event(ui::ET_MOUSE_PRESSED, gfx::Point(), gfx::Point(),
                               ui::EventTimeForNow(), 0, 0);
    views::test::ButtonTestApi(api_->reset_permissions_button())
        .NotifyClick(event);
    // After resetting permissions, button doesn't disappear but is disabled.
    EXPECT_TRUE(api_->reset_permissions_button()->GetVisible());
    EXPECT_FALSE(api_->reset_permissions_button()->GetEnabled());

    // In the ask state, the toggle is in the off state, indicating that
    // permission isn't granted.
    EXPECT_FALSE(api_->GetPermissionToggleIsOnAt(0));
  } else {
    // User cannot set permission to default on the main page in the v2.
    // Setting to the default via the UI should keep the button around.
    api_->SimulateUserSelectingComboboxItemAt(0, 0);
    EXPECT_EQ(u"Ask (default)", api_->GetPermissionComboboxTextAt(0));
    EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(num_expected_children));
  }

  // However, since the setting is now default, recreating the dialog with those
  // settings should omit the permission from the UI.
  //
  // TODO(https://crbug.com/829576): Reconcile the comment above with the fact
  // that |num_expected_children| is not, at this point, 0 and therefore the
  // permission is not being omitted from the UI.
  api_->SetPermissionInfo(list);
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(num_expected_children));
}

// Test resetting blocked in Incognito permission.
TEST_P(PageInfoBubbleViewTest, ResetBlockedInIncognitoPermission) {
  if (!is_page_info_v2_enabled()) {
    return;
  }

  // This test uses OnSitePermissionChanged to inform the bubble of the
  // permission changes, in production code this means the user interacted with
  // page info and is thus reported to the sentiment service, inflating the
  // expected interaction count.
  EXPECT_CALL(*mock_sentiment_service_, InteractedWithPageInfo).Times(5);

  api_->SetOffTheRecordProfile();

  PermissionInfoList list(1);
  list.back().type = ContentSettingsType::NOTIFICATIONS;
  list.back().source = content_settings::SETTING_SOURCE_POLICY;
  list.back().setting = CONTENT_SETTING_BLOCK;

  // Initially, no permissions are shown because they are all set to default.
  size_t num_expected_children = 0;
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(num_expected_children));
  EXPECT_FALSE(api_->reset_permissions_button());

  num_expected_children = list.size();
  api_->SetPermissionInfo(list);
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(num_expected_children));

  // Because permission is autoblocked, no reset button initially is shown.
  EXPECT_FALSE(api_->reset_permissions_button()->GetVisible());
  EXPECT_FALSE(api_->reset_permissions_button()->GetEnabled());

  // Autoblocked permissions don't have toggles or state labels.
  EXPECT_FALSE(api_->GetToggleViewAt(0));
  EXPECT_FALSE(api_->GetStateLabelAt(0));

  // Verify labels match the settings on the PermissionInfoList.
  EXPECT_EQ(u"Notifications", api_->GetPermissionLabelTextAt(0));

  PageInfo::PermissionInfo window_placement_permission;
  window_placement_permission.type = ContentSettingsType::WINDOW_PLACEMENT;
  window_placement_permission.setting = CONTENT_SETTING_ALLOW;
  window_placement_permission.default_setting = CONTENT_SETTING_ASK;
  window_placement_permission.source = content_settings::SETTING_SOURCE_USER;
  list.push_back(window_placement_permission);

  num_expected_children = list.size();
  api_->SetPermissionInfo(list);
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(num_expected_children));

  // Because a non-managed permission was added, reset button is visible and
  // enabled.
  EXPECT_TRUE(api_->reset_permissions_button()->GetVisible());
  EXPECT_TRUE(api_->reset_permissions_button()->GetEnabled());
  // Although there are only one resettable permission, multiple rows are
  // shown. Because of that use plural version of the "permission" word.
  EXPECT_EQ(u"Reset permissions", api_->reset_permissions_button()->GetText());

  // User managed permissions have toggles. |camera_permission| is allowed and
  // the toggle must be on.
  EXPECT_TRUE(api_->GetToggleViewAt(1));
  EXPECT_TRUE(api_->GetPermissionToggleIsOnAt(1));

  const ui::MouseEvent event(ui::ET_MOUSE_PRESSED, gfx::Point(), gfx::Point(),
                             ui::EventTimeForNow(), 0, 0);
  views::test::ButtonTestApi(api_->reset_permissions_button())
      .NotifyClick(event);
  // After resetting permissions, button doesn't disappear but is disabled.
  EXPECT_TRUE(api_->reset_permissions_button()->GetVisible());
  EXPECT_FALSE(api_->reset_permissions_button()->GetEnabled());

  // Show state label for user managed permission, indicating that permission
  // is in the default ask state now. Autoblocked permission doesn't change.
  EXPECT_FALSE(api_->GetStateLabelAt(0));
  EXPECT_EQ(u"Can ask to open and place windows on your screens",
            api_->GetStateLabelAt(1)->GetText());

  // In the ask state, the toggle is in the off state, indicating that
  // permission isn't granted.
  EXPECT_FALSE(api_->GetPermissionToggleIsOnAt(1));
}

// Test UI construction and reconstruction with USB devices.
TEST_P(PageInfoBubbleViewTest, SetPermissionInfoWithUsbDevice) {
  EXPECT_CALL(*mock_sentiment_service_, InteractedWithPageInfo);
  constexpr size_t kExpectedChildren = 0;
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(kExpectedChildren));

  const auto origin = url::Origin::Create(GURL(kUrl));

  // Connect the UsbChooserContext with FakeUsbDeviceManager.
  device::FakeUsbDeviceManager usb_device_manager;
  mojo::PendingRemote<device::mojom::UsbDeviceManager> usb_manager;
  usb_device_manager.AddReceiver(usb_manager.InitWithNewPipeAndPassReceiver());
  UsbChooserContext* store =
      UsbChooserContextFactory::GetForProfile(web_contents_helper_->profile());
  store->SetDeviceManagerForTesting(std::move(usb_manager));

  auto device_info = usb_device_manager.CreateAndAddDevice(
      0, 0, "Google", "Gizmo", "1234567890");
  store->GrantDevicePermission(origin, *device_info);

  PermissionInfoList list;
  api_->SetPermissionInfo(list);
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(kExpectedChildren + 1));

  const auto& chosen_object_children = api_->GetChosenObjectChildren();
  EXPECT_EQ(3u, chosen_object_children.size());

  views::Label* label = GetChosenObjectTitle(chosen_object_children);
  EXPECT_EQ(u"Gizmo", label->GetText());

  views::Button* button = GetChosenObjectButton(chosen_object_children);
  const ui::MouseEvent event(ui::ET_MOUSE_PRESSED, gfx::Point(), gfx::Point(),
                             ui::EventTimeForNow(), 0, 0);
  views::test::ButtonTestApi(button).NotifyClick(event);
  api_->SetPermissionInfo(list);
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(kExpectedChildren));
  EXPECT_FALSE(store->HasDevicePermission(origin, *device_info));
}

// Test resetting USB devices permission.
TEST_P(PageInfoBubbleViewTest, ResetPermissionInfoWithUsbDevice) {
  if (!is_page_info_v2_enabled()) {
    return;
  }
  EXPECT_CALL(*mock_sentiment_service_, InteractedWithPageInfo);

  constexpr size_t kExpectedChildren = 0;
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(kExpectedChildren));
  EXPECT_FALSE(api_->reset_permissions_button());

  const auto origin = url::Origin::Create(GURL(kUrl));

  // Connect the UsbChooserContext with FakeUsbDeviceManager.
  device::FakeUsbDeviceManager usb_device_manager;
  mojo::PendingRemote<device::mojom::UsbDeviceManager> usb_manager;
  usb_device_manager.AddReceiver(usb_manager.InitWithNewPipeAndPassReceiver());
  UsbChooserContext* store =
      UsbChooserContextFactory::GetForProfile(web_contents_helper_->profile());
  store->SetDeviceManagerForTesting(std::move(usb_manager));

  auto device_info = usb_device_manager.CreateAndAddDevice(
      0, 0, "Google", "Gizmo", "1234567890");
  store->GrantDevicePermission(origin, *device_info);

  PermissionInfoList list;
  api_->SetPermissionInfo(list);
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(kExpectedChildren + 1));
  EXPECT_TRUE(api_->reset_permissions_button()->GetVisible());
  EXPECT_TRUE(api_->reset_permissions_button()->GetEnabled());
  EXPECT_EQ(u"Reset permission", api_->reset_permissions_button()->GetText());

  const auto& chosen_object_children = api_->GetChosenObjectChildren();
  EXPECT_EQ(3u, chosen_object_children.size());

  views::Label* label = GetChosenObjectTitle(chosen_object_children);
  EXPECT_EQ(u"Gizmo", label->GetText());

  const ui::MouseEvent event(ui::ET_MOUSE_PRESSED, gfx::Point(), gfx::Point(),
                             ui::EventTimeForNow(), 0, 0);
  views::test::ButtonTestApi(api_->reset_permissions_button())
      .NotifyClick(event);
  api_->SetPermissionInfo(list);
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(kExpectedChildren));
  EXPECT_FALSE(api_->reset_permissions_button());
  EXPECT_FALSE(store->HasDevicePermission(origin, *device_info));
}

namespace {

constexpr char kWebUsbPolicySetting[] = R"(
    [
      {
        "devices": [{ "vendor_id": 6353, "product_id": 5678 }],
        "urls": ["http://www.example.com"]
      }
    ])";

}  // namespace

// Test UI construction and reconstruction with policy USB devices.
TEST_P(PageInfoBubbleViewTest, SetPermissionInfoWithPolicyUsbDevices) {
  constexpr size_t kExpectedChildren = 0;
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(kExpectedChildren));

  const auto origin = url::Origin::Create(GURL(kUrl));

  // Add the policy setting to prefs.
  Profile* profile = web_contents_helper_->profile();
  profile->GetPrefs()->Set(prefs::kManagedWebUsbAllowDevicesForUrls,
                           ReadJson(kWebUsbPolicySetting));
  UsbChooserContext* store = UsbChooserContextFactory::GetForProfile(profile);

  auto objects = store->GetGrantedObjects(origin);
  EXPECT_EQ(objects.size(), 1u);

  PermissionInfoList list;
  api_->SetPermissionInfo(list);
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(kExpectedChildren + 1));

  const auto& chosen_object_children = api_->GetChosenObjectChildren();
  EXPECT_EQ(3u, chosen_object_children.size());

  views::Label* label = GetChosenObjectTitle(chosen_object_children);
  EXPECT_EQ(u"Unknown product 0x162E from Google Inc.", label->GetText());

  views::Button* button = GetChosenObjectButton(chosen_object_children);
  EXPECT_EQ(button->GetState(), views::Button::STATE_DISABLED);

  views::Label* desc_label =
      GetChosenObjectDescriptionLabel(chosen_object_children);
  EXPECT_EQ(u"USB device allowed by your administrator", desc_label->GetText());

  // Policy granted USB permissions should not be able to be deleted.
  const ui::MouseEvent event(ui::ET_MOUSE_PRESSED, gfx::Point(), gfx::Point(),
                             ui::EventTimeForNow(), 0, 0);
  views::test::ButtonTestApi(button).NotifyClick(event);
  api_->SetPermissionInfo(list);
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(kExpectedChildren + 1));
}

// Test UI construction and reconstruction with both user and policy USB
// devices.
TEST_P(PageInfoBubbleViewTest, SetPermissionInfoWithUserAndPolicyUsbDevices) {
  EXPECT_CALL(*mock_sentiment_service_, InteractedWithPageInfo);
  constexpr size_t kExpectedChildren = 0;
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(kExpectedChildren));

  const auto origin = url::Origin::Create(GURL(kUrl));

  // Add the policy setting to prefs.
  Profile* profile = web_contents_helper_->profile();
  profile->GetPrefs()->Set(prefs::kManagedWebUsbAllowDevicesForUrls,
                           ReadJson(kWebUsbPolicySetting));

  // Connect the UsbChooserContext with FakeUsbDeviceManager.
  device::FakeUsbDeviceManager usb_device_manager;
  mojo::PendingRemote<device::mojom::UsbDeviceManager> device_manager;
  usb_device_manager.AddReceiver(
      device_manager.InitWithNewPipeAndPassReceiver());
  UsbChooserContext* store = UsbChooserContextFactory::GetForProfile(profile);
  store->SetDeviceManagerForTesting(std::move(device_manager));

  auto device_info = usb_device_manager.CreateAndAddDevice(
      0, 0, "Google", "Gizmo", "1234567890");
  store->GrantDevicePermission(origin, *device_info);

  auto objects = store->GetGrantedObjects(origin);
  EXPECT_EQ(objects.size(), 2u);

  PermissionInfoList list;
  api_->SetPermissionInfo(list);
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(kExpectedChildren + 2));

  const ui::MouseEvent event(ui::ET_MOUSE_PRESSED, gfx::Point(), gfx::Point(),
                             ui::EventTimeForNow(), 0, 0);

  // The first object is the user granted permission for the "Gizmo" device.
  {
    const auto& chosen_object_children = api_->GetChosenObjectChildren();
    EXPECT_EQ(3u, chosen_object_children.size());

    views::Label* label = GetChosenObjectTitle(chosen_object_children);
    EXPECT_EQ(u"Gizmo", label->GetText());

    views::Button* button = GetChosenObjectButton(chosen_object_children);
    EXPECT_NE(button->GetState(), views::Button::STATE_DISABLED);

    views::Label* desc_label =
        GetChosenObjectDescriptionLabel(chosen_object_children);
    EXPECT_EQ(u"USB device", desc_label->GetText());

    views::test::ButtonTestApi(button).NotifyClick(event);
    api_->SetPermissionInfo(list);
    EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(kExpectedChildren + 1));
    EXPECT_FALSE(store->HasDevicePermission(origin, *device_info));
  }

  // The policy granted permission should now be the first child, since the user
  // permission was deleted.
  {
    const auto& chosen_object_children = api_->GetChosenObjectChildren();
    EXPECT_EQ(3u, chosen_object_children.size());

    views::Label* label = GetChosenObjectTitle(chosen_object_children);
    EXPECT_EQ(u"Unknown product 0x162E from Google Inc.", label->GetText());

    views::Button* button = GetChosenObjectButton(chosen_object_children);
    EXPECT_EQ(button->GetState(), views::Button::STATE_DISABLED);

    views::Label* desc_label =
        GetChosenObjectDescriptionLabel(chosen_object_children);
    EXPECT_EQ(u"USB device allowed by your administrator",
              desc_label->GetText());

    views::test::ButtonTestApi(button).NotifyClick(event);
    api_->SetPermissionInfo(list);
    EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(kExpectedChildren + 1));
  }
}

TEST_P(PageInfoBubbleViewTest, SetPermissionInfoForUsbGuard) {
  // This test exercises PermissionSelectorRow in a way that it is not used in
  // practice. In practice, every setting in PermissionSelectorRow starts off
  // "set", so there is always one option checked in the resulting MenuModel.
  // This test creates settings that are left at their defaults, leading to zero
  // checked options, and checks that the text on the MenuButtons is right.
  PermissionInfoList list(1);
  list.back().type = ContentSettingsType::USB_GUARD;
  list.back().source = content_settings::SETTING_SOURCE_USER;
  list.back().setting = CONTENT_SETTING_ASK;

  // Initially, no permissions are shown because they are all set to default.
  size_t num_expected_children = 0;
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(num_expected_children));

  // Verify calling SetPermissionInfo() directly updates the UI.
  num_expected_children += is_page_info_v2_enabled()
                               ? list.size()
                               : kViewsPerPermissionRow * list.size();
  list.back().setting = CONTENT_SETTING_BLOCK;
  api_->SetPermissionInfo(list);
  if (is_page_info_v2_enabled()) {
    EXPECT_FALSE(api_->GetPermissionToggleIsOnAt(0));
  } else {
    EXPECT_EQ(u"Block", api_->GetPermissionComboboxTextAt(0));
  }

  // Simulate a user selection via the UI. Note this will also cover logic in
  // PageInfo to update the pref.
  if (is_page_info_v2_enabled()) {
    api_->SimulateTogglingPermissionAt(0);
  } else {
    api_->SimulateUserSelectingComboboxItemAt(0, 2);
  }
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(num_expected_children));
  if (is_page_info_v2_enabled()) {
    EXPECT_TRUE(api_->GetPermissionToggleIsOnAt(0));
  } else {
    EXPECT_EQ(u"Ask", api_->GetPermissionComboboxTextAt(0));
  }

  if (!is_page_info_v2_enabled()) {
    // User cannot set permission to default on the main page in the v2.
    // Setting to the default via the UI should keep the button around.
    api_->SimulateUserSelectingComboboxItemAt(0, 0);
    EXPECT_EQ(u"Ask (default)", api_->GetPermissionComboboxTextAt(0));
    EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(num_expected_children));
  }

  // However, since the setting is now default, recreating the dialog with
  // those settings should omit the permission from the UI.
  //
  // TODO(https://crbug.com/829576): Reconcile the comment above with the fact
  // that |num_expected_children| is not, at this point, 0 and therefore the
  // permission is not being omitted from the UI.
  api_->SetPermissionInfo(list);
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(num_expected_children));
}

// Test UI construction and reconstruction with policy USB devices.
TEST_P(PageInfoBubbleViewTest, SetPermissionInfoWithPolicySerialPorts) {
  constexpr size_t kExpectedChildren = 0;
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(kExpectedChildren));

  // Add the policy setting to prefs.
  web_contents_helper_->local_state()->Set(
      prefs::kManagedSerialAllowUsbDevicesForUrls, ReadJson(R"([
               {
                 "devices": [{ "vendor_id": 6353, "product_id": 5678 }],
                 "urls": [ "http://www.example.com" ]
               }
             ])"));

  PermissionInfoList list;
  api_->SetPermissionInfo(list);
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(kExpectedChildren + 1));

  const auto& chosen_object_children = api_->GetChosenObjectChildren();
  EXPECT_EQ(3u, chosen_object_children.size());

  views::Label* label = GetChosenObjectTitle(chosen_object_children);
  EXPECT_EQ(u"USB device from Google Inc. (product 162E)", label->GetText());

  views::Button* button = GetChosenObjectButton(chosen_object_children);
  EXPECT_EQ(button->GetState(), views::Button::STATE_DISABLED);

  views::Label* desc_label =
      GetChosenObjectDescriptionLabel(chosen_object_children);
  EXPECT_EQ(u"Serial port allowed by your administrator",
            desc_label->GetText());

  // Policy granted serial port permissions should not be able to be deleted.
  const ui::MouseEvent event(ui::ET_MOUSE_PRESSED, gfx::Point(), gfx::Point(),
                             ui::EventTimeForNow(), 0, 0);
  views::test::ButtonTestApi(button).NotifyClick(event);
  api_->SetPermissionInfo(list);
  EXPECT_TRUE(api_->ValidatePermissionsChildrenCount(kExpectedChildren + 1));
}

// Test that updating the number of cookies used by the current page doesn't add
// any extra views to Page Info.
TEST_P(PageInfoBubbleViewTest, UpdatingSiteDataRetainsLayout) {
#if defined(OS_WIN) && BUILDFLAG(ENABLE_VR)
  constexpr size_t kExpectedChildren = 6;
#else
  constexpr size_t kExpectedChildren = 5;
#endif

  EXPECT_EQ(kExpectedChildren, api_->current_view()->children().size());

  // Create a fake list of cookies.
  PageInfoUI::CookieInfo first_party_cookies;
  first_party_cookies.allowed = 10;
  first_party_cookies.blocked = 0;
  first_party_cookies.is_first_party = true;

  PageInfoUI::CookieInfo third_party_cookies;
  third_party_cookies.allowed = 6;
  third_party_cookies.blocked = 32;
  third_party_cookies.is_first_party = false;

  const CookieInfoList cookies = {first_party_cookies, third_party_cookies};

  // Update the number of cookies.
  api_->SetCookieInfo(cookies);

  EXPECT_EQ(kExpectedChildren, api_->current_view()->children().size());

  std::u16string expected;
  // Check the number of cookies shown is correct.
  if (is_page_info_v2_enabled()) {
    expected = l10n_util::GetPluralStringFUTF16(
        IDS_PAGE_INFO_NUM_COOKIES,
        first_party_cookies.allowed + third_party_cookies.allowed);
  } else {
    expected = l10n_util::GetPluralStringFUTF16(
        IDS_PAGE_INFO_NUM_COOKIES_PARENTHESIZED,
        first_party_cookies.allowed + third_party_cookies.allowed);
  }
  size_t index = api_->GetCookiesLinkText().find(expected);
  EXPECT_NE(std::string::npos, index);
}

// Tests opening the bubble between navigation start and finish. The bubble
// should be updated to reflect the secure state after the navigation commits.
TEST_P(PageInfoBubbleViewTest, OpenPageInfoBubbleAfterNavigationStart) {
  SecurityStateTabHelper::CreateForWebContents(
      web_contents_helper_->web_contents());
  std::unique_ptr<content::NavigationSimulator> navigation =
      content::NavigationSimulator::CreateRendererInitiated(
          GURL(kSecureUrl),
          web_contents_helper_->web_contents()->GetMainFrame());
  navigation->Start();
  api_->CreateView();
  if (is_page_info_v2_enabled()) {
    EXPECT_EQ(kHostname, api_->GetWindowTitle());
    EXPECT_FALSE(api_->certificate_button());
    EXPECT_TRUE(api_->security_details_label());
    EXPECT_EQ(api_->GetSecuritySummaryText(),
              l10n_util::GetStringUTF16(IDS_PAGE_INFO_NOT_SECURE_SUMMARY));
  } else {
    EXPECT_EQ(l10n_util::GetStringUTF16(IDS_PAGE_INFO_NOT_SECURE_SUMMARY),
              api_->GetWindowTitle());
  }

  // Set up a test SSLInfo so that Page Info sees the connection as secure.
  uint16_t cipher_suite = 0xc02f;  // TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256
  int connection_status = 0;
  net::SSLConnectionStatusSetCipherSuite(cipher_suite, &connection_status);
  net::SSLConnectionStatusSetVersion(net::SSL_CONNECTION_VERSION_TLS1_2,
                                     &connection_status);
  net::SSLInfo ssl_info;
  ssl_info.connection_status = connection_status;
  ssl_info.cert =
      net::ImportCertFromFile(net::GetTestCertsDirectory(), "ok_cert.pem");
  ASSERT_TRUE(ssl_info.cert);

  navigation->SetSSLInfo(ssl_info);

  navigation->Commit();
  if (is_page_info_v2_enabled()) {
    // In page info v2, in secure state description and learn more link aren't
    // shown on the main page.
    EXPECT_EQ(kHostname, api_->GetWindowTitle());
    EXPECT_FALSE(api_->security_details_label());
    EXPECT_EQ(api_->GetSecurityInformationButtonText(),
              l10n_util::GetStringUTF16(IDS_PAGE_INFO_SECURE_SUMMARY));

    api_->navigation_handler()->OpenSecurityPage();
    EXPECT_TRUE(api_->security_details_label());
  } else {
    EXPECT_EQ(l10n_util::GetStringUTF16(IDS_PAGE_INFO_SECURE_SUMMARY),
              api_->GetWindowTitle());
  }
}

TEST_P(PageInfoBubbleViewTest, EnsureCloseCallback) {
  api_->current_view()->GetWidget()->CloseWithReason(
      views::Widget::ClosedReason::kCloseButtonClicked);
  api_->WaitForBubbleClose();
  EXPECT_EQ(false, api_->reload_prompt());
  EXPECT_EQ(views::Widget::ClosedReason::kCloseButtonClicked,
            api_->closed_reason());
}

TEST_P(PageInfoBubbleViewTest, CheckHeaderInteractions) {
  // Confirm that interactions with the header tips are reported to the
  // sentiment service correctly.
  const ui::MouseEvent event(ui::ET_MOUSE_PRESSED, gfx::Point(), gfx::Point(),
                             ui::EventTimeForNow(), 0, 0);
  if (is_page_info_v2_enabled()) {
    // Navigating to the security page constitutes an interaction.
    EXPECT_CALL(*mock_sentiment_service_, InteractedWithPageInfo).Times(3);
    api_->navigation_handler()->OpenSecurityPage();
    auto* page_view = static_cast<PageInfoSecurityContentView*>(
        api_->current_view()->children()[1]);
    page_view->SecurityDetailsClicked(event);
    page_view->ResetDecisionsClicked();
  } else {
    EXPECT_CALL(*mock_sentiment_service_, InteractedWithPageInfo).Times(2);
    auto* bubble_view = static_cast<PageInfoBubbleView*>(api_->current_view());
    bubble_view->SecurityDetailsClicked(event);
    bubble_view->ResetDecisionsClicked();
  }
}

TEST_P(PageInfoBubbleViewTest, CertificateButtonShowsEvCertDetails) {
  SecurityStateTabHelper::CreateForWebContents(
      web_contents_helper_->web_contents());
  std::unique_ptr<content::NavigationSimulator> navigation =
      content::NavigationSimulator::CreateRendererInitiated(
          GURL(kSecureUrl),
          web_contents_helper_->web_contents()->GetMainFrame());
  navigation->Start();
  api_->CreateView();

  // Set up a test SSLInfo so that Page Info sees the connection as secure and
  // using an EV certificate.
  uint16_t cipher_suite = 0xc02f;  // TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256
  int connection_status = 0;
  net::SSLConnectionStatusSetCipherSuite(cipher_suite, &connection_status);
  net::SSLConnectionStatusSetVersion(net::SSL_CONNECTION_VERSION_TLS1_2,
                                     &connection_status);
  net::SSLInfo ssl_info;
  ssl_info.connection_status = connection_status;
  ssl_info.cert =
      net::ImportCertFromFile(net::GetTestCertsDirectory(), "ev_test.pem");
  ASSERT_TRUE(ssl_info.cert);
  ssl_info.cert_status = net::CERT_STATUS_IS_EV;

  navigation->SetSSLInfo(ssl_info);

  navigation->Commit();
  if (is_page_info_v2_enabled()) {
    // In page info v2, in secure state certificate button isn't shown on the
    // main page.
    EXPECT_EQ(kHostname, api_->GetWindowTitle());
    EXPECT_FALSE(api_->certificate_button());
    EXPECT_FALSE(api_->security_summary_label());
    EXPECT_EQ(api_->GetSecurityInformationButtonText(),
              l10n_util::GetStringUTF16(IDS_PAGE_INFO_SECURE_SUMMARY));

    api_->navigation_handler()->OpenSecurityPage();
    EXPECT_TRUE(api_->certificate_button());
    EXPECT_EQ(l10n_util::GetStringUTF16(IDS_PAGE_INFO_SECURE_SUMMARY),
              api_->GetSecuritySummaryText());
  } else {
    EXPECT_EQ(l10n_util::GetStringUTF16(IDS_PAGE_INFO_SECURE_SUMMARY),
              api_->GetWindowTitle());
  }

  // The certificate button subtitle should show the EV certificate organization
  // name and country of incorporation.
  EXPECT_EQ(l10n_util::GetStringFUTF16(
                IDS_PAGE_INFO_SECURITY_TAB_SECURE_IDENTITY_EV_VERIFIED,
                u"Test Org", u"US"),
            api_->GetCertificateButtonSubtitleText());
}

// Regression test for crbug.com/1069113. Test cert includes country and state
// but not locality.
TEST_P(PageInfoBubbleViewTest, EvDetailsShowForCertWithStateButNoLocality) {
  SecurityStateTabHelper::CreateForWebContents(
      web_contents_helper_->web_contents());
  std::unique_ptr<content::NavigationSimulator> navigation =
      content::NavigationSimulator::CreateRendererInitiated(
          GURL(kSecureUrl),
          web_contents_helper_->web_contents()->GetMainFrame());
  navigation->Start();
  api_->CreateView();

  // Set up a test SSLInfo so that Page Info sees the connection as secure and
  // using an EV certificate.
  uint16_t cipher_suite = 0xc02f;  // TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256
  int connection_status = 0;
  net::SSLConnectionStatusSetCipherSuite(cipher_suite, &connection_status);
  net::SSLConnectionStatusSetVersion(net::SSL_CONNECTION_VERSION_TLS1_2,
                                     &connection_status);
  net::SSLInfo ssl_info;
  ssl_info.connection_status = connection_status;
  ssl_info.cert = net::ImportCertFromFile(net::GetTestCertsDirectory(),
                                          "ev_test_state_only.pem");
  ASSERT_TRUE(ssl_info.cert);

  ssl_info.cert_status = net::CERT_STATUS_IS_EV;

  navigation->SetSSLInfo(ssl_info);

  navigation->Commit();
  if (is_page_info_v2_enabled()) {
    // In page info v2, in secure state certificate button isn't shown on the
    // main page.
    EXPECT_EQ(kHostname, api_->GetWindowTitle());
    EXPECT_FALSE(api_->certificate_button());
    EXPECT_FALSE(api_->security_summary_label());
    EXPECT_EQ(api_->GetSecurityInformationButtonText(),
              l10n_util::GetStringUTF16(IDS_PAGE_INFO_SECURE_SUMMARY));

    api_->navigation_handler()->OpenSecurityPage();
    EXPECT_TRUE(api_->certificate_button());
    EXPECT_EQ(l10n_util::GetStringUTF16(IDS_PAGE_INFO_SECURE_SUMMARY),
              api_->GetSecuritySummaryText());
  } else {
    EXPECT_EQ(l10n_util::GetStringUTF16(IDS_PAGE_INFO_SECURE_SUMMARY),
              api_->GetWindowTitle());
  }

  // The certificate button subtitle should show the EV certificate organization
  // name and country of incorporation.
  EXPECT_EQ(l10n_util::GetStringFUTF16(
                IDS_PAGE_INFO_SECURITY_TAB_SECURE_IDENTITY_EV_VERIFIED,
                u"Test Org", u"US"),
            api_->GetCertificateButtonSubtitleText());
}

INSTANTIATE_TEST_SUITE_P(All,
                         PageInfoBubbleViewTest,
                         ::testing::Values(false, true));
