// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/chromeos/net/network_diagnostics/network_diagnostics_routine.h"

#include "base/test/bind.h"
#include "base/test/task_environment.h"
#include "content/public/test/browser_task_environment.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace chromeos {
namespace network_diagnostics {

namespace {

constexpr mojom::RoutineVerdict kInitialVerdict =
    mojom::RoutineVerdict::kNotRun;
constexpr mojom::RoutineVerdict kVerdict = mojom::RoutineVerdict::kNoProblem;

constexpr mojom::RoutineType kType = mojom::RoutineType::kHttpFirewall;

}  // namespace

class NetworkDiagnosticsRoutineTest : public ::testing::Test {
 public:
  // Minimal definition for an inherited NetworkDiagnosticsRoutine class.
  class TestNetworkDiagnosticsRoutine : public NetworkDiagnosticsRoutine {
   public:
    TestNetworkDiagnosticsRoutine() = default;
    TestNetworkDiagnosticsRoutine(const TestNetworkDiagnosticsRoutine&) =
        delete;
    TestNetworkDiagnosticsRoutine& operator=(
        const TestNetworkDiagnosticsRoutine&) = delete;
    ~TestNetworkDiagnosticsRoutine() override {}

    // NetworkDiagnosticRoutine:
    mojom::RoutineType Type() override { return kType; }
    void Run() override { ExecuteCallback(); }
    void AnalyzeResultsAndExecuteCallback() override {}
  };

  NetworkDiagnosticsRoutineTest() {
    test_network_diagnostics_routine_ =
        std::make_unique<TestNetworkDiagnosticsRoutine>();
  }

  TestNetworkDiagnosticsRoutine* test_network_diagnostics_routine() {
    return test_network_diagnostics_routine_.get();
  }

  void set_verdict(mojom::RoutineVerdict verdict) {
    test_network_diagnostics_routine()->set_verdict(verdict);
  }

  void set_problems(mojom::RoutineProblemsPtr problems) {
    test_network_diagnostics_routine()->set_problems(problems.Clone());
  }

  void set_can_run(bool can_run) { can_run_ = can_run; }

 private:
  std::unique_ptr<TestNetworkDiagnosticsRoutine>
      test_network_diagnostics_routine_;
  bool can_run_ = true;

  base::test::TaskEnvironment task_environment_;
};

TEST_F(NetworkDiagnosticsRoutineTest, TestDefaultRoutine) {
  TestNetworkDiagnosticsRoutine* routine = test_network_diagnostics_routine();
  set_verdict(kVerdict);
  base::RunLoop run_loop;

  routine->RunRoutine(
      base::BindLambdaForTesting([&](mojom::RoutineResultPtr result) {
        EXPECT_EQ(result->verdict, kVerdict);
        EXPECT_FALSE(result->timestamp.is_null());
        run_loop.Quit();
      }));

  run_loop.Run();
}

TEST_F(NetworkDiagnosticsRoutineTest, NotRunRoutine) {
  TestNetworkDiagnosticsRoutine* routine = test_network_diagnostics_routine();
  set_can_run(false);
  base::RunLoop run_loop;

  routine->RunRoutine(
      base::BindLambdaForTesting([&](mojom::RoutineResultPtr result) {
        EXPECT_EQ(result->verdict, kInitialVerdict);
        EXPECT_FALSE(result->timestamp.is_null());
        run_loop.Quit();
      }));

  run_loop.Run();
}

}  // namespace network_diagnostics
}  // namespace chromeos
