// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/views/permission_bubble/permission_prompt_bubble_view.h"
#include "base/strings/utf_string_conversions.h"
#include "chrome/browser/ui/views/permission_bubble/permission_prompt_style.h"
#include "chrome/grit/generated_resources.h"
#include "chrome/test/views/chrome_views_test_base.h"
#include "components/permissions/permission_util.h"
#include "components/permissions/request_type.h"
#include "components/permissions/test/mock_permission_request.h"
#include "ui/base/l10n/l10n_util.h"
#include "url/gurl.h"

using PermissionPromptBubbleViewTest = ChromeViewsTestBase;

namespace {

class TestDelegate : public permissions::PermissionPrompt::Delegate {
 public:
  explicit TestDelegate(
      const GURL& origin,
      const std::vector<permissions::RequestType> request_types) {
    std::transform(
        request_types.begin(), request_types.end(),
        std::back_inserter(requests_), [&](auto& request_type) {
          return std::make_unique<permissions::MockPermissionRequest>(
              origin, request_type);
        });
    std::transform(requests_.begin(), requests_.end(),
                   std::back_inserter(raw_requests_),
                   [](auto& req) { return req.get(); });
  }

  const std::vector<permissions::PermissionRequest*>& Requests() override {
    return raw_requests_;
  }

  GURL GetRequestingOrigin() const override {
    return raw_requests_.front()->requesting_origin();
  }

  GURL GetEmbeddingOrigin() const override {
    return GURL("https://embedder.example.com");
  }

  void Accept() override {}
  void AcceptThisTime() override {}
  void Deny() override {}
  void Closing() override {}

  bool WasCurrentRequestAlreadyDisplayed() override { return false; }

 private:
  std::vector<std::unique_ptr<permissions::PermissionRequest>> requests_;
  std::vector<permissions::PermissionRequest*> raw_requests_;
};

std::unique_ptr<PermissionPromptBubbleView> CreateBubble(
    TestDelegate* delegate) {
  return std::make_unique<PermissionPromptBubbleView>(
      nullptr, delegate, base::TimeTicks::Now(),
      PermissionPromptStyle::kBubbleOnly);
}

}  // namespace

TEST_F(PermissionPromptBubbleViewTest,
       AccessibleTitleMentionsOriginAndPermissions) {
  TestDelegate delegate(GURL("https://test.origin"),
                        {permissions::RequestType::kMicStream,
                         permissions::RequestType::kCameraStream});
  auto bubble = CreateBubble(&delegate);

  EXPECT_PRED_FORMAT2(::testing::IsSubstring, "microphone",
                      base::UTF16ToUTF8(bubble->GetAccessibleWindowTitle()));
  EXPECT_PRED_FORMAT2(::testing::IsSubstring, "camera",
                      base::UTF16ToUTF8(bubble->GetAccessibleWindowTitle()));
  // The scheme is not included.
  EXPECT_PRED_FORMAT2(::testing::IsSubstring, "test.origin",
                      base::UTF16ToUTF8(bubble->GetAccessibleWindowTitle()));
}

TEST_F(PermissionPromptBubbleViewTest,
       AccessibleTitleDoesNotMentionTooManyPermissions) {
  TestDelegate delegate(GURL(), {permissions::RequestType::kGeolocation,
                                 permissions::RequestType::kNotifications,
                                 permissions::RequestType::kMicStream,
                                 permissions::RequestType::kCameraStream});
  auto bubble = CreateBubble(&delegate);

  const auto title = base::UTF16ToUTF8(bubble->GetAccessibleWindowTitle());
  EXPECT_PRED_FORMAT2(::testing::IsSubstring, "location", title);
  EXPECT_PRED_FORMAT2(::testing::IsSubstring, "notifications", title);
  EXPECT_PRED_FORMAT2(::testing::IsNotSubstring, "microphone", title);
  EXPECT_PRED_FORMAT2(::testing::IsNotSubstring, "camera", title);
}

TEST_F(PermissionPromptBubbleViewTest,
       AccessibleTitleFileSchemeMentionsThisFile) {
  TestDelegate delegate(GURL("file:///tmp/index.html"),
                        {permissions::RequestType::kMicStream});
  auto bubble = CreateBubble(&delegate);

  EXPECT_PRED_FORMAT2(::testing::IsSubstring,
                      base::UTF16ToUTF8(l10n_util::GetStringUTF16(
                          IDS_PERMISSIONS_BUBBLE_PROMPT_THIS_FILE)),
                      base::UTF16ToUTF8(bubble->GetAccessibleWindowTitle()));
}

TEST_F(PermissionPromptBubbleViewTest,
       AccessibleTitleIncludesOnlyVisiblePermissions) {
  TestDelegate delegate(GURL(), {permissions::RequestType::kMicStream,
                                 permissions::RequestType::kCameraStream,
                                 permissions::RequestType::kCameraPanTiltZoom});
  auto bubble = CreateBubble(&delegate);

  const auto title = base::UTF16ToUTF8(bubble->GetAccessibleWindowTitle());
  EXPECT_PRED_FORMAT2(::testing::IsSubstring, "microphone", title);
  EXPECT_PRED_FORMAT2(::testing::IsSubstring, "move your camera", title);
  EXPECT_PRED_FORMAT2(::testing::IsNotSubstring, "use your camera", title);
}
